
import numpy as np


class FrequencyManager:
    """
    Handles frequency planning, range calculations, and filtering logic
    """
    
    def __init__(self, config, logger):
        self.config = config
        self.logger = logger
    
    
    
    def log_frequency_plan(self):
        """
        Log the frequency planning information for debugging
        """
        self.logger.debug(f"=== Frequency Plan ===")
        self.logger.debug(f"Total center frequencies: {len(self.config.CENTER_FREQS)}")
        self.logger.debug(f"PSD bandwidth: {self.config.SAMPLE_RATE/1e6:.3f} MHz")
        self.logger.debug(f"Sample rate: {self.config.SAMPLE_RATE/1e6:.3f} MHz")
        self.logger.debug(f"FFT size: {self.config.FFT_SIZE}")
        
        if len(self.config.CENTER_FREQS) > 0:
            self.logger.debug(f"Frequency range: {self.config.CENTER_FREQS[0]/1e6:.3f} - {self.config.CENTER_FREQS[-1]/1e6:.3f} MHz")
            freq_step = (self.config.CENTER_FREQS[1] - self.config.CENTER_FREQS[0]) / 1e6 if len(self.config.CENTER_FREQS) > 1 else 0
            self.logger.debug(f"Frequency step: {freq_step:.3f} MHz")
    
    def get_frequency_overlap(self):
        """
        Calculate the frequency overlap between adjacent center frequencies
        
        Returns:
            dict: Analysis of frequency overlap including:
                - overlap_hz: Overlap in Hz between adjacent PSDs
                - overlap_ratio: Ratio of overlap to PSD bandwidth
                - has_overlap: Boolean indicating if there's any overlap
                - freq_step_hz: Frequency step between centers
        """
        if len(self.config.CENTER_FREQS) < 2:
            return {
                'overlap_hz': 0,
                'overlap_ratio': 0,
                'has_overlap': False,
                'freq_step_hz': 0,
                'psd_bandwidth_hz': self.config.SAMPLE_RATE
            }
        
        # Calculate frequency step between adjacent centers
        freq_step_hz = self.config.CENTER_FREQS[1] - self.config.CENTER_FREQS[0]
        
        # Each PSD covers the full sample rate bandwidth
        psd_bandwidth_hz = self.config.SAMPLE_RATE
        
        # Overlap occurs when frequency step is less than PSD bandwidth
        overlap_hz = max(0, psd_bandwidth_hz - freq_step_hz)
        overlap_ratio = overlap_hz / psd_bandwidth_hz if psd_bandwidth_hz > 0 else 0
        has_overlap = overlap_hz > 0
        
        return {
            'overlap_hz': overlap_hz,
            'overlap_ratio': overlap_ratio,
            'has_overlap': has_overlap,
            'freq_step_hz': freq_step_hz,
            'psd_bandwidth_hz': psd_bandwidth_hz
        }
    
    def should_use_masking(self):
        """
        Determine if PSD masking is needed based on frequency plan
        
        Returns:
            bool: True if masking is needed, False if simple concatenation is sufficient
        """
        overlap_info = self.get_frequency_overlap()
        
        # If there's no overlap, no masking is needed
        if not overlap_info['has_overlap']:
            self.logger.debug("No frequency overlap detected - masking not needed")
            return False
        
        # If overlap is very small (< 5%), might as well concatenate
        if overlap_info['overlap_ratio'] < 0.05:
            self.logger.debug(f"Minimal overlap ({overlap_info['overlap_ratio']*100:.1f}%) - masking not needed")
            return False
        
        self.logger.debug(f"Significant overlap ({overlap_info['overlap_ratio']*100:.1f}%) - masking needed")
        return True
    
    
    def calculate_psd_stitcher_config(self):
        """
        Calculate optimal PSDStitcher configuration based on frequency plan
        
        Returns:
            dict: Configuration parameters for PSDStitcher including:
                - no_masking: Whether to use masking or simple concatenation
                - edge_bin_divisor: Edge masking divisor (if masking enabled)
                - mid_bin_divisor: Middle masking divisor (if masking enabled)
                - reasoning: Text explanation of the configuration choice
        """
        overlap_info = self.get_frequency_overlap()
        
        self.logger.debug("=== PSD Stitcher Configuration Analysis ===")
        self.logger.debug(f"Frequency step: {overlap_info['freq_step_hz']/1e6:.3f} MHz")
        self.logger.debug(f"PSD bandwidth: {overlap_info['psd_bandwidth_hz']/1e6:.3f} MHz")
        self.logger.debug(f"Overlap: {overlap_info['overlap_hz']/1e6:.3f} MHz ({overlap_info['overlap_ratio']*100:.1f}%)")
        
        # Always include overlap_ratio for the new overlap-based stitching
        base_config = {
            'overlap_ratio': overlap_info['overlap_ratio'],
            'psd_bandwidth_hz': overlap_info['psd_bandwidth_hz'],
            'freq_step_hz': overlap_info['freq_step_hz']
        }
        
        if not self.should_use_masking():
            config = {
                **base_config,
                'no_masking': True,
                'edge_bin_divisor': 8,  # FFT_SIZE / 8 edge removal
                'mid_bin_divisor': 8,   # FFT_SIZE / 8 mid removal
                'reasoning': f"No masking needed - overlap is {overlap_info['overlap_ratio']*100:.1f}%"
            }
        else:
            config = {
                **base_config,
                'no_masking': False,
                'edge_bin_divisor': 8,  # FFT_SIZE / 8 edge removal
                'mid_bin_divisor': 8,   # FFT_SIZE / 8 mid removal
                'reasoning': f"Masking enabled - {overlap_info['overlap_ratio']*100:.1f}% overlap detected"
            }
        
        self.logger.debug(f"Configuration: {config}")
        self.logger.debug("=== End PSD Stitcher Configuration ===")
        
        return config