
import datetime
import logging
import time
import signal
import random
import numpy as np
import matplotlib.pyplot as plt

from .signal_wrappers import PSD, Spectrogram, IQ

LOG = logging.getLogger(__name__)

violations_enabled = True
def sigh_enable_violate(*args):
    global violations_enabled
    violations_enabled = True
    LOG.info("Enabled violations")

def sigh_disable_violate(*args):
    global violations_enabled
    violations_enabled = False
    LOG.info("Disabled violations")

class Receiver:
    def __init__(self, args, logger, config):
        self.args = args
        self.logger = logger
        self.config = config

        if args.band.max % args.step_size != 0:
            raise ValueError(f"band max ({args.band.max}) must be a multiple of step-size")
        for tx in [*args.grant, *args.incumbent]:
            if tx.min % args.step_size != 0:
                raise ValueError(f"tx min ({tx.min}) must be a multiple of step-size")
            if tx.max % args.step_size != 0:
                raise ValueError(f"tx max ({tx.max}) must be a multiple of step-size")

        signal.signal(signal.SIGUSR1, sigh_enable_violate)
        signal.signal(signal.SIGUSR2, sigh_disable_violate)
        pass

    #
    # For ZMS dynamic mode ... just a nop in simulated mode
    #
    def updateConfig(self, range=None, gain=None, dwelltime=None):
        self.config.update(range=range, gain=gain, dwelltime=dwelltime)
        self.config.dump()
        return self

    def get_full_spectrum_psd(self):
        band = self.args.band
        grants = self.args.grant
        incumbents = self.args.incumbent
        step_size = self.args.step_size
        steps = range(int(band.min), int(band.max) + 1, int(step_size))
        center = band.min + (band.max - band.min) / 2
        violation_shift = self.args.violation_shift
        violation_gain = self.args.violation_gain
        violation = violations_enabled
        now = datetime.datetime.utcnow()

        if violation and (violation_shift or violation_gain):
            grants = [ x.violate(violation_shift, violation_gain) for x in grants ]

        all_spectrum = np.empty(len(steps), dtype=np.float32)
        all_frequencies = np.empty(len(steps), dtype=np.float32)
        all_center_freqs = np.full(len(steps), center, dtype=np.float32)
        all_times = np.full(len(steps), int(time.time()), dtype=int)
        all_devs = np.full(len(steps), "rf0", dtype=str)

        sigmf_meta = [
            dict(start_freq=steps[0], end_freq=steps[-1],
                 center_freq=center, step_size=step_size,
                 start_time=now, end_time=now, len=len(steps))
        ]

        ii = 0
        for fi in steps:
            is_violation = 0
            power = band.power + (random.random() * 2 - 1) * band.power * band.power_factor
            for tx in grants:
                if fi < tx.min or fi > tx.max:
                    continue
                txpower = tx.power + (random.random() * 2 - 1) * tx.power * tx.power_factor
                if fi < (tx.min + step_size):
                    txpower -= 0.75 * (txpower - band.power)
                elif fi < (tx.min + step_size * 2):
                    txpower -= 0.25 * (txpower - band.power)
                elif fi > (tx.max - step_size):
                    txpower -= 0.75 * (txpower - band.power)
                elif fi > (tx.max - step_size * 2):
                    txpower -= 0.25 * (txpower - band.power)
                power = max(txpower, power)
                if violation:
                    is_violation = 1
            for tx in incumbents:
                if fi < tx.min or fi > tx.max:
                    continue
                txpower = tx.power + (random.random() * 2 - 1) * tx.power * tx.power_factor
                if fi < (tx.min + step_size):
                    txpower -= 0.75 * (txpower - band.power)
                elif fi < (tx.min + step_size * 2):
                    txpower -= 0.25 * (txpower - band.power)
                elif fi > (tx.max - step_size):
                    txpower -= 0.75 * (txpower - band.power)
                elif fi > (tx.max - step_size * 2):
                    txpower -= 0.25 * (txpower - band.power)
                power = max(txpower, power)
            above = power - band.power
            all_spectrum[ii] = power
            all_frequencies[ii] = fi
            ii += 1
            pass

        return PSD(self.config, all_spectrum, all_frequencies,
                   all_times, all_devs, all_center_freqs, None, sigmf_meta)
    pass

