import numpy as np
import scipy.signal
import json
import time
import datetime
import os
import matplotlib.pyplot as plt

from .config import Configuration
from .receiver import Receiver
from .output_router import OutputRouter
from .spectrum_plotter import Plot

#
# For legacy mode when running in Powder
#
EMULAB_BOOTDIR = "/var/emulab/boot"
EMULAB_SAVEDIR = "/var/emulab/save"

class Monitor:
    def __init__(self, args, logger):
        self.args = args
        self.logger = logger
        self.legacy = False
        self.devname = None

        if self.args.sim_mode:
            from .sim_receiver import Receiver
        else:
            from .receiver import Receiver
            pass

        if self.args.antenna:
            if self.args.device:
                devname = self.args.device
            else:
                devname = self.args.antenna
                pass

            self.device_config = {
                "devices" : {
                    devname : {
                        "name"     : devname,
                        "channels" :  {"0" : self.args.antenna}
                    }
                }
            }
        elif self.args.configfile and os.path.exists(self.args.configfile):
            with open(self.args.configfile, 'r') as f:
                self.device_config = json.load(f)
                pass
        else:
            raise Exception(
                "Must supply a device config file or --antenna/--device")

        #
        # Legacy mode write files to the WB store directory
        #
        if False and os.path.exists(EMULAB_BOOTDIR):
            with open(EMULAB_BOOTDIR + "/mydomain", 'r') as f:
                domain = f.readline().rstrip("\n")
                self.domain = domain
                pass

            with open(EMULAB_BOOTDIR + "/nickname", 'r') as f:
                nickname = f.readline().rstrip("\n")
                tokens = nickname.split(".")
                self.eid = tokens[1]
                self.pid = tokens[2]
                pass
            
            self.legacy = True
            self.devname = list(self.device_config["devices"].keys())[0]
            pass

        if self.args.www_mode:
            if not (self.args.device or self.devname):
                raise Exception("Must provide --device (ex: 'nuc1:rf0') with --www-mode") 
            if not os.path.exists(self.args.www_dir):
                raise Exception("WWW directory (--www-dir) does not exist")
            if not os.access(self.args.www_dir, os.W_OK):
                raise Exception("WWW directory (--www-dir) is not writable")
            pass

        self.logger.info("Device Config: " + str(self.device_config))
        self.config = Configuration(args, self.device_config)
        self.logger.info("Config: " + str(vars(self.config)))
        
        # Initialize output router
        domain = getattr(self, 'domain', None)
        pid = getattr(self, 'pid', None)
        eid = getattr(self, 'eid', None)
        self.output_router = OutputRouter(args, logger, self.legacy, self.devname, domain, pid, eid)

        #
        # In ZMS mode, hand off Receiver initialization to the ZMS module
        #
        if args.zms_mode:
            from .zms import ZMS
            self.zms = ZMS(self.args, self.logger, self.config)
            return

        self.receiver = Receiver(self.args, logger, self.config)

    def start(self):
        if self.args.zms_mode:
            return self.zms.start()

        loops = self.args.repeat
        
        #
        # If plotting, create a plot that will be updated when looping
        #
        if self.args.plot:
            self.plotter = Plot(self.args.repeat != 1)
            pass

        while True:
            dt_now_str = datetime.datetime.now().strftime("%m/%d/%Y %H:%M:%S")
            self.logger.debug(f"Starting PSD generation at {dt_now_str}")
            start_time = time.time()
            psd_kwargs = dict()
            if len(self.config.CENTER_FREQS) == 1:
                psd_kwargs = dict(include_samples=True)
            psd = self.get_full_spectrum_psd(**psd_kwargs)
            end_time = time.time()
            self.logger.debug(f"PSD generation completed in {end_time - start_time:.2f} seconds")
            
            # Route output using the output router
            should_continue = self.output_router.route_output(psd, self.plotter if self.args.plot else None)
            if not should_continue:
                # Keep the last plot open if plotting
                if self.args.plot and hasattr(self, 'plotter'):
                    self.plotter.wait_for_close()
                return
            
            if self.args.repeat:
                loops -= 1
                if loops == 0:
                    # Keep the last plot open if plotting
                    if self.args.plot and hasattr(self, 'plotter'):
                        self.plotter.wait_for_close()
                    return
                pass
            
            if self.args.interval:
                time.sleep(self.args.interval)
                pass
            pass

        #
        # If interactively plotting, hold here
        #
        if self.args.plot and self.args.repeat != 1:
            plt.show(block=True)
            pass
        pass

    # Get samples for one center frequency.
    def get_samples(self, frequency):
        return self.receiver.get_samples(frequency)

    # Get PSD for all center frequencies.
    def get_full_spectrum_psd(self, include_samples=False):
        return self.receiver.get_full_spectrum_psd(include_samples=include_samples)

    # Get I/Q samples for all center frequencies.
    def get_full_spectrum(self):
        return self.receiver.get_full_spectrum()

    # Output methods moved to OutputRouter class
        
    pass

# Plot class moved to spectrum_plotter.py
