import os
import sys
import argparse
import logging

# Only when a daemon
LOGFILE     = "/local/logs/rdz-monitor.log"
# Only when storing PSDs for a web server
WEBDIR      = "/local/www"
# Default configuration
CONFIGFILE  = "/etc/rfmonitor/device_config.json"

from .monitor import Monitor
from .config import *

# For simulated monitor mode.
class FreqRangeDesc:
    def __init__(self, min, max, power, power_factor=None):
        self.min = min
        self.max = max
        self.power = power
        self.power_factor = power_factor

    @classmethod
    def from_string(cls, s):
        sa = s.split(",")
        if len(sa) != 4:
            raise ValueError("invalid frequency range descriptor")
        return FreqRangeDesc(*[float(x) for x in sa])

    def __repr__(self):
        return f"FreqRangeDesc(min={self.min},max={self.max},power={self.power},power_factor={self.power_factor})"

    def violate(self, shift, gain):
        return FreqRangeDesc(self.min + shift, self.max + shift, self.power + gain, self.power_factor)

class DefaultDestEnvAction(argparse.Action):
    def __init__(self, option_strings, dest, required=False, default=None, **kwargs):
        """An argparse.Action that initializes the default value of the arg from an environment variable named `dest.upper()` (where dest is the storage location of the value post-parse, e.g. `args.dest`); and, if the arg was required, *unsets* it from being required, so that argparse does not fail the parse if the argument is not supplied.  This is certainly a bit unfortunate since it changes the helptext behavior, but nothing to do about that."""
        dest_upper = dest.upper()
        if dest_upper in os.environ:
            default = os.environ[dest_upper]
        if required and default:
            required = False
        super(DefaultDestEnvAction, self).__init__(
            option_strings, dest, default=default, required=required, **kwargs)

    def __call__(self, parser, namespace, values, option_string=None):
        setattr(namespace, self.dest, values)

def parse_args():
    parser = argparse.ArgumentParser(
        prog="rfmonitor",
        description="Run the RF monitor and report results",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(
        "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "-c", "--configfile", default=CONFIGFILE, type=str,
        help="Optional config file. Note that --antenna and --device override")
    parser.add_argument(
        "-d", "--debug", default=False, action='store_true',
        help="Turn on on copious debugging output")
    parser.add_argument(
        "-D", "--debug-all", default=False, action='store_true',
        help="Turn on on copious debugging output in all modules")
    parser.add_argument(
        "-g", "--gain", default=30, type=int,
        help="Set the receiver gain.")
    parser.add_argument(
        "-f", "--output", default=None, type=str,
        help="Output file for samples or psd, defaults to STDOUT.")
    parser.add_argument(
        "-R", "--range", default="100e6-6e9", type=str,
        help="Spectrum range to scan.")
    parser.add_argument(
        "-C", "--center", default=None, type=float,
        help="Instead of scanning a range, scan a single center frequency.")
    parser.add_argument("-s", "--sample-rate", default=10e6, type=float,
        help="Sample rate (Hz).")
    parser.add_argument(
        "--fft-size", default=DEFAULT_FFT_SIZE, type=int,
        help="FFT size.")
    parser.add_argument(
        "--psd-fft-overlap", default=0.5, type=float, help="Overlap for the PSD FFT. 0.5 means 50%% overlap. Set to 0 for no overlap (Bartlett's method).")
    parser.add_argument(
        "--dwell-time", default=DEFAULT_DWELL_TIME_S, type=float,
        help="How much time to spend at each center frequency (seconds).")
    parser.add_argument(
        "--psd-window", default="hann", type=str,
        choices=["hann", "bartlett", "boxcar", "triang", "blackman", "hamming", "flattop",
            "parzen", "bohman", "blackmanharris", "nuttall", "barthann", "cosine", "exponential",
            "tukey", "taylor", "lanczos"],
        help="PSD window to use.")
    parser.add_argument(
        "--no-restart-streamer", default=False, action='store_true',
        help="Do not restart the streamer when retuning the center frequency. " +
        "Defaults to False")
    parser.add_argument(
        "--plot", default=False, action='store_true',
        help="Plot combined results in an X11 window instead of writing output")
    parser.add_argument(
        "--plot-individual", default=False, action='store_true',
        help="Plot individual PSD captures before the combined plot")
    parser.add_argument(
        "--csv", default=False, action='store_true',
        help="Write a CSV file of the results and exit")
    parser.add_argument(
        "--spectrogram", default=False, action='store_true',
        help="Generate a spectrogram instead of PSD")
    parser.add_argument(
        "--antenna", default=None, type=str,
        help="Antenna to use (RX2 or TX/RX). Defaults to channel 0 for now.")
    parser.add_argument(
        "--device", default=None, type=str,
        help="Device name for output. For example: nuc1:rf0. Defaults to antenna.")
    parser.add_argument(
        "--interval", default=10, type=int,
        help="Interval between runs when running more than once (--repeat).")
    parser.add_argument(
        "--repeat", type=int, default=0,
        help="Repeat count, zero means run forever.")
    parser.add_argument(
        "--uhd-args", type=str, default="",
        help="UHD device arguments.")

    #
    # These arguments are optional, and enable ZMS mode.
    #
    parser.add_argument(
        "--zms-mode", default=False, action='store_true',
        help="Enable ZMS mode, be sure to provide all the other args")
    parser.add_argument(
        "--is-powder", default=False, action='store_true',
        help="Enable POWDER-RDZ mode (and check for RDZ-in-RDZ nested case)")
    parser.add_argument(
        "--monitor-id", action=DefaultDestEnvAction, default=None, type=str, 
        help="ZMC monitor ID")
    parser.add_argument(
        "--monitor-description", action=DefaultDestEnvAction, default=None, type=str,
        help="ZMC monitor description string")
    parser.add_argument(
        "--element-token", action=DefaultDestEnvAction, default=None, type=str,
        help="Element token")
    parser.add_argument(
        "--zmc-http", action=DefaultDestEnvAction, default=None, type=str,
        help="ZMC URL")
    parser.add_argument(
        "--dst-http", action=DefaultDestEnvAction, default=None, type=str,
        help="DST URL")
    parser.add_argument(
        "--zms-format", default="sigmf", type=str,
        choices=["sigmf", "csv"],
        help="OpenZMS format (sigmf or legacy powder-specific csv)")
    parser.add_argument(
        "--zms-task", default="psd", type=str,
        choices=["psd", "spectrogram", "samples"],
        help="Static collection task for OpenZMS mode (psd, spectrogram, samples)")
    parser.add_argument(
        "--no-dynamic", default=False, action="store_true")
    parser.add_argument(
        "--zms-auto", default=False, action="store_true")

    #
    # These arguments are optional, and enable simulated monitor mode. 
    #
    parser.register("type", "FreqRangeDesc", FreqRangeDesc.from_string)
    parser.add_argument(
        "--sim-mode", default=False, action='store_true',
        help="Enable simulated monitor mode, be sure to provide all the other args")
    parser.add_argument(
        "--band", type='FreqRangeDesc', default="900e6,928e6,-120.0,0.005",
        help="A frequency range descriptor that defines the noise characteristics for the PSD: 'minfreq,maxfreq,noise-power,noise-power-rand-pct' .")
    parser.add_argument(
        "--grant", type='FreqRangeDesc', default=[], action="append",
        help="A frequency range descriptor whose range is in the band that defines power characteristics for a particular OpenZMS grant transmitter in the band: 'minfreq,maxfreq,power,power-rand-pct' .")
    parser.add_argument(
        "--incumbent", type='FreqRangeDesc', default=[], action="append",
        help="A frequency range descriptor whose range is in the band that defines power characteristics for a particular incumbent transmitter in the band: 'minfreq,maxfreq,power,power-rand-pct' .")
    parser.add_argument(
        "--step-size", type=float, default=50e3,
        help="Step size for power bins in PSD.")
    parser.add_argument(
        "--violation-shift", type=float, default=-2e6,
        help="Frequency shift (Hz) when violations are toggled on.")
    parser.add_argument(
        "--violation-gain", type=float, default=0.0,
        help="Power gain (dB) when violations are toggled on.")

    #
    # These arguments are optional, and enable WWW Server mode. 
    #
    parser.add_argument(
        "--www-mode", default=False, action='store_true',
        help="Enable WWW/WEB mode, be sure to provide all the other args")
    parser.add_argument(
        "--www-dir", default=WEBDIR, type=str,
        help="Local directory to write PSD file to")
    
    args = parser.parse_args(sys.argv[1:])

    if args.is_powder and not args.zmc_http:
        parser.print_help()
        print("\nError: --zmc-http is required when --is-powder is set")
        exit(1)

    return args

def main():
    args = parse_args()

    #
    # One logger for all
    #
    log_format_string = "[%(levelname)s] [%(module)s] %(message)s"
    if args.debug or args.debug_all:
        level = logging.DEBUG
    else:
        level = logging.INFO
        pass

    if args.debug_all:
        logging.basicConfig(level=level, format=log_format_string)
    else:
        logging.basicConfig(format=log_format_string)
        pass
    logger = logging.getLogger("monitor")
    logger.setLevel(level)
    
    monitor = Monitor(args, logger).start()
    pass

if __name__ == "__main__":
    main()
