import os
import sys
import gzip
import tempfile
import shutil
import datetime


class OutputRouter:
    """
    Routes PSD output to the appropriate handler based on configuration
    """
    
    def __init__(self, args, logger, legacy=False, devname=None, domain=None, pid=None, eid=None):
        self.args = args
        self.logger = logger
        self.legacy = legacy
        self.devname = devname
        self.domain = domain
        self.pid = pid
        self.eid = eid
    
    def route_output(self, psd, plotter=None):
        """
        Route PSD output based on arguments configuration
        Returns True if processing should continue, False if should exit
        """
        if self.args.plot_individual:
            # Plot individual PSD captures
            psd.plot_individual()
            
        # CSV output
        if self.args.csv:
            self.logger.debug("CSV mode detected, writing CSV output...")
            self._write_csv(psd)
            self.logger.debug("CSV output complete")
        
        # Plot output
        if self.args.plot:
            if plotter:
                plotter.plotPSD(psd)
            else:
                psd.plot()
        
        # WWW mode output
        if self.args.www_mode:
            self._www_save(psd)
        
        # Legacy mode output
        if self.legacy:
            self._wb_save(psd)
        
        # Default dump output (only if no other output was specified)
        if not (self.args.csv or self.args.plot or self.args.www_mode or self.legacy):
            psd.dump()
        
        # Exit after first iteration if CSV was requested and repeat is 1
        if self.args.csv and self.args.repeat == 1:
            return False
        
        return True  # Continue processing
    
    def _write_csv(self, psd):
        """Write CSV output"""
        self.logger.debug(f"_write_csv called, output file: {self.args.output}")
        if self.args.output:
            self.logger.debug(f"Writing CSV to file: {self.args.output}")
            with open(self.args.output, "w") as f:
                psd.writecsv(f)
        else:
            self.logger.debug("Writing CSV to stdout")
            psd.writecsv(sys.stdout)
    
    def _www_save(self, psd):
        """Save for web server mode"""
        #
        # Files go in a subdir named by the day, This avoids ending up
        # with a giant directory that takes forever to load and has a
        # menu with 100s of entries.
        #
        now    = datetime.datetime.now()
        utcnow = int(now.timestamp())
        subdir = now.strftime("20%y-%m-%d")
        wwwdir = self.args.www_dir + "/" + subdir
        if not os.path.exists(wwwdir):
            os.mkdir(wwwdir)
            os.chmod(wwwdir, 0o0775)
        csvname = "%s-%s.csv.gz" % (self.args.device, utcnow)
        csvfile = wwwdir + "/" + csvname

        # Create temp file, gzip, move into place.
        with tempfile.NamedTemporaryFile(mode="w", delete=False) as fp:
            psd.writecsv(fp)
            fp.close()
            with open(fp.name, mode='rb') as f_in:
                with gzip.open(csvfile, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)
            os.remove(fp.name)
    
    def _wb_save(self, psd):
        """Legacy mode for Powder testbed to write into the wbstore directory."""
        EMULAB_SAVEDIR = "/var/emulab/save"
        
        # Files go into the local wbstore directory, unless running on the
        # Mothership, in which case it goes to the project directory
        if self.domain == "emulab.net":
            savedir = "/proj/%s/exp/%s" % (self.pid, self.eid)
        else:
            savedir = EMULAB_SAVEDIR

        now     = datetime.datetime.now()
        utcnow  = int(now.timestamp())
        csvname = "%s-%s.csv.gz" % (self.devname, utcnow)
        csvfile = savedir + "/" + csvname

        # Create temp file, gzip, move into place.
        with tempfile.NamedTemporaryFile(mode="w", delete=False) as fp:
            psd.writecsv(fp)
            fp.close()
            with open(fp.name, mode='rb') as f_in:
                with gzip.open(csvfile, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)
            os.remove(fp.name)

        # An extra step for the Mothership.
        if self.domain != "emulab.net":
            return
        
        mdir = "/proj/%s/monitor" % (self.pid,) 
        if not os.path.exists(mdir):
            os.mkdir(mdir)
            os.chmod(mdir, 0o0775)
            pass
        tfile = mdir + "/" + self.eid + ".gz"
        if os.path.exists(tfile):
            os.remove(tfile)
            pass
        cmd = "/bin/tar -cf %s.tmp -C /proj %s/exp/%s/%s" % (tfile, self.pid, self.eid, csvname)
        self.logger.info(cmd)
        if os.system(cmd):
            raise Exception("cmd failed")
        cmd = "/bin/mv %s.tmp %s" % (tfile, tfile)
        if os.system(cmd):
            raise Exception("cmd failed")
        pass
