#
# Configuration computed from the command line arguments
#
import math
import numpy as np

# Various defaults for parameters
DEFAULT_SCAN_RANGE = "100e6-6e9"
DEFAULT_FFT_SIZE = 2**10
DEFAULT_NUM_FFTS = 2**10
MIN_FFT_SIZE = 32
DEFAULT_INST_BW_HZ = 10e6
DEFAULT_DWELL_TIME_S = 0.01  # Time to dwell at each center frequency in seconds


class Configuration:
    def __init__(self, args, device_config):
        self.args = args
        self.device_config = device_config
        self.init()

    def init(self):
        # sample rate must be twice the bandwidth of interest when using an LO offset.
        # and we want an extra margin for anti-aliasing filter effects. 
        self.SAMPLE_RATE = self.args.sample_rate
        self.FFT_SIZE = self.args.fft_size 
        self.DWELL_TIME = self.args.dwell_time
        
        if self.args.center:
            # Center the spectrum around the specified frequency
            self.CENTER_FREQS = np.array([float(self.args.center)])
            # Start at center - half sample rate, end at center + half sample rate
            self.START_FREQ = self.args.center - self.SAMPLE_RATE / 2
            self.END_FREQ = self.args.center + self.SAMPLE_RATE / 2
        else:
            if not self.args.range:
                self.args.range = DEFAULT_SCAN_RANGE
                pass

            start_freq, stop_freq = self.args.range.split("-")
            self.START_FREQ = float(start_freq)
            self.END_FREQ = float(stop_freq)
            
            # Generate center frequencies with spacing based on FFT_SIZE / MIN_FFT_SIZE * 7
            # Step size: SAMPLE_RATE * 7 / MIN_FFT_SIZE
            step_size = self.SAMPLE_RATE * 7 / MIN_FFT_SIZE
            
            # First center should be at START_FREQ + SAMPLE_RATE/2 to ensure we start at the requested frequency
            # Last center should be at END_FREQ - SAMPLE_RATE/2 to ensure we end at the requested frequency
            first_center = self.START_FREQ + self.SAMPLE_RATE / 2
            last_center = self.END_FREQ - self.SAMPLE_RATE / 2
            
            # Generate center frequencies with one extra capture before and after
            # This helps with edge bin handling in the PSD stitcher
            extended_first = first_center - step_size
            extended_last = last_center + step_size
            
            self.CENTER_FREQS = np.arange(
                extended_first,
                extended_last,
                step_size
            )
        
        self.NUM_STEPS = len(self.CENTER_FREQS)
        self.NUM_FFTS = self.args.dwell_time * self.SAMPLE_RATE // self.FFT_SIZE
        self.REPEAT = self.args.repeat
        self.INTERVAL = self.args.interval
        self.GAIN = self.args.gain

    #
    # This is for supporting ZMS configurable monitoring.
    #
    def update(self, range=None, gain=None, dwell_time=None):
        if range:
            (a,b) = range.split("-")
            if a == b:
                self.args.center = float(a)
                self.args.range = None
            else:
                self.args.center = None
                self.args.range = range
                pass
            pass
        if gain != None:
            self.args.gain = gain
            pass
        # If not specified, we want to return to default.
        self.args.dwell_time = DEFAULT_DWELL_TIME_S
        self.init()
        pass

    def dump(self):
        print(str(vars(self)))
        pass
    
    pass
