#!/bin/python

import time
import uuid
import os
import sys
import signal
import traceback
import argparse
import logging
import datetime
import dateutil.parser
import asyncio
import threading
import subprocess
import functools
import shlex
import base64
import copy
import httpx

#
# Note from David
#
# * If you do monitor state update --status active , being an admin, the
#   returned MonitorState.Status will be immediately set to active and zmc
#   will generate a MonitorPending created event telling your monitor to
#   "go active"; your monitor must push monitor state update --op-status
#   active
#
# * If you do monitor state update --op-status active as the monitor token
#   from the monitor, you will be "self-activating" and the returned
#   MonitorState.Status will be active .  In this case, you will not get a
#   MonitorPending create event, because we look for this case where you have
#   just set opstatus to active, so nothing needs to be done.
#
# * If you do monitor pending create --status active the
#   MonitorState.Status will be left untouched until the monitor responds to
#   the MonitorPending event with a MonitorStateUpdate API call, setting
#   opstatus as commanded, and referencing the last_pending_id.
#
# About taskable monitors:
#
# * op_status is active whenever the taskable monitor has tasks to run or
#   can accept a new task.  A taskable monitor only pauses when told to by
#   the zmc.  I think it's the same thing as the configurable monitor.
#   When zmc pauses the monitor, it stops executing tasks.  It should still
#   schedule new tasks if so notified, but not run them.
#
# * During a pause, tasks that would have run are aged out of the task list
#   (never run).
#

from zmsclient.zmc.client import ZmsZmcClient
from zmsclient.dst.client import ZmsDstClient
from zmsclient.zmc.v1.models import Subscription, EventFilter, Error, AnyObject
from zmsclient.dst.v1.models import Observation, ObservationKind
from zmsclient.common.subscription import ZmsSubscriptionCallback
from zmsclient.zmc.v1.models.monitor_pending import MonitorPending
from zmsclient.zmc.v1.models.monitor_state import MonitorState
from zmsclient.zmc.v1.models.monitor_op_status import MonitorOpStatus
from zmsclient.zmc.v1.models.monitor_task_state import MonitorTaskState
from zmsclient.zmc.v1.models.update_monitor_state_op_status import UpdateMonitorStateOpStatus
from .config import Configuration
from .receiver import Receiver

# Global logger
LOG = None

# Not sure where to pick these up
EVENT_TYPE_CREATED         = 2
EVENT_TYPE_UPDATED         = 3
EVENT_TYPE_DELETED         = 4
EVENT_TYPE_STARTED         = 18
EVENT_TYPE_STOPPED         = 19
EVENT_SOURCETYPE_ZMC       = 2
EVENT_CODE_MONITOR         = 2005
EVENT_CODE_MONITOR_STATE   = 2009
EVENT_CODE_MONITOR_PENDING = 2010
EVENT_CODE_MONITOR_ACTION  = 2011
EVENT_CODE_MONITOR_TASK    = 2012

def ask_exit(signame, loop):
    LOG.info("got signal %s: exit" % signame)
    loop.stop()    

#
# Wrapper class for everything in this module
#
class ZMS:
    def __init__(self, args, logger, config):
        self.progargs = args
        self.logger   = logger
        self.config   = config

        global LOG
        LOG = self.logger

        if not (args.monitor_id and args.monitor_description and
                args.element_token and args.dst_http):
            raise Exception("Not all ZMS arguments were provided, try --help")

        self.dstclient = ZmsDstClient(
            args.dst_http, args.element_token,
            detailed=False, raise_on_unexpected_status=True,
            httpx_args={"transport" : httpx.HTTPTransport(retries=3)})

        self.zmcclient = ZmsZmcClient(
            args.zmc_http, args.element_token,
            detailed=False, raise_on_unexpected_status=True,
            httpx_args={"transport" : httpx.HTTPTransport(retries=3)})

        #
        # The initial config will most likely be overriden in dynamic mode.
        #
        self.monitor = Monitor(
            self.progargs, self.config, self.dstclient, self.zmcclient)

        # Just in case a monitor does not have its type set or is "static"
        # This is the same as --no-dynamic, so set that.
        if self.monitor.zmstype == "static":
            args.no_dynamic = True
            pass

        self.ZMCsubscription = None
        self.scheduler = None
        self.heartbeat = None
        if not args.no_dynamic:
            self.scheduler = Scheduler(self.monitor)
            self.heartbeat = HeartBeat(self.monitor)
            self.ZMCsubscription = ZMCSubscriptionCallback(
                self.zmcclient, self.scheduler, self.monitor,
                subscription=Subscription(
                    id=str(uuid.uuid4()), filters=[]),
                reconnect_on_error=True)
        pass

    def start(self):
        if self.progargs.no_dynamic:
            self.monitor.run()
            return

        asyncio.run(self.async_start())
        pass

    async def async_start(self):
        loop = asyncio.get_running_loop()
        
        for signame in {'SIGINT', 'SIGTERM', 'SIGHUP'}:
            loop.add_signal_handler(
                getattr(signal, signame),
                functools.partial(ask_exit, signame, loop))
            pass

        subs = [self.ZMCsubscription]

        try:
            self.scheduler.start()
            runnable = [sub.start() for sub in subs]
            self.heartbeat.start()
            await asyncio.gather(*runnable)
        except asyncio.CancelledError:
            self.scheduler.stop()
            self.heartbeat.stop()
            self.scheduler.join(timeout=5)
            self.heartbeat.join(timeout=5)
            for sub in subs:
                await sub.stop()
                pass
            raise
        pass
        
    pass

#
# We use a scheduler for both Configurable and Taskable monitors
# A Pending event means schedule "now"
#
class Scheduler(threading.Thread):
    def __init__(self, monitor):
        super(Scheduler, self).__init__()
        self.monitor = monitor
        self.lock = threading.Lock()
        self.condition = threading.Condition(lock=self.lock)
        # Yep, simple simple simple.
        self.schedule  = []
        self.current = None
        self.done = False
        # The ZMC can force the monitor to suspend operation
        self.suspended = False

        if monitor.zmstype == "configurable":
            #
            # A configurable monitor might start out running.
            #
            if monitor.status == "active":
                self.schedule = [ self.Task(time.time(), None, "pending", None) ]
                pass
            pass
        elif monitor.zmstype == "taskable":
            if monitor.status == "paused":
                self.suspended = True
                pass
            
            #
            # Get the initial set of tasks (if any) and schedule them.
            #
            for t in monitor.initialTaskList():
                self.scheduleTask(t)
                pass
            pass
        monitor.notifyExit = self.notifyExited
        pass
    
    def stop(self):
        LOG.info("Stopping the scheduler")
        self.done = True
        self.lock.acquire()
        self.condition.notify()
        self.lock.release()
        pass

    # Callback for the monitor to tell the schedule something went wrong
    def notifyExited(self):
        LOG.info("Scheduler:notifyExited")
        # If already locked, do not worry. This is just an optimization
        # to find out sooner that the monitor has died for some reason.
        if not self.done and not self.lock.locked():
            self.lock.acquire()
            self.condition.notify()
            self.lock.release()
            pass
        pass

    class Task:
        def __init__(self, when, end, type_, item):
            self.when  = when
            self.stop  = end
            self.type  = type_
            self.item  = item
            pass

        def __str__(self):
            return "Task: %r %r %r" % (self.when, self.stop, self.item.id)
        pass

    def run(self):
        with self.condition:
            while not self.done:
                timeo = None

                LOG.info("Sched run %r %s", self.suspended, self.current);

                if self.current:
                    #
                    # See if it is supposed to be done.
                    # The docs say might get woken up before the timeout expires.
                    #
                    if self.current.stop:
                        if time.time() >= self.current.stop:
                            if self.current.item.interval != None:
                                item = self.current.item
                                self.insertTaskIntoSchedule(
                                    time.time() + item.interval,
                                    time.time() + item.interval + item.duration,
                                    item)
                                self.stopMonitor()
                            else:
                                self.stopMonitor(finished=True)
                                pass
                            pass
                        else:
                            timeo = self.current.stop - time.time()
                            pass
                        pass
                    elif not self.monitor.isRunning():
                        if self.monitor.lastError():
                            #
                            # For an error, report and we are done.
                            #
                            self.stopMonitor()
                            pass
                        elif self.current.item.interval != None:
                            item = self.current.item
                            self.insertTaskIntoSchedule(
                                time.time() + item.interval,
                                time.time() + item.interval + item.duration,
                                item)
                            self.stopMonitor()
                        else:
                            self.stopMonitor(finished=True)
                            pass
                        pass
                    pass

                if not self.current and len(self.schedule):
                    #
                    # First item is all we care about at the moment
                    #
                    item = self.schedule[0]
                    if time.time() >= item.when:
                        item = self.schedule.pop(0)
                        if self.suspended:
                            # Just pretending. 
                            self.current = item
                        else:
                            self.startMonitor(item)
                            pass

                        # In this simple scheduler, we will get behind.
                        slip = time.time() - item.when
                        item.when = time.time()
                        
                        if item.stop:
                            item.stop += slip
                            timeo = item.stop - time.time()
                            pass
                        elif self.suspended:
                            # We have to wake up at some time and call the
                            # current task done. Cheesy,
                            timeo = 2
                            pass
                        LOG.info("Starting %s", item)
                    else:
                        timeo = item.when - time.time()
                        pass
                    pass

                #
                # Update the monitor task status for the next hearbeat
                #
                if False:
                    if (self.current or len(self.schedule)) and not self.suspended:
                        next_status = "active"
                    else:
                        next_status = "paused"
                        pass
                    if self.monitor.status != next_status:
                        # No need to lock for this change
                        self.monitor.status = next_status
                        self.monitor.updateOpStatus()
                        pass
                    pass

                try:
                    LOG.info("Scheduler waiting for %r seconds", timeo)
                    self.condition.wait(timeout=timeo)
                except:
                    pass
                pass
            pass
        self.stopMonitor()
        LOG.info("Scheduler thread is exiting")
        pass

    def schedulePending(self, pending):
        if self.done:
            return

        #
        # There will be only one Pending at a time. And if there is a
        # Task running (or scheduled) those get wiped out and just the
        # new pending is run.
        #
        if len(self.schedule):
            LOG.info("schedulePending: Clearing the task list")
            self.schedule = []
            pass
        self.stopMonitor()
            
        self.schedule.append(self.Task(time.time(), None, "pending", pending))
        self.condition.notify()
        pass

    def scheduleTask(self, task):
        if self.done:
            return
        
        action = self.monitor.getTaskAction(task.monitor_action_id)
        if not action:
            return

        with self.condition:
            if task.start == None:
                when = time.time()
            else:
                when = dateutil.parser.isoparse(task.start).timestamp()
                pass

            if task.duration == None:
                #
                # See if we can determine a plausable duration as an upper bound.
                #
                if task.parameters and "dwelltime" in task.parameters:
                    task.duration = task.parameters["dwelltime"] + 2
                    end = when + task.duration                    
                else:
                    end = None
                    pass
            else:
                end = when + task.duration
                pass

            # Need more error handling for schedule conflicts.
            if self.current:
                if self.current.stop == None:
                    LOG.error("Cannnot schedule task; a forever task is running %r", task)
                    self.monitor.updateTaskState(
                        task, error="Cannot schedule task; a forever task is running")
                    return

                if end == None:
                    LOG.error("Cannnot schedule task; another task is also scheduled %r", task)
                    self.monitor.updateTaskState(
                        task, error="Cannot schedule task; another task is scheduled")
                    return
                
                # Schedule this task to run when the current task ends
                dur  = end - when
                when = self.current.stop
                end  = when + dur
                pass

            self.insertTaskIntoSchedule(when, end, task)

            # No need to notify if something is running.
            if not self.current:
                self.condition.notify()
                pass
            pass
        pass

    # Call this with the lock taken.
    def insertTaskIntoSchedule(self, when, end, task):
        LOG.info("Scheduler:insertTaskIntoSchedule %r %r %r", task.id, when, end)
        LOG.debug("Scheduler:insertTaskIntoSchedule %r", task)
        
        newTask = self.Task(when, end, "task", task)
        
        if len(self.schedule) == 0 or when <= self.schedule[0].when:
            self.schedule.insert(0, newTask)
        elif when > self.schedule[-1].when:
            self.schedule.append(newTask)
        else:
            for i, T in enumerate(self.schedule):
                if when < T.when:
                    self.schedule.insert(i, newTask)
                    break
                pass
            pass
        return newTask

    #
    # This will stop a current task or delete a future task.
    # 
    def deleteTask(self, task):
        with self.condition:
            if self.current and task.id == self.current.item.id:
                self.stopMonitor()
                return

            for i, T in enumerate(self.schedule):
                if task.id == T.item.id:
                    self.schedule.pop(i)
                    return
                pass
            pass
        pass

    def handlePending(self, pending):
        LOG.info("Scheduler:handlePending")

        with self.condition:
            if self.monitor.zmstype == "configurable":
                # configurable monitors start and stop with Pendings
                # The Pending will be acked.
                if pending.status != "active":
                    self.stopMonitor(pending)
                else:
                    self.schedulePending(pending)
                    pass
                return

            # Taskable monitor pause/resume the schedule with Pendings
            self.monitor.lock.acquire()
            # Ack the pending, which changes the monitor status as well.
            self.monitor.updateOpStatus(pending=pending);
            
            if pending.status != "active":
                self.suspended = True
                if self.current:
                    self.stopTask(self.current.item)
                    pass
            else:
                self.suspended = False
                if self.current:
                    # The schedule has still been running, so need to sync
                    # up the parameters.
                    self.startTask(self.current.item)
                    pass
                pass
            self.monitor.lock.release()
            pass
        pass

    def startMonitor(self, task):
        LOG.info("Scheduler:startMonitor")
        self.current = task
        if task.type == "pending":
            self.startPending(task.item)
        else:
            self.startTask(task.item)
            pass
        pass

    def stopMonitor(self, pending=None, finished=False):
        if not self.current:
            return
        LOG.info("Scheduler:stopMonitor finished:%r", finished)
        if self.current.type == "pending":
            self.stopPending(pending)
        else:
            self.stopTask(self.current.item, finished=finished)
            pass
        self.current = None
        pass

    def startTask(self, task):
        LOG.info("Scheduler:startTask")
        #
        # Lock out the heartbeat while making the changes.
        #
        self.monitor.lock.acquire()
        if task.parameters:
            self.monitor.updateParamsFromAnyObject(task.parameters)
            pass
        self.monitor.updateSettingsFromTask(task)

        # Start the monitor with new params
        self.monitor.startMonitor()
        self.monitor.updateTaskState(task, runnable=True)
        self.monitor.lock.release()
        pass

    def stopTask(self, task, finished=False):
        LOG.info("Scheduler:stopTask")
        self.monitor.lock.acquire()
        self.monitor.stopMonitor()
        if not self.done:
            if self.monitor.lastError():
                self.monitor.updateTaskState(task, error=self.monitor.lastError())
            elif finished:
                self.monitor.updateTaskState(task, finished=True)
            else:
                self.monitor.updateTaskState(task, runnable=True)
                pass
            pass

        self.monitor.lock.release()
        pass

    def stopPending(self, pending=None):
        LOG.info("Scheduler:stopPending")
        self.monitor.lock.acquire()
        self.monitor.stopMonitor()
        if not self.done:
            self.monitor.updateOpStatus(pending=pending);
            pass

        self.monitor.lock.release()
        if pending and pending.status != "paused":
            sys.exit(2)
            pass
        pass

    def startPending(self, pending):
        #
        # Lock out the heartbeat while making the changes.
        #
        self.monitor.lock.acquire()
        if pending and pending.parameters:
            self.monitor.updateParamsFromAnyObject(pending.parameters)
            pass

        # Start the monitor with new params
        self.monitor.startMonitor()
        self.monitor.updateOpStatus(pending)
        self.monitor.lock.release()
        pass
    pass
        

#
# Heartbeat task
# 1) Signal we are alive. Duh.
# 2) Advertise what can be configured.
# 3) What else?
#
class HeartBeat(threading.Thread):
    def __init__(self, monitor):
        super(HeartBeat, self).__init__()
        self.monitor = monitor
        self.done    = False
        pass
    
    def run(self):
        LOG.info("Heartbeat starting")
        lastackby = 0
        
        while not self.done:
            #
            # Count down till when the next heartbeat needs to go.
            # Simple for now, maybe a timeout later.
            #
            ackby    = self.monitor.state.status_ack_by
            now      = datetime.datetime.now(datetime.timezone.utc)
            if ackby == lastackby:
                # Something went wrong last update. Keep trying
                seconds = 60
            else:
                lastackby = ackby
                duration  = ackby - now
                seconds   = duration.total_seconds()
                pass

            LOG.debug("Heartbeat %r %r %r", ackby, now, seconds)

            if seconds >= 60:
                seconds = seconds - 30
                LOG.info("Heartbeat sleeping for %r seconds", seconds)
                while seconds > 0:
                    time.sleep(2)
                    if self.done:
                        break
                    seconds -= 2
                    pass
                LOG.info("Heartbeat done sleeping")
                pass
            
            if self.done:
                break;

            self.monitor.updateOpStatus()
            LOG.info("Heartbeat next ackby is %r",
                     self.monitor.state.status_ack_by)
            pass
        LOG.info("Heartbeat thread is exiting")
        pass

    def stop(self):
        LOG.info("HeartBeat stopping")
        self.done = True
        # Fix this.
        time.sleep(3)
        pass

    pass

#
# Monitor thread. Run the monitor, looping until told to stop.
#
class Monitor:
    def __init__(self, progargs, config, dstclient, zmcclient):
        self.monitor_id  = progargs.monitor_id
        self.element_token = progargs.element_token
        self.dst_http    = progargs.dst_http
        self.zms_format  = progargs.zms_format
        self.zms_kind    = progargs.zms_task
        self.monitor     = None
        self.state       = None
        self.status      = None
        self.pending_id  = None
        self.description = progargs.monitor_description
        self.dstclient   = dstclient
        self.zmcclient   = zmcclient
        self.min_freq    = config.START_FREQ
        self.max_freq    = config.END_FREQ
        self.gain        = config.GAIN
        self.interval    = config.INTERVAL
        self.repeat      = config.REPEAT
        self.dwelltime   = config.DWELLTIME
        self.dynamic     = not progargs.no_dynamic
        self.is_powder   = progargs.is_powder
        self.lock        = threading.RLock()
        self._stop       = False
        self.receiver    = None
        self.config      = config
        self.progargs    = progargs
        self.thread      = None
        self.zmstype     = None
        self.lasterror   = None
        self.notifyExit  = None

        #
        # Must map outer monitor to inner monitor for rdzinrdz.
        # This test is bogus.
        #
        # This will raise an exception if it fails
        #
        if self.is_powder and self.zmcclient._base_url.find("rdz.powderwireless.net") < 0:
            self.mapOuterMonitor();
            pass

        #
        # In dynamic mode we need our current state from ZMC.
        #
        if self.dynamic:
            # This will throw an error
            self.getState()
            #
            # When starting up, if the status is not active or paused,
            # then we force active. In other words, only "paused" means
            # the monitor should not send observations.
            #
            if self.status != "paused" and self.status != "active":
                self.status = "active"
                pass
        else:
            # Not dynamic, always start up.
            self.status = "active";
            pass

        self.receiver = Receiver(self.progargs, LOG, self.config)
        pass

    # Update parameters from an any_object. Yuck
    def updateParamsFromAnyObject(self, parameters):
        params = parameters.to_dict()
        
        if "dwelltime" in params:
            self.dwelltime = params["dwelltime"]
        else:
            self.dwelltime = None
            pass

        if "gain" in params:
            self.gain = params["gain"]
        else:
            self.gain = 15
            pass

        if "interval" in params:
            self.interval = params["interval"]
        else:
            self.interval = 5
            pass

        if "center_freq" in params:
            self.min_freq = self.max_freq = params["center_freq"]
        elif "min_freq" in params and "max_freq" in params:
            self.min_freq = params["min_freq"]
            self.max_freq = params["max_freq"]
        else:
            self.min_freq = 3350000000
            self.max_freq = 3550000000
            pass
        pass

    # For a taskable monitor, need to find the Action and set kind/format.
    def updateSettingsFromTask(self, task):
        action = self.getTaskAction(task.monitor_action_id)
        if action:
            self.zms_format = action.format_
            self.zms_kind   = action.kind
            pass
        pass

    def getTaskAction(self, monitor_action_id):
        for i, action in enumerate(self.monitor.actions):
            if action.id == monitor_action_id:
                return action
            pass
        LOG.error("getTaskAction: No matching action for %r", monitor_action_id)
        return None

    def run(self):
        self._stop = False
        self.lasterror = None

        # Need to update the configuration in dynamic mode.
        if self.dynamic:
            range = str(self.min_freq) + "-" + str(self.max_freq)

            # The Receiver will change only if the params require
            # re-initializing the device
            self.receiver = self.receiver.updateConfig(
                range=range, gain=self.gain, dwelltime=self.dwelltime)
            pass

        while not self._stop:
            LOG.info("Monitor doing something")

            try:
                item = None
                if self.zms_kind == "spectrogram":
                    item = self.receiver.get_full_spectrum().compute_spectrogram()
                else:
                    item = self.receiver.get_full_spectrum_psd()
                LOG.debug("Monitor is done")
                
                if self._stop:
                    break

                if self.zms_format == "csv" and self.progargs.zms_task == "psd":
                    self.upload_legacy(item)
                else:
                    self.upload_sigmf(item)

                #
                # If we were only instructed to run a certain number of
                # iterations, see if we should stop.
                #
                # A repeat setting may not make sense for a configurable or
                # taskable monitor.
                #
                if self.repeat > 0:
                    self.repeat -= 1
                    if self.repeat == 0:
                        self._stop = True
                        break

                #
                # We do not want to blocking sleep for a long time since then
                # we might end up waiting for a long time to reconfig.
                #
                if self.interval:
                    count = self.interval
                    while count > 0:
                        time.sleep(1)
                        if self._stop:
                            break
                        count -= 1
                        pass
                    pass
                pass
            except Exception as ex:
                LOG.error("Monitor Exception")
                LOG.exception(ex)
                self.lasterror = str(ex)
                break
            pass

        LOG.info("Monitor thread is exiting")
        self.notifyExit()
        pass

    def isRunning(self):
        return self.thread and self.thread.is_alive()

    def lastError(self):
        return self.lasterror

    def setStatus(self, status, notify=False):
        LOG.info("Monitor setStatus %r %r", status, notify)

        self.lock.acquire()
        self.status = status
        if notify:
            self.updateOpStatus()
            pass
        self.lock.release()
        pass

    def stop(self):
        self.stopMonitor()
        pass

    # Start the monitor in a new thread/task
    def startMonitor(self):
        LOG.info("Monitor:startMonitor: current status: %r", self.status)
        if not self.thread:
            self.thread = threading.Thread(target=self.run)
            self.thread.start()
            pass
        pass

    def stopMonitor(self):
        LOG.info("Monitor:stopMonitor: current status: %r", self.status)
        if self.thread:
            self._stop = True            
            self.thread.join(timeout=10)
            self.thread = None
            LOG.info("Monitor has stopped")
            pass
        pass
    
    def upload_sigmf(self, psd):
        metadata = psd.make_sigmf_metadata()
        LOG.debug(metadata)
        arch = psd.make_sigmf_archive(gzip=False)

        # Send POST request
        headers = {
            'Content-Type': 'application/sigmf-archive',
            'Content-Encoding': 'gzip',
            'X-Api-Token': self.element_token,
            'X-Api-Monitor-Id': self.monitor_id
        }
        url = self.dst_http + "/observations"
        response = httpx.post(url, data=arch, headers=headers)
        LOG.debug(response)
        LOG.debug(response.status_code)
        LOG.debug(response.text)
    
    #
    # Construct the observation and send it up.
    #
    def upload_legacy(self, psd):
        min_freq = None
        max_freq = None
        header   = "frequency,power,center_freq";
        psdIter  = iter(psd)

        # First line is the min frequency
        first    = next(psdIter)
        last_freq = first.frequency
        last_center_freq = first.center_freq
        min_freq = first.frequency
        data     = header + "\n"
        data    += "{},{},{}\n".format(first.frequency,first.power,first.center_freq)

        freq_step = 0
        for freq in psdIter:
            if last_center_freq != freq.center_freq:
                LOG.debug(f"center_freq change: {freq.center_freq}")
                last_center_freq = freq.center_freq
            if (freq.frequency - last_freq) != freq_step:
                new_step = freq.frequency - last_freq
                LOG.debug(f"freq_step change: {freq_step} -> {new_step} ({freq.frequency},{freq.center_freq})")
                freq_step = new_step
            last_freq = freq.frequency
            line = "{},{},{}".format(freq.frequency,freq.power,freq.center_freq)
            #LOG.debug(line)
            data += line + "\n"
            max_freq = freq.frequency
            pass

        observation = Observation(
            monitor_id  = self.monitor_id,
            description = self.description,
            kind        = ObservationKind.PSD,
            types       = "powder.rfmonitor.v2.psd",
            labels      = "ota,sweep",
            format_     = "csv",
            min_freq    = int(min_freq * 1000000),
            max_freq    = int(max_freq * 1000000),
            starts_at   = datetime.datetime.now(datetime.timezone.utc),
        )
        LOG.debug(str(observation))
        # After print
        observation.data = base64.b64encode(data.encode("ascii")).decode()

        LOG.info("Monitor pushing observation.data")
        response = self.dstclient.create_observation(body=observation)
        if not response:
            LOG.info("Could not create new observation")
            pass
        LOG.debug(response)
        pass

    #
    # When reporting to an RDZinRDZ, we have to map the outer monitor ID to
    # an inner monitor ID for the report.
    #
    def mapOuterMonitor(self):
        LOG.info("Mapping outer monitor ID to inner ID")
        
        inner = self.zmcclient.list_monitors(monitor=self.monitor_id)
        if not inner or not inner.monitors or len(inner.monitors) == 0:
            raise Exception("Could not map outer monitor to inner monitor")

        mon = inner.monitors[0]
        LOG.info("Mapped to inner monitor: %r", mon.id)
        self.monitor_id = mon.id
        pass

    #
    # Ask the ZNC for our state object.
    #
    def getState(self):
        monitor = self.zmcclient.get_monitor(self.monitor_id, elaborate=True)
        if not monitor:
            raise Exception("Could not get the monitor object from the ZMC")

        self.zmstype = monitor.type
        if not monitor.type or monitor.type == "static":
            self.status  = "active"
            self.zmstype = "static"
            return

        #LOG.info("getState %r", monitor)

        self.monitor = monitor
        # This will change over time.
        self.state = monitor.state

        #
        # David says:
        #
        # If Monitor.Pending is not null and
        #    Monitor.Pending.Id != Monitor.State.LastPendingId,
        #  set your Parameters to be Monitor.Pending.Parameters, and then
        #  heartbeat that you applied that PendingId.
        #
        # Or, if Monitor.Pending is null, just make sure to initialize your
        #  Parameters from Monitor.State.Parameters -- and make sure to pause
        #  immediately if Monitor.State.Status is paused instead of going active
        #
        # For a Taskable monitor, the scheduler will pull the initial task list.
        #
        # This is the basis for the first heartbeat.
        #
        if (not monitor.pending or
            monitor.pending.id == monitor.state.last_pending_id):

            # At the moment the params can be null, in which case better have
            # reasonable command line arguments.
            if monitor.state.parameters and self.zmstype == "configurable":
                self.updateParamsFromAnyObject(monitor.state.parameters)
                pass
            self.status = self.state.status
            if monitor.pending and monitor.pending.id == monitor.state.last_pending_id:
                self.pending_id = monitor.pending.id
                pass
            self.updateOpStatus()
            return

        # At the moment the params can be null, in which case better have
        # reasonable command line arguments.
        if monitor.pending.parameters and self.zmstype == "configurable":
            self.updateParamsFromAnyObject(monitor.pending.parameters)
            pass
        self.status = monitor.pending.status
        self.pending_id = monitor.pending.id
        self.updateOpStatus()
        pass

    # For the scheduler startup, give it the initial set of tasks
    def initialTaskList(self):
        return self.monitor.tasks
    
    def updateOpStatus(self, pending = None):
        LOG.info("Monitor UpdateOpStatus")

        self.lock.acquire()
        if pending:
            self.pending_id = pending.id
            self.status = pending.status
            pass

        # Too bad the generated api code could deal with plain strings.
        op_status = self.status
        if type(op_status) == str:
            op_status = MonitorOpStatus(op_status)
            pass
        
        opstatus = UpdateMonitorStateOpStatus(
            op_status = op_status,
            parameters = AnyObject.from_dict(
                src_dict={
                    "min_freq" : int(self.min_freq),
                    "max_freq" : int(self.max_freq),
                    "interval" : self.interval,
                    "gain"     : self.gain,
                })
        )
        # Are we responding to Pending.
        if self.pending_id:
            opstatus.last_pending_id = self.pending_id;
            self.pending_id = None
            pass

        LOG.info("Monitor UpdateOpStatus: %r", opstatus)

        try:
            state = self.zmcclient.update_monitor_state_op_status(
                monitor_id=self.monitor_id, body=opstatus)
        except Exception as ex:
            LOG.exception(ex)
            self.lock.release()
            return
        
        LOG.debug("Monitor UpdateOpStatus result: %r", state)

        # Not sure what to do when there are errors.
        if not state:
            LOG.error("Could not update monitor state")
        elif isinstance(state, Error):
            LOG.error("Error updating the monitor: %r", state)
        else:
            self.state = state
            pass
        
        self.lock.release()
        pass

    def updateTaskState(self, task, runnable=False, finished=False, error=None):
        LOG.info("Monitor updateTaskState %r %r %r %r",
                 task.id, runnable, finished, error);

        self.lock.acquire()

        taskstate = MonitorTaskState(
            monitor_task_id = task.id,
            runnable = runnable,
            finished = finished,
            error = False
        )
        if error:
            taskstate.error = True
            taskstate.message = error
            pass

        LOG.debug("Monitor updateTaskState: %r", taskstate)
        
        state = self.zmcclient.update_monitor_task_state(
            monitor_id=self.monitor_id, monitor_task_id=task.id, body=taskstate)
        
        LOG.debug("Monitor updateTaskState result: %r", state)
        
        if not state:
            LOG.error("Could not update monitor task state")
        elif isinstance(state, Error):
            LOG.error("Error updating the monitor task state: %r", state)
        else:
            task.state = state
            pass
        
        # Not sure what to do when there are errors.
        self.lock.release()
        pass

    #
    # We need to track Actions in case new ones are added while the
    # monitor is running. Also going to handle update here as well
    #
    # Delete is not really needed.
    #
    def addAction(self, newAction):
        for i, action in enumerate(self.monitor.actions):
            if action.id == newAction.id:
                self.monitor.actions[i] = newAction
                LOG.info("Action updated: %r", newAction.id)
                return
            pass
        LOG.info("New action added: %r", newAction.id)
        self.monitor.actions.append(newAction)
        pass

    pass

#
# Command task, Takes orders from the RDZ via events. 
# 1) Change the gain.
# 2) Change the frequency range
# 3) Change ...
#
class ZMCSubscriptionCallback(ZmsSubscriptionCallback):
    def __init__(self, zmcclient, scheduler, monitor, **kwargs):
        super(ZMCSubscriptionCallback, self).__init__(zmcclient, **kwargs)
        self.zmcclient  = zmcclient
        self.scheduler  = scheduler
        self.monitor    = monitor
        self.monitor_id = monitor.monitor_id
        self.done       = False
        pass

    async def start(self):
        LOG.info("ZMCSubscriptionCallback calling run_callbacks")
        await self.run_callbacks()

    async def on_event(self, ws, evt, message):
        if evt.header.source_type != EVENT_SOURCETYPE_ZMC:
            LOG.error("on_event: unexpected source type: %r (%r)",
                      evt.header.source_type, message)
            return

        #LOG.info("on_event: %r", evt)

        # Since we are subscribed to all ZMC events for this element,
        # there will be chatter we do not care about.
        if (evt.header.code != EVENT_CODE_MONITOR and
            evt.header.code != EVENT_CODE_MONITOR_PENDING and
            evt.header.code != EVENT_CODE_MONITOR_STATE and
            evt.header.code != EVENT_CODE_MONITOR_ACTION and
            evt.header.code != EVENT_CODE_MONITOR_TASK):
            return

        # Have to filter for the monitor_id, for now. Until the ZMC can
        # can handle the monitor subscriptions
        if evt.header.code == EVENT_CODE_MONITOR:
            if evt.object_.id != self.monitor_id:
                return
        elif evt.object_.monitor_id != self.monitor_id:
            return

        LOG.debug("on_event %r %r", evt.header.code, evt.header.type)

        try:
            if evt.header.code == EVENT_CODE_MONITOR_PENDING:
                self.handlePendingEvent(evt.object_)
            elif evt.header.code == EVENT_CODE_MONITOR_ACTION:
                self.handleActionEvent(evt.object_, evt.header.type)
            elif evt.header.code == EVENT_CODE_MONITOR_TASK:
                self.handleTaskEvent(evt.object_, evt.header.type)
                pass
        except Exception as ex:
            LOG.exception(ex)
            pass
        pass

    async def stop(self):
        LOG.info("ZMCSubscriptionCallback stop")
        self.done = True
        await self.disconnect()
        pass

    def handlePendingEvent(self, pending):
        LOG.info("handlePendingEvent: %r", pending)

        self.scheduler.handlePending(pending)
        pass

    def handleTaskEvent(self, task, typecode):
        LOG.info("handleTaskEvent: %r", typecode)
        if typecode == EVENT_TYPE_CREATED or typecode == EVENT_TYPE_DELETED:
            LOG.debug("handleTaskEvent: %r", task)
            pass

        if typecode == EVENT_TYPE_CREATED:
            self.scheduler.scheduleTask(task)
        elif typecode == EVENT_TYPE_DELETED:
            self.scheduler.deleteTask(task)
            pass
        pass
    
    def handleActionEvent(self, action, typecode):
        LOG.info("handleActionEvent: %r", typecode)
        if typecode == EVENT_TYPE_CREATED or typecode == EVENT_TYPE_UPDATED:
            LOG.debug("handleActionEvent: %r", action)
            pass

        # Leaving deleted actions around is harmless.
        if typecode == EVENT_TYPE_CREATED or typecode == EVENT_TYPE_UPDATED:
            self.monitor.addAction(action)
            pass
        pass
    
    pass

