#
# Configuration computed from the command line arguments
#
import math
import numpy as np

# Various defaults for parameters
DEFAULT_SCAN_RANGE = "100e6-6e9"
DEFAULT_FFT_SIZE   = 2**9
DEFAULT_INST_BW    = 30.72e6
DEFAULT_PSD_METHOD = "welch"
DEFAULT_ANALOG_BW_CUTOFF_FACTOR = 0.9

class Configuration:
    def __init__(self, args, device_config):
        self.args = args
        self.device_config = device_config
        self.init()
        pass

    def init(self):
        # maximum instantaneous bandwidth is equal to the maximum sampling rate
        self.MAX_INST_BW = self.args.inst_bw
        self.ANALOG_BW_CUTOFF_FACTOR = self.args.analog_bw_factor
        self.ANALOG_BW = self.MAX_INST_BW * self.ANALOG_BW_CUTOFF_FACTOR

        if self.args.lo_offset:
            self.LO_OFFSET = self.args.lo_offset
        else:
            self.LO_OFFSET = self.MAX_INST_BW / 2
            pass

        # This needs to be configurable.
        self.FFT_SIZE  = self.args.fft_size
        # reject frequencies above the analog bandwidth cutoff 
        self.N = int(np.floor(self.FFT_SIZE*self.ANALOG_BW_CUTOFF_FACTOR))
        if self.N % 2 == 1:
            self.N+=1  # N should be even
        # how many bins to remove on both sides of the array
        self.N_REMOVE = (self.FFT_SIZE-self.N)//2 
        
        if self.args.center:
            self.CENTER_FREQS = np.array([float(self.args.center)])
            self.START_FREQ = self.args.center
            self.END_FREQ = self.args.center + self.ANALOG_BW
        else:
            if not self.args.range:
                self.args.range = DEFAULT_SCAN_RANGE
                pass
        
            (a,b) = self.args.range.split("-")
            self.START_FREQ = float(a)
            self.END_FREQ = float(b)
            self.CENTER_FREQS = np.arange(self.START_FREQ, self.END_FREQ, self.ANALOG_BW - math.fmod(self.ANALOG_BW, self.ANALOG_BW / (self.FFT_SIZE * self.ANALOG_BW_CUTOFF_FACTOR)))
            pass
        
        self.NUM_STEPS = len(self.CENTER_FREQS)

        if self.args.dwelltime:
            self.NUM_FFTS = int((self.args.dwelltime * self.MAX_INST_BW) / self.FFT_SIZE)
        else:
            self.NUM_FFTS = 2**10
            pass

        self.DWELLTIME = self.args.dwelltime
        self.REPEAT = self.args.repeat
        self.INTERVAL = self.args.interval
        self.GAIN = self.args.gain
        self.N_BINS_OUT_PER_STEP = self.N
        pass

    #
    # This is for supporting ZMS configurable monitoring.
    #
    def update(self, range=None, gain=None, dwelltime=None):
        if range:
            (a,b) = range.split("-")
            if a == b:
                self.args.center = float(a)
                self.args.range = None
            else:
                self.args.center = None
                self.args.range = range
                pass
            pass
        if gain != None:
            self.args.gain = gain
            pass
        # If not specified, we want to return to default (which is None).
        self.args.dwelltime = dwelltime
        self.init()
        pass

    def dump(self):
        print(str(vars(self)))
        pass
    
    pass
