import json
import time
import sys
import signal
import argparse
import logging
import os
import os.path

# Only when a daemon
LOGFILE     = "/local/logs/rdz-monitor.log"
# Only when storing PSDs for a web server
WEBDIR      = "/local/www"
# Default configuration
CONFIGFILE  = "/etc/rfmonitor/device_config.json"

from .monitor import Monitor
from .config import *

def parse_args():
    parser = argparse.ArgumentParser(
        prog="rfmonitor",
        description="Run the RF monitor and report results")
    parser.add_argument(
        "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "-c", "--configfile", default=CONFIGFILE, type=str,
        help="Optional config file. Note that --antenna and --device override")
    parser.add_argument(
        "-d", "--debug", default=False, action='store_true',
        help="Turn on on copious debugging output")
    parser.add_argument(
        "-D", "--debug-all", default=False, action='store_true',
        help="Turn on on copious debugging output in all modules")
    parser.add_argument(
        "-g", "--gain", default=30, type=int,
        help="Set the receiver gain")
    parser.add_argument(
        "-f", "--output", default=None, type=str,
        help="Output file for samples or psd, defaults to STDOUT")
    parser.add_argument(
        "-R", "--range", default=DEFAULT_SCAN_RANGE, type=str,
        help="Spectrum range to scan. Defaults to " + DEFAULT_SCAN_RANGE)
    parser.add_argument(
        "-C", "--center", default=None, type=float,
        help="Instead of scanning a range, scan a single center frequency")
    parser.add_argument(
        "--dwelltime", default=None, type=float,
        help="Dwell time for each center frequency.")
    parser.add_argument(
        "--fft-size", default=DEFAULT_FFT_SIZE, type=int,
        help="FFT size. Defaults to " + str(DEFAULT_FFT_SIZE))
    parser.add_argument(
        "--inst-bw", default=DEFAULT_INST_BW, type=str,
        help="Instantaneous bandwidth. Defaults to " + str(DEFAULT_INST_BW))
    parser.add_argument(
        "--analog-bw-factor", default=DEFAULT_ANALOG_BW_CUTOFF_FACTOR, type=float,
        help="Anolog bandwidth cutoff factor. Defaults to " + str(DEFAULT_ANALOG_BW_CUTOFF_FACTOR))
    parser.add_argument(
        "--lo-offset", type=float,
        help="Override the default LO offset, which is INST_BW/2")
    parser.add_argument(
        "--psd-method", default=DEFAULT_PSD_METHOD, type=str,
        choices=["welch", "bartlett"],
        help="PSD method. Defaults to " + str(DEFAULT_PSD_METHOD))
    parser.add_argument(
        "--no-restart-streamer", default=False, action='store_true',
        help="Do not restart the streamer when retuning the center frequency. " +
        "Defaults to False")
    parser.add_argument(
        "--plot", default=False, action='store_true',
        help="Plot results in an X11 window instead of writing output")
    parser.add_argument(
        "--csv", default=False, action='store_true',
        help="Write a CSV file of the results and exit")
    parser.add_argument(
        "--spectrogram", default=False, action='store_true',
        help="Generate a spectrogram instead of PSD")
    parser.add_argument(
        "--antenna", default=None, type=str,
        help="Antenna to use (RX2 or TX/RX). Defaults to channel 0 for now")
    parser.add_argument(
        "--device", default=None, type=str,
        help="Device name for output. For example: nuc1:rf0. Defaults to antenna")
    parser.add_argument(
        "--interval", default=10, type=int,
        help="Interval between runs when running more then once (--repeat)")
    parser.add_argument(
        "--repeat", type=int, default=0,
        help="Repeat count, zero means run forever")

    #
    # These arguments are optional, and enable ZMS mode. 
    #
    parser.add_argument(
        "--zms-mode", default=False, action='store_true',
        help="Enable ZMS mode, be sure to provide all the other args")
    parser.add_argument(
        "--is-powder", default=False, action='store_true',
        help="Enable POWDER-RDZ mode (and check for RDZ-in-RDZ nested case)")
    parser.add_argument(
        "--monitor-id", default=None, type=str, 
        help="ZMC monitor ID")
    parser.add_argument(
        "--monitor-description", default=None, type=str,
        help="ZMC monitor description string")
    parser.add_argument(
        "--element-token", default=None, type=str,
        help="Element token")
    parser.add_argument(
        "--zmc-http", default=None, type=str,
        help="ZMC URL")
    parser.add_argument(
        "--dst-http", default=None, type=str,
        help="DST URL")
    parser.add_argument(
        "--zms-format", default="csv", type=str,
        choices=["sigmf", "csv"],
        help="OpenZMS format (sigmf or legacy powder-specific csv)")
    parser.add_argument(
        "--zms-task", default="psd", type=str,
        choices=["psd", "spectrogram", "samples"],
        help="Static collection task for OpenZMS mode (psd, spectrogram, samples)")
    parser.add_argument(
        "--no-dynamic", default=False, action="store_true")

    #
    # These arguments are optional, and enable WWW Server mode. 
    #
    parser.add_argument(
        "--www-mode", default=False, action='store_true',
        help="Enable WWW/WEB mode, be sure to provide all the other args")
    parser.add_argument(
        "--www-dir", default=WEBDIR, type=str,
        help="Local directory to write PSD file to")
    
    args = parser.parse_args(sys.argv[1:])

    if args.is_powder and not args.zmc_http:
        parser.print_help()
        print("\nError: --zmc-http is required when --is-powder is set")
        exit(1)

    return args

def main():
    args = parse_args()

    #
    # One logger for all
    #
    format = "[%(levelname)s] [%(module)s] %(message)s"
    if args.debug or args.debug_all:
        level = logging.DEBUG
    else:
        level = logging.INFO
        pass

    if args.debug_all:
        logging.basicConfig(level=level, format=format)
    else:
        logging.basicConfig(format=format)
        pass
    logger = logging.getLogger("monitor")
    logger.setLevel(level)
    
    monitor = Monitor(args, logger).start()
    pass

if __name__ == "__main__":
    main()
