from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.constraint import Constraint
from ...models.error import Error
from ...types import UNSET, Response, Unset


def _get_kwargs(
    grant_id: str,
    *,
    body: Constraint,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_force_update: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    if not isinstance(x_api_force_update, Unset):
        headers["X-Api-Force-Update"] = x_api_force_update

    _kwargs: Dict[str, Any] = {
        "method": "post",
        "url": "/grants/{grant_id}/constraints".format(
            grant_id=grant_id,
        ),
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Constraint, Error]]:
    if response.status_code == HTTPStatus.CREATED:
        try:
            response_201 = Constraint.from_dict(response.json())

            return response_201
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.CONFLICT:
        try:
            response_409 = Error.from_dict(response.json())

            return response_409
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Constraint, Error]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    client: AuthenticatedClient,
    grant_id: str,
    *,
    body: Constraint,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_force_update: Union[Unset, str] = UNSET,
) -> Response[Union[Constraint, Error]]:
    """Create a new constraint on an existing unsubmitted grant object.  Will fail if grant has been
    submitted to the scheduler.

    Args:
        grant_id (str): The target grant.
        x_api_elaborate (Union[Unset, str]):
        x_api_force_update (Union[Unset, str]):  Example: true.
        body (Constraint): A spectrum constraint.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Constraint, Error]]
    """

    kwargs = _get_kwargs(
        grant_id=grant_id,
        body=body,
        x_api_elaborate=x_api_elaborate,
        x_api_force_update=x_api_force_update,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    client: AuthenticatedClient,
    grant_id: str,
    *,
    body: Constraint,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_force_update: Union[Unset, str] = UNSET,
) -> Optional[Union[Constraint, Error]]:
    """Create a new constraint on an existing unsubmitted grant object.  Will fail if grant has been
    submitted to the scheduler.

    Args:
        grant_id (str): The target grant.
        x_api_elaborate (Union[Unset, str]):
        x_api_force_update (Union[Unset, str]):  Example: true.
        body (Constraint): A spectrum constraint.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Constraint, Error]
    """

    return sync_detailed(
        grant_id=grant_id,
        client=client,
        body=body,
        x_api_elaborate=x_api_elaborate,
        x_api_force_update=x_api_force_update,
    ).parsed


async def asyncio_detailed(
    client: AuthenticatedClient,
    grant_id: str,
    *,
    body: Constraint,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_force_update: Union[Unset, str] = UNSET,
) -> Response[Union[Constraint, Error]]:
    """Create a new constraint on an existing unsubmitted grant object.  Will fail if grant has been
    submitted to the scheduler.

    Args:
        grant_id (str): The target grant.
        x_api_elaborate (Union[Unset, str]):
        x_api_force_update (Union[Unset, str]):  Example: true.
        body (Constraint): A spectrum constraint.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Constraint, Error]]
    """

    kwargs = _get_kwargs(
        grant_id=grant_id,
        body=body,
        x_api_elaborate=x_api_elaborate,
        x_api_force_update=x_api_force_update,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    client: AuthenticatedClient,
    grant_id: str,
    *,
    body: Constraint,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_force_update: Union[Unset, str] = UNSET,
) -> Optional[Union[Constraint, Error]]:
    """Create a new constraint on an existing unsubmitted grant object.  Will fail if grant has been
    submitted to the scheduler.

    Args:
        grant_id (str): The target grant.
        x_api_elaborate (Union[Unset, str]):
        x_api_force_update (Union[Unset, str]):  Example: true.
        body (Constraint): A spectrum constraint.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Constraint, Error]
    """

    return (
        await asyncio_detailed(
            grant_id=grant_id,
            client=client,
            body=body,
            x_api_elaborate=x_api_elaborate,
            x_api_force_update=x_api_force_update,
        )
    ).parsed
