#!/bin/python

import attrs
import sys
import argparse
import logging
from typing import cast, Optional, Dict, Any, Union

from zmsclient.zmc.client import ZmsZmcClient
from zmsclient.dst.client import ZmsDstClient
from zmsclient.zmc.v1.models import Grant, Constraint
from zmsclient.zmc.v1.models import Spectrum, SpectrumList, SpectrumConstraint, Policy
from zmsclient.zmc.v1.models import Location, LocationList
from zmsclient.zmc.v1.models import Antenna, AntennaList
from zmsclient.zmc.v1.models import Radio, RadioList, RadioPort
from zmsclient.zmc.v1.models import Monitor, MonitorList, MonitorAction
from zmsclient.zmc.v1.models import Action, ActionList
from zmsclient.dst.v1.models import Metric, MetricList
from zmsclient.zmc.v1.types import Unset as ZmcUnset
from zmsclient.common import DefaultDestEnvAction

LOG = logging.getLogger(__name__)

def fatal(msg: str, *args, **kwargs):
    LOG.error(msg, *args, **kwargs)
    exit(-1)

# Map outer IDs to inner IDs and back.
@attrs.define
class ObjectMap:
    """Map outer IDs to inner IDs and back."""
    id_map = dict()

    def add_object(self, inner: Any, outer: Any):
        self.id_map[inner.id] = outer.id
        self.id_map[outer.id] = inner.id

    def map_to_inner_id(self, outer_id: Union[ZmcUnset, None, str]) -> Union[ZmcUnset, None, str]:
        # If this is an UNSET or None value, just return it
        if not outer_id:
            return outer_id
        return self.id_map.get(outer_id, None)

#
# Grab our grant, this becomes the initial spectrum
#
def importSpectrum(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient,
                   element_id: str, parent_grant_list: str,
                   objmap: ObjectMap, impotent: bool = False):
    for outer_id in parent_grant_list.split(","):
        resp = outer_zmc_client.get_grant(outer_id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Grant):
            fatal("Could not get grant %r: %r", outer_id, resp)
        grant = cast(Grant, resp.parsed)

        #
        # We need to follow the replacement chain to find the current
        # grant.
        #
        current_outer_id = outer_id
        while grant.replacement:
            current_outer_id = grant.replacement.new_grant_id
            resp = outer_zmc_client.get_grant(current_outer_id, x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, Grant):
                fatal("Could not get replacement grant %r: %r", current_outer_id, resp)
            grant = cast(Grant, resp.parsed)

        #
        # Use the original grant ID in the name to key on.
        #
        resp = inner_zmc_client.list_spectrum(
            element_id=element_id, spectrum="Outer RDZ grant: " + outer_id,
            x_api_elaborate="True")
        if resp.is_success and isinstance(resp.parsed, SpectrumList):
            inner_spectrum = cast(SpectrumList, resp.parsed)

        #
        # Check if already added.
        #
        if (inner_spectrum and inner_spectrum.spectrum and
            len(inner_spectrum.spectrum)):
            spec = inner_spectrum.spectrum[0]
            LOG.info("Spectrum already exists: %r", spec.id)
            LOG.debug("Fetched inner spectrum: %r", spec)
            #
            # if the grant has been replaced, lets update the expiration. 
            #
            if spec.ext_id != current_outer_id:
                LOG.info("Updating spectrum expiration: %r", grant.expires_at)
                spec.ext_id = current_outer_id
                spec.expires_at = grant.expires_at
                if not impotent:
                    resp = inner_zmc_client.update_spectrum(
                        spectrum_id=str(spec.id), body=spec)
                    LOG.debug("updated spectrum: %r", resp)
            continue

        resp = outer_zmc_client.get_grant(current_outer_id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Grant):
            fatal("Could not get grant: %r ", current_outer_id,)
        else:
            grant = resp.parsed

        # Restrict the inner spectrum based on the inner OpenZMS grant.
        constraints = []
        for gc in (grant.constraints or []):
            if not gc.constraint:
                fatal("Grant constraint has no constraint: %r", gc)
            else:
                constraint = gc.constraint
            new = Constraint(min_freq=constraint.min_freq,
                            max_freq=constraint.max_freq,
                            max_eirp=constraint.max_eirp,
                            exclusive=True)
            constraints.append(SpectrumConstraint(constraint=new))

        # Stub policy, do not think I need to worry too much about this
        policy = Policy(element_id=element_id,
                        allowed=True,
                        auto_approve=True,
                        disable_emit_check=True,
                        priority=500)

        LOG.info("Spectrum does not exist yet: %r", outer_id)

        # Lets set the URL to the outer grant page.
        outer_grant_url = outer_zmc_client._base_url + "/grants/" + outer_id

        spectrum = Spectrum(name="Outer RDZ grant: " + outer_id,
                        description=outer_zmc_client._base_url,
                        enabled=True,
                        url=outer_grant_url,
                        ext_id=current_outer_id,
                        element_id=element_id,
                        starts_at=grant.starts_at,
                        expires_at=grant.expires_at,
                        constraints=constraints,
                        policies=[policy])
        
        LOG.debug("Creating inner spectrum: %r", spectrum)
        if impotent:
            return

        resp = inner_zmc_client.create_spectrum(body=spectrum)
        if not resp.is_success or not isinstance(resp.parsed, Spectrum):
            fatal("Could not create new spectrum")
    
#
# Copy locations in.
#
def importLocations(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient,
                    element_id: str, objmap: ObjectMap, name: Optional[str] = None,
                    impotent: bool = False):
    kwargs = dict()
    if name:
        kwargs["name"] = name
    resp = outer_zmc_client.list_locations(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, LocationList):
        fatal("Could not list outer locations: %r", resp)
    locationList = cast(LocationList, resp.parsed)

    for location in locationList.locations:
        #
        # Check if already added.
        #
        resp = inner_zmc_client.list_locations(items_per_page=1000,
            element_id=element_id, name=location.name)
        if not resp.is_success or not isinstance(resp.parsed, LocationList):
            fatal("Could not list inner locations: %r", resp)
        innerLocationList = cast(LocationList, resp.parsed)
        if innerLocationList.locations and len(innerLocationList.locations):
            loc = innerLocationList.locations[0]
            LOG.info("Location already exists: %r, %r", loc.name, loc.id)
            LOG.debug("Location: %r", loc)
            objmap.add_object(location, loc)            
            continue

        LOG.info("Location does not exist yet: %r", location.name)
        if impotent:
            continue

        newloc = Location(name=location.name,
                          element_id=element_id,
                          srid=location.srid,
                          x=location.x,
                          y=location.y,
                          z=location.z)
        LOG.debug("Creating location: %r", newloc)
    
        resp = inner_zmc_client.create_location(body=newloc)
        if not resp.is_success or not isinstance(resp.parsed, Location):
            fatal("Could not create location: %r", location.id)
        else:
            newloc = resp.parsed

        objmap.add_object(location, resp.parsed)
        LOG.info("Created location: %r", newloc.id)
        LOG.debug("Created location: %r", newloc)

#
# Copy locations in.
#
def importAntennas(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient,
                   element_id: str, objmap: ObjectMap, name: Optional[str] = None,
                   impotent: bool = False):
    kwargs = dict()
    if name:
        kwargs["antenna"] = name
    resp = outer_zmc_client.list_antennas(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, AntennaList):
        fatal("Could not list outer antennas: %r", resp)
    antennas = cast(AntennaList, resp.parsed)
    for antenna in antennas.antennas:
        #
        # Check if already added.
        #
        resp = inner_zmc_client.list_antennas(items_per_page=1000,
            element_id=element_id, antenna=antenna.name)
        if not resp.is_success or not isinstance(resp.parsed, AntennaList):
            fatal("Could not list inner antennas: %r", resp)
        innerAntennaList = cast(AntennaList, resp.parsed)
        if innerAntennaList and len(innerAntennaList.antennas):
            ant = innerAntennaList.antennas[0]
            LOG.info("Antenna already exists: %r, %r", ant.name, ant.id)
            LOG.debug("Antenna: %r", ant)
            objmap.add_object(antenna, ant)            
            continue

        LOG.info("Antenna does not exist yet: %r", antenna.name)
        if impotent:
            continue
        
        newantenna = Antenna(name=antenna.name,
                             element_id=element_id,
                             description=antenna.description,
                             type=antenna.type,
                             vendor=antenna.vendor,
                             model=antenna.model,
                             gain=antenna.gain,
                             beam_width=antenna.beam_width)
        LOG.debug("Creating new antenna: %r", newantenna)

        resp = inner_zmc_client.create_antenna(body=newantenna)
        if not resp.is_success or not isinstance(resp.parsed, Antenna):
            fatal("Could not create antenna: %r", antenna.id)
        else:
            newantenna = resp.parsed

        objmap.add_object(antenna, newantenna)
        LOG.info("Created antenna: %r", newantenna.id)
        LOG.debug("Created antenna: %r", newantenna)

#
# Copy radios (and their ports) in.
#
def importRadios(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient,
                 element_id: str, objmap: ObjectMap, name: Optional[str] = None,
                 impotent: bool = False):
    kwargs = dict()
    if name:
        kwargs["radio"] = name
    resp = outer_zmc_client.list_radios(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, RadioList):
        fatal("Could not list outer radios: %r", resp)
    radioList = cast(RadioList, resp.parsed)
    for radio in radioList.radios:
        if not radio.enabled:
            continue

        # Reload with elaborate so we get the ports.
        resp = outer_zmc_client.get_radio(str(radio.id), x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Radio):
            fatal("Could not reload radio: %r", radio.id)
        else:
            radio = resp.parsed

        #
        # Check if already added.
        #
        resp = inner_zmc_client.list_radios(items_per_page=1000,
            element_id=element_id, radio=radio.id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, RadioList):
            fatal("Could not list inner radios: %r", resp)
        innerRadioList = cast(RadioList, resp.parsed)

        if innerRadioList and innerRadioList.radios and len(innerRadioList.radios) > 0:
            innerRadio = innerRadioList.radios[0]
            LOG.info("Radio already exists: %r, %r", innerRadio.name, innerRadio.id)
            LOG.debug("Radio: %r", innerRadio)

            # Reload with elaborate so we get the ports.
            resp = inner_zmc_client.get_radio(str(innerRadio.id), x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, Radio):
                fatal("Could not reload inner radio: %r", innerRadio.id)
            else:
                rad = resp.parsed
            objmap.add_object(radio, innerRadio)

            #
            # Need mappings for the ports, for monitors below.
            # Need to deal with port changes (additions,deletions). Later
            #

            if not radio.ports or len(radio.ports) == 0:
                LOG.warning("Outer radio has no ports: %r", radio.id)
                continue
            for port in radio.ports:
                found = False
                if innerRadio.ports and len(innerRadio.ports) > 0:
                    for iport in innerRadio.ports:
                        if iport.device_id == port.id:
                            LOG.info("Radio Port already exists: %r, %r", iport.name, iport.id)
                            objmap.add_object(port, iport)
                            found = True
                            break
                if not found:
                    # Add the port
                    newport = RadioPort(
                        name=radio.name,
                        radio_id=str(innerRadio.id),
                        device_id=port.id,
                        tx=port.tx,
                        rx=port.rx,
                        enabled=port.enabled,
                        min_freq=port.min_freq,
                        max_freq=port.max_freq,
                        max_power=port.max_power,
                        antenna_id=objmap.map_to_inner_id(port.antenna_id),
                        antenna_location_id=objmap.map_to_inner_id(port.antenna_location_id))
                    LOG.debug("Creating new radio port: %r", newport)

                    resp = inner_zmc_client.create_radio_port(str(innerRadio.id), body=newport)
                    if not resp.is_success or not isinstance(resp.parsed, RadioPort):
                        fatal("Could not create radio port: %r (%r)", port.id, resp.parsed)
                    else:
                        newport = resp.parsed
                    LOG.debug("Created new radio port: %r", newport)

                    objmap.add_object(port, newport)
                    LOG.info("Created radio port: %r", newport.id)

            continue

        LOG.info("Radio does not exist yet: %r", radio.name)
        if impotent:
            continue
        
        #
        # Need to create the local radio before we can add ports
        #
        newradio = Radio(name=radio.name,
                         element_id=element_id,
                         description=radio.description,
                         device_id=radio.id,
                         enabled=radio.enabled,
                         location_id=objmap.map_to_inner_id(radio.location_id))
        LOG.debug("Creating new radio: %r", newradio)

        resp = inner_zmc_client.create_radio(body=newradio)
        if not resp:
            fatal("Could not create radio: %r", radio.id)
        else:
            newradio = cast(Radio, resp.parsed)
        LOG.debug("Created new radio: %r", newradio)

        objmap.add_object(radio, newradio)
        LOG.info("Created radio: %r", newradio.id)

        if not radio.ports or len(radio.ports) == 0:
            continue

        radio_id = str(newradio.id)
        for port in radio.ports:
            newport = RadioPort(
                name=radio.name,
                radio_id=radio_id,
                device_id=port.id,
                tx=port.tx,
                rx=port.rx,
                enabled=port.enabled,
                min_freq=port.min_freq,
                max_freq=port.max_freq,
                max_power=port.max_power,
                antenna_id=objmap.map_to_inner_id(port.antenna_id),
                antenna_location_id=objmap.map_to_inner_id(port.antenna_location_id))
            LOG.debug("Creating new radio port: %r", newport)

            resp = inner_zmc_client.create_radio_port(radio_id, body=newport)
            if not resp.is_success or not isinstance(resp.parsed, RadioPort):
                fatal("Could not create radio port: %r", port.id)
            else:
                newport = resp.parsed
            LOG.debug("Created new radio port: %r", newport)

            objmap.add_object(port, newport)
            LOG.info("Created radio port: %r", newport.id)

#
# Load Actions which are first class.
#
def importActions(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient,
                  element_id: str, objmap: ObjectMap, name: Optional[str] = None,
                  impotent: bool = False):
    
    kwargs: Dict[str, Any] = dict(include_public=True)
    if name:
        kwargs["action"] = name
    resp = outer_zmc_client.list_actions(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, ActionList):
        fatal("Could not list outer actions: %r", resp)
    actions = cast(ActionList, resp.parsed)
    for action in actions.actions:
        #
        # Check if already added.
        #
        resp = inner_zmc_client.list_actions(
            action=action.id, items_per_page=1000)
        if not resp.is_success or not isinstance(resp.parsed, ActionList):
            fatal("Could not list inner actions: %r", resp)
        innerActionList = cast(ActionList, resp.parsed)
        
        if innerActionList and innerActionList.actions and len(innerActionList.actions):
            act = innerActionList.actions[0]
            LOG.info("Action already exists: %r, %r", act.name, act.id)
            LOG.debug("Action: %r", act)

            objmap.add_object(action, act)
            continue
        
        LOG.info("Action does not exist yet: %r", action.name)
        if impotent:
            continue
        
        # XXX We have to put the outer id into the inner description
        description = str(action.id)
        if action.description:
            description = action.description + " : " + description
        newaction = Action(name=action.name,
                           element_id=element_id,
                           description=description,
                           html_url=action.html_url,
                           kind=action.kind,
                           type=action.type,
                           format_=action.format_,
                           parameter_defs=action.parameter_defs,
                           # This seems okay for an inner ZMS,
                           # Also, the non-admin token cannot set it anyway
                           is_public=False)
        LOG.debug("Creating new action: %r", newaction)

        resp = inner_zmc_client.create_action(body=newaction)
        LOG.debug("%r", resp)
        if not resp.is_success or not isinstance(resp.parsed, Action):
            fatal("Could not create action: %r", action.id)
        newaction = cast(Action, resp.parsed)

        objmap.add_object(action, newaction)
        LOG.info("Created action: %r", newaction.id)
        LOG.debug("Created action: %r", newaction)

#
# Copy monitors in.
#
def importMonitors(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient,
                   element_id: str, objmap: ObjectMap, name: Optional[str] = None,
                   impotent: bool = False):
    kwargs = dict()
    if name:
        kwargs["monitor"] = name
    resp = outer_zmc_client.list_monitors(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MonitorList):
        fatal("Could not list outer monitors: %r", resp)
    monitorList = cast(MonitorList, resp.parsed)
    for monitor in monitorList.monitors:
        if not monitor.enabled:
            continue
        
        # Reload with elaborate so we get the ports.
        resp = outer_zmc_client.get_monitor(str(monitor.id), x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Monitor):
            fatal("Could not reload monitor: %r", monitor.id)
        monitor = cast(Monitor, resp.parsed)

        #
        # Check if already added.
        #
        resp = inner_zmc_client.list_monitors(
            element_id=element_id, monitor=monitor.id, items_per_page=1000,
            x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, MonitorList):
            fatal("Could not list inner monitors: %r", resp)
        innerMonitorList = cast(MonitorList, resp.parsed)
        
        if innerMonitorList and innerMonitorList.monitors and len(innerMonitorList.monitors):
            mon = innerMonitorList.monitors[0]
            LOG.info("Monitor already exists: %r, %r", mon.name, mon.id)
            LOG.debug("Monitor: %r", mon)

            # Reload with elaborate so we get the ports.
            resp = inner_zmc_client.get_monitor(str(mon.id), x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, Monitor):
                fatal("Could not reload inner radio: %r", mon.id)
            mon = cast(Monitor, resp.parsed)
            objmap.add_object(monitor, mon)
            continue

        LOG.info("Monitor does not exist yet: %r", monitor.name)
        if impotent:
            continue
        
        #
        # Need to create the local radio before we can add ports
        #
        radio_port_id = None
        if monitor.radio_port_id:
            radio_port_id = objmap.map_to_inner_id(monitor.radio_port_id)
            if not radio_port_id:
                fatal("No radio_port_id mapping for monitor: %r", monitor.name)

        monitored_radio_port_id = None
        if monitor.monitored_radio_port_id:
            monitored_radio_port_id = objmap.map_to_inner_id(monitor.monitored_radio_port_id)
            if not monitored_radio_port_id:
                fatal("No monitored_radio_port_id mapping for monitor: %r", monitor.name)
        
        newmon = Monitor(name=monitor.name,
                         element_id=element_id,
                         description=monitor.description,
                         device_id=monitor.id,
                         enabled=True,
                         type=monitor.type,
                         types=monitor.types,
                         formats=monitor.formats,
                         parameter_defs=monitor.parameter_defs,
                         auto_managed=False)
        if radio_port_id:
            newmon.radio_port_id = radio_port_id
        if monitored_radio_port_id:
            newmon.monitored_radio_port_id = monitored_radio_port_id
        LOG.debug("Creating new monitor: %r", newmon)

        resp = inner_zmc_client.create_monitor(body=newmon)
        if not resp.is_success or not isinstance(resp.parsed, Monitor):
            fatal("Could not create monitor: %r (%r)" % (monitor.id, resp.parsed))
        newmon = cast(Monitor, resp.parsed)
        LOG.debug("Created new monitor: %r", newmon)

        objmap.add_object(monitor, newmon)
        LOG.info("Created monitor: %r", newmon.id)

        #
        # Now we can do Monitor Actions.
        #
        if not monitor.actions or len(monitor.actions) == 0:
            continue
        for action in monitor.actions:
            action_id = objmap.map_to_inner_id(action.action_id)
            if not action_id or not isinstance(action_id, str):
                fatal("No action_id mapping for monitor action: %r", action.action_id)
            
            innerAction = inner_zmc_client.create_monitor_action(
                str(newmon.id),
                body=MonitorAction(monitor_id=str(newmon.id),
                                   action_id=str(action_id)))
            if not innerAction:
                fatal("Could not create monitor Action: %r", action_id)
                pass
            LOG.debug("Created new monitor Action: %r", innerAction)

#
# Copy Metrics in
#
def importMetrics(inner_dst_client: ZmsDstClient, outer_dst_client:ZmsDstClient,
                  element_id: str, objmap: ObjectMap, name: Optional[str] = None,
                  impotent: bool = False):
    kwargs = dict()
    if name:
        kwargs["metric"] = name
    resp = outer_dst_client.list_metrics(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MetricList):
        fatal("Could not list outer metrics: %r", resp)
    metricList = cast(MetricList, resp.parsed)
    for metric in metricList.metrics:
        #
        # Check if already added.
        #
        resp = inner_dst_client.list_metrics(
            element_id=element_id, metric=metric.name, items_per_page=1000)
        if not resp.is_success or not isinstance(resp.parsed, MetricList):
            fatal("Could not list inner metrics: %r", resp)
        innerMetricList = cast(MetricList, resp.parsed)
        
        if innerMetricList and innerMetricList.metrics and len(innerMetricList.metrics):
            met = innerMetricList.metrics[0]
            LOG.info("Metric already exists: %r, %r", met.name, met.id)
            LOG.debug("Metric: %r", met)

            objmap.add_object(metric, met)
            continue

        LOG.info("Metric does not exist yet: %r", metric.name)
        if impotent:
            continue

        if metric.description:
            description = metric.description + " (" + str(metric.id) + ")"
        else:
            description = metric.id
            pass
        newmet = Metric(name=metric.name,
                        element_id=element_id,
                        description=description,
                        unit=metric.unit,
                        schema=metric.schema)
        LOG.debug("Creating new metric: %r", newmet)

        resp = inner_dst_client.create_metric(body=newmet)
        if not resp.is_success or not isinstance(resp.parsed, Metric):
            fatal("Could not create metric: %r", metric.id)
        newmet = cast(Metric, resp.parsed)
        LOG.debug("Created new metric: %r", newmet)

        objmap.add_object(metric, newmet)
        LOG.info("Created metric: %r", newmet.id)

def main():
    parser = argparse.ArgumentParser(
        prog="loadzmc",
        description="Load objects (radios, antennas, monitors, locations) from one ZMC into another.  Optionally accept a grant from the source ZMC and create it as a Spectrum object (outer to inner).")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Impotent: do not inject events")
    parser.add_argument(
        "--element-id", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) element ID")
    parser.add_argument(
        "--element-token", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) element token")
    parser.add_argument(
        "--zmc-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) ZMC URL")
    parser.add_argument(
        "--parent-zmc-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) ZMC URL")
    parser.add_argument(
        "--dst-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) DST URL")
    parser.add_argument(
        "--parent-dst-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) DST URL")
    parser.add_argument(
        "--parent-element-token", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) element token")
    parser.add_argument(
        "--parent-grant-list", action=DefaultDestEnvAction, type=str, required=False,
        help="Upstream (outer) grant list to transform into Spectrum objects in inner ZMC")
    parser.add_argument(
        "--radio", default=None, type=str,
        help="Filter by radio name; pass '' to skip entirely")
    parser.add_argument(
        "--action", default=None, type=str,
        help="Filter by action name; pass '' to skip entirely")
    parser.add_argument(
        "--monitor", default=None, type=str,
        help="Filter by monitor name; pass '' to skip entirely")
    parser.add_argument(
        "--antenna", default=None, type=str,
        help="Filter by antenna name; pass '' to skip entirely")
    parser.add_argument(
        "--location", default=None, type=str,
        help="Filter by location name; pass '' to skip entirely")
    parser.add_argument(
        "--metric", default=None, type=str,
        help="Filter by metric name; pass '' to skip entirely")

    args = parser.parse_args(sys.argv[1:])

    logging.basicConfig()
    if args.debug:
        LOG.setLevel(logging.DEBUG)
        logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)

    innerZMC   = ZmsZmcClient(args.zmc_http, args.element_token,
                              raise_on_unexpected_status=True)
    outerZMC   = ZmsZmcClient(args.parent_zmc_http, args.parent_element_token,
                              raise_on_unexpected_status=True)
    innerDST   = ZmsDstClient(args.dst_http, args.element_token,
                              raise_on_unexpected_status=True)
    outerDST   = ZmsDstClient(args.parent_dst_http, args.parent_element_token,
                              raise_on_unexpected_status=True)
    obj_map = ObjectMap()

    if args.parent_grant_list:
        importSpectrum(
            innerZMC, outerZMC, args.element_id, args.parent_grant_list,
            obj_map, impotent=args.impotent)
    if args.location != "":
        importLocations(
            innerZMC, outerZMC, args.element_id,
            obj_map, name=args.location, impotent=args.impotent)
    if args.antenna != "":
        importAntennas(
            innerZMC, outerZMC, args.element_id,
            obj_map, name=args.antenna, impotent=args.impotent)
    if args.radio != "":
        importRadios(
            innerZMC, outerZMC, args.element_id,
            obj_map, name=args.radio, impotent=args.impotent)
    if args.action != "":
        importActions(
            innerZMC, outerZMC, args.element_id,
            obj_map, name=args.action, impotent=args.impotent)
    if args.monitor != "":
        importMonitors(
            innerZMC, outerZMC, args.element_id,
            obj_map, name=args.monitor, impotent=args.impotent)
    if args.metric != "":
        importMetrics(
            innerDST, outerDST, args.element_id,
            obj_map, name=args.metric, impotent=args.impotent)

if __name__ == "__main__":
    main()
