import argparse
import csv
import dateutil
import logging
import uuid
from typing import (List, Dict)

from zmsclient.dst.v1.models import (Metric, Value, AnyObject)

from . import BaseDataType

LOG = logging.getLogger(__name__)

class ViaviCellAdvisorCsvDataType(BaseDataType):
    name: str = "viavi-celladvisor-csv"
    metric_defn: Metric = Metric(
        name="viavi.celladvisor-csv.v1",
        element_id=str(uuid.UUID(int=0)),
        description="Viavi CellAdvisor CSV Metrics v1",
        schema=AnyObject.from_dict({
            "$id": "https://gitlab.flux.utah.edu/openzms/zms-dst-schemas/schemas/metrics/viavi/celladvisor-csv/v1/schema.json",
            "type": "object",
            "title": "Viavi CellAdvisor CSV Metrics v1",
            "$schema": "https://json-schema.org/draft/2020-12/schema",
            "properties": {
                "attenuation": {
                    "type": "integer",
                    "description": "Attenuation.",
                    "x-tag": True
                },
                "preamp_1": {
                    "type": "string",
                    "description": "Preamp 1 state.",
                    "enum": ["on", "off"],
                    "x-tag": True
                },
                "preamp_2": {
                    "type": "string",
                    "description": "Preamp 2 state.",
                    "enum": ["on", "off"],
                    "x-tag": True
                },
                "dnc_preamp": {
                    "type": "string",
                    "description": "DNC preamp state.",
                    "enum": ["on", "off"],
                    "x-tag": True
                },
                "carrier": {
                    "type": "integer",
                    "description": "Carrier number.",
                    "x-tag": True
                },
                "center_freq": {
                    "type": "number",
                    "description": "Center frequency in MHz.",
                    "x-unit": "MHz",
                    "x-tag": True
                },
                "bandwidth": {
                    "type": "integer",
                    "description": "Bandwidth in MHz.",
                    "x-unit": "MHz",
                    "x-tag": True
                },
                "ssb_scs": {
                    "type": "number",
                    "description": "SSB subcarrier spacing in kHz.",
                    "x-unit": "kHz",
                    "x-tag": True
                },
                "ssb_center_freq": {
                    "type": "number",
                    "description": "SSB center frequency in MHz.",
                    "x-unit": "MHz",
                    "x-tag": True
                },
                "l": {
                    "type": "integer",
                    "description": "SSB L value.",
                    "x-tag": True
                },
                "pci": {
                    "type": "integer",
                    "description": "Physical cell ID.",
                    "x-tag": True
                },
                "ssb_index": {
                    "type": "integer",
                    "description": "SSB index.",
                    "x-tag": True
                },
                "channel_pwr": {
                    "type": "number",
                    "description": "Channel power in dBm.",
                    "x-unit": "dBm",
                    "x-group": "power"
                },
                "s_ss_rsrp": {
                    "type": "number",
                    "description": "SSB RSRP in dBm.",
                    "x-unit": "dBm",
                    "x-group": "power"
                },
                "p_ss_rsrp": {
                    "type": "number",
                    "description": "PSS RSRP in dBm.",
                    "x-unit": "dBm",
                    "x-group": "power"
                },
                "pbch_evm": {
                    "type": "number",
                    "description": "PBCH EVM in percent.",
                    "x-unit": "percent",
                    "x-group": "relative"
                },
                "time_error": {
                    "type": "number",
                    "description": "Time error in microseconds.",
                    "x-unit": "us",
                    "x-group": "time-error"
                },
                "frequency_error": {
                    "type": "number",
                    "description": "Frequency error in Hz.",
                    "x-unit": "Hz",
                    "x-group": "freq-error"
                },
                "s_ss_rssi": {
                    "type": "number",
                    "description": "SSB RSSI in dBm.",
                    "x-unit": "dBm",
                    "x-group": "power"
                },
                "pbch_dm_rs_rsrp": {
                    "type": "number",
                    "description": "PBCH DM-RS RSRP in dBm.",
                    "x-unit": "dBm",
                    "x-group": "power"
                },
                "pbch_dm_rs_evm": {
                    "type": "number",
                    "description": "PBCH DM-RS EVM in percent.",
                    "x-unit": "percent",
                    "x-group": "relative"
                },
                "mcc": {
                    "type": "integer",
                    "description": "Mobile country code.",
                    "x-tag": True
                },
                "mnc": {
                    "type": "integer",
                    "description": "Mobile network code.",
                    "x-tag": True
                },
                "latitude": {
                    "type": "number",
                    "description": "Latitude.",
                    "x-unit": "degrees",
                    "x-group": "geospatial"
                },
                "longitude": {
                    "type": "number",
                    "description": "Longitude.",
                    "x-unit": "degrees",
                    "x-group": "geospatial"
                }
            }
        })
    )
    column_map: Dict[str,int] = dict(
        latitude=2,
        longitude=3,
        attenuation=4,
        preamp_1=5,
        preamp_2=6,
        dnc_preamp=7,
        carrier=8,
        center_freq=9,
        bandwidth=10,
        ssb_scs=11,
        ssb_center_freq=12,
        l=13,
        pci=14,
        ssb_index=15,
        channel_pwr=16,
        s_ss_rsrp=17,
        p_ss_rsrp=18,
        pbch_evm=19,
        time_error=20,
        frequency_error=21,
        s_ss_rssi=22,
        pbch_dm_rs_rsrp=23,
        pbch_dm_rs_evm=24,
        mcc=25,
        mnc=26,
        nci=27,
        country=28,
        operator=29,
    )

    @classmethod
    def add_args(cls, parser: argparse.ArgumentParser):
        """Add any additional arguments needed for this data type."""
        pass

    @classmethod
    def get_metric_definitions(cls) -> List[Metric]:
        """Define the metrics that will be collected."""
        return [cls.metric_defn]

    # Parse the given file.
    #
    # Example format:
    #
    # Date,Time,Latitude,Longitude,Attenuation,Preamp 1,Preamp 2,DNC Preamp,Carrier No,Center Frequency (MHz),Bandwidth,SSB SCS,SSB Center Frequency (MHZ),L,PCI,SSB Index,Channel PWR (dBm),S-SS RSRP (dBm),P-SS RSRP (dBm),PBCH EVM (%),Time Error (us),Frequency Error (Hz),S-SS RSSI (dBm),PBCH DM-RS RSRP (dBm),PBCH DM-RS EVM (%),MCC,MNC,NCI,Country,Operator
    # "Friday, September 26, 2025",11:07:00.684 AM MDT,---,---,0,On,On,On,2,3365.76,20,0.03,3365.76,8,4,0,-84.3244,-118.78,-117.69,64.0101,0.545247,-97.4253,-97.7418,-115.893,82.9125,---,---,---,---,---
    # "Friday, September 26, 2025",11:07:01.213 AM MDT,---,---,0,On,On,On,1,3365.76,20,0.03,3365.76,8,---,---,---,---,---,---,---,---,---,---,---,999,99,---,-999,-999
    #
    def load_data(self):
        """Parse the given file."""
        with open(self.args.data_file, "r") as f:
            csvreader = csv.reader(f)
            i = 0
            for row in csvreader:
                i += 1
                if len(row) < 30:
                    LOG.warning("Skipping short row: %s", row)
                    continue
                if row[0].startswith("Date"):
                    # Header row
                    continue
                try:
                    #dt = datetime.datetime.strptime(
                    #    row[0] + " " + row[1], "%A, %B %d, %Y %I:%M:%S.%f %p %Z")
                    #if dt.tzinfo is None:
                    #    raise ValueError(f"Cannot parse row {i}: date/time is missing timezone info: {row}")
                    dt = dateutil.parser.parse(row[0] + " " + row[1])
                except Exception as e:
                    raise ValueError(f"Cannot parse row {i} with bad date/time ({e}): {row}")
                if row[self.column_map["channel_pwr"]] == "---":
                    # No signal, skip
                    continue
                vd = dict(
                    channel_pwr=float(row[self.column_map["channel_pwr"]]),
                    s_ss_rsrp=float(row[self.column_map["s_ss_rsrp"]]),
                    p_ss_rsrp=float(row[self.column_map["p_ss_rsrp"]]),
                    pbch_evm=float(row[self.column_map["pbch_evm"]]),
                    time_error=float(row[self.column_map["time_error"]]),
                    frequency_error=float(row[self.column_map["frequency_error"]]),
                    s_ss_rssi=float(row[self.column_map["s_ss_rssi"]]),
                    pbch_dm_rs_rsrp=float(row[self.column_map["pbch_dm_rs_rsrp"]]),
                    pbch_dm_rs_evm=float(row[self.column_map["pbch_dm_rs_evm"]])
                )
                td = dict(
                    attenuation=float(row[self.column_map["attenuation"]]),
                    preamp_1=row[self.column_map["preamp_1"]].lower(),
                    preamp_2=row[self.column_map["preamp_2"]].lower(),
                    dnc_preamp=row[self.column_map["dnc_preamp"]].lower(),
                    carrier=int(row[self.column_map["carrier"]]),
                    center_freq=float(row[self.column_map["center_freq"]]),
                    bandwidth=int(row[self.column_map["bandwidth"]]),
                    ssb_scs=float(row[self.column_map["ssb_scs"]]),
                    ssb_center_freq=float(row[self.column_map["ssb_center_freq"]]),
                    l=int(row[self.column_map["l"]]),
                    pci=int(row[self.column_map["pci"]]),
                    ssb_index=int(row[self.column_map["ssb_index"]]),
                )
                try:
                    vd["latitude"] = float(row[self.column_map["latitude"]])
                    vd["longitude"] = float(row[self.column_map["longitude"]])
                except Exception as e:
                    pass
                try:
                    td["mcc"] = int(row[self.column_map["mcc"]])
                except Exception as e:
                    pass
                try:
                    td["mnc"] = int(row[self.column_map["mnc"]])
                except Exception as e:
                    pass
                v = Value(
                    monitor_id=self.mh.monitor_id,
                    metric_id=str(uuid.UUID(int=0)),  # filled in later
                    created_at=dt,
                    fields=AnyObject.from_dict(vd),
                    tags=AnyObject.from_dict(td)
                )
                self.values.append(v)
                LOG.debug("Parsed row %d: %s", i, v)

    def prepare_values(self):
        """Prepare any values before uploading."""
        if len(self.values) == 0:
            LOG.warning("No values to prepare")
            return
        for v in self.values:
            m = self.mh.get_metric(self.metric_defn.name)
            if not m:
                raise ValueError(f"Metric {self.metric_defn.name} not found in MetricsHelper")
            v.metric_id = m.id
            m = self.mh.get_series(self.metric_defn.name)
            if not m:
                raise ValueError(f"Series for metric {self.metric_defn.name} not found in MetricsHelper")
            v.series_id = m.id