import argparse
import gpxpy
import dateutil
import logging
import uuid
from typing import List

from zmsclient.dst.v1.models import (Metric, Value, AnyObject)

from . import BaseDataType

LOG = logging.getLogger(__name__)

class GpxDataType(BaseDataType):
    # {"class":"SKY","device":"/dev/ttyUSB1","xdop":0.86,"ydop":0.89,"vdop":0.80,"tdop":1.14,"hdop":1.00,"gdop":2.34,"pdop":1.30,"nSat":9,"uSat":6,"satellites":[{"PRN":2,"el":26.0,"az":313.0,"ss":28.0,"used":true,"gnssid":0,"svid":2},{"PRN":8,"el":27.0,"az":272.0,"ss":22.0,"used":false,"gnssid":0,"svid":8},{"PRN":10,"el":67.0,"az":25.0,"ss":27.0,"used":true,"gnssid":0,"svid":10},{"PRN":18,"el":16.0,"az":140.0,"ss":23.0,"used":true,"gnssid":0,"svid":18},{"PRN":23,"el":33.0,"az":78.0,"ss":23.0,"used":true,"gnssid":0,"svid":23},{"PRN":24,"el":21.0,"az":49.0,"ss":27.0,"used":true,"gnssid":0,"svid":24},{"PRN":27,"el":27.0,"az":236.0,"ss":27.0,"used":false,"gnssid":0,"svid":27},{"PRN":28,"el":16.0,"az":168.0,"ss":22.0,"used":false,"gnssid":0,"svid":28},{"PRN":32,"el":76.0,"az":224.0,"ss":17.0,"used":true,"gnssid":0,"svid":32}]}
    # {"class":"TPV","device":"/dev/ttyUSB1","mode":3,"time":"2025-10-06T15:25:56.000Z","ept":0.005,"lat":40.769662317,"lon":-111.832074533,"altHAE":1517.7000,"altMSL":1530.7000,"alt":1530.7000,"epx":12.835,"epy":13.358,"epv":18.400,"track":148.7000,"magtrack":137.3000,"magvar":11.5,"speed":6.431,"climb":0.900,"eps":26.72,"epc":36.80,"geoidSep":-13.000,"eph":19.000,"sep":24.700}
    #
    # Example GPX file format:
    # <?xml version="1.0" encoding="UTF-8"?>
    # <gpx xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://www.topografix.com/GPX/1/1" xsi:schemaLocation="http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd" version="1.1" creator="Open GPX Tracker for iOS">
    #    <trk>
    #            <trkseg>
    #                    <trkpt lat="40.76934055543372" lon="-111.84597388311491">
    #                            <ele>1455.7677420917898</ele>
    #                            <time>2025-09-26T16:44:54Z</time>
    #                    </trkpt>


    name: str = "gpx"
    metric_defn: Metric = Metric(
        name="zms.location.gps.v1",
        element_id=str(uuid.UUID(int=0)),
        description="OpenZMS GPS Location Data v1",
        schema=AnyObject.from_dict({
            "$id": "https://gitlab.flux.utah.edu/openzms/zms-dst-schemas/schemas/metrics/zms/location/gps/v1/schema.json",
            "type": "object",
            "title": "OpenZMS GPS Location Data v1",
            "$schema": "https://json-schema.org/draft/2020-12/schema",
            "properties": {
                "systemid": {
                    "type": "integer",
                    "description": "NMEA 4.11 GPS system identifier (1=GPS, 2=GLONASS, 3=Galileo, 4=BDS).",
                    "x-tag": True
                },
                "signalid": {
                    "type": "integer",
                    "description": "NMEA 4.11 Signal identifier.",
                    "x-tag": True
                },
                "device": {
                    "type": "string",
                    "description": "Measuring device.",
                    "x-tag": True
                },
                "mode": {
                    "type": "integer",
                    "description": "Mode (0=unknown, 1=no fix, 2=2D fix, 3=3D fix).",
                    "x-group": "system",
                    "x-tag": True
                },
                "lat": {
                    "type": "number",
                    "description": "Latitude.",
                    "x-unit": "degrees",
                    "x-group": "coordinates"
                },
                "lon": {
                    "type": "number",
                    "description": "Longitude.",
                    "x-unit": "degrees",
                    "x-group": "coordinates"
                },
                "alt": {
                    "type": "number",
                    "description": "Altitude.",
                    "x-unit": "meters",
                    "x-group": "elevation"
                },
                "altmsl": {
                    "type": "number",
                    "description": "Altitude (mean sea level).",
                    "x-unit": "meters",
                    "x-group": "elevation"
                },
                "althae": {
                    "type": "number",
                    "description": "Altitude (height above ellipsoid).",
                    "x-unit": "meters",
                    "x-group": "elevation"
                },
                "track": {
                    "type": "number",
                    "description": "Track.",
                    "x-unit": "degrees",
                    "x-group": "heading"
                },
                "magtrack": {
                    "type": "number",
                    "description": "Magnetic track.",
                    "x-unit": "degrees",
                    "x-group": "heading"
                },
                "magvar": {
                    "type": "number",
                    "description": "Magnetic variation.",
                    "x-unit": "degrees",
                    "x-group": "heading"
                },
                "speed": {
                    "type": "number",
                    "description": "Speed.",
                    "x-unit": "meters/second",
                    "x-group": "rate"
                },
                "climb": {
                    "type": "number",
                    "description": "Climb rate.",
                    "x-unit": "meters/second",
                    "x-group": "rate"
                },
                "geoidsep": {
                    "type": "number",
                    "description": "Geoid separation.",
                    "x-unit": "meters",
                    "x-group": "elevation"
                },
                "sep": {
                    "type": "number",
                    "description": "Estimated vertical error.",
                    "x-unit": "meters",
                    "x-group": "error"
                },
                "eph": {
                    "type": "number",
                    "description": "Estimated horizontal error.",
                    "x-unit": "meters",
                    "x-group": "error"
                },
                "epv": {
                    "type": "number",
                    "description": "Estimated vertical error.",
                    "x-unit": "meters",
                    "x-group": "error"
                },
                "epx": {
                    "type": "number",
                    "description": "Estimated longitude error.",
                    "x-unit": "meters",
                    "x-group": "error"
                },
                "epy": {
                    "type": "number",
                    "description": "Estimated latitude error.",
                    "x-unit": "meters",
                    "x-group": "error"
                },
                "ept": {
                    "type": "number",
                    "description": "Estimated timestamp error.",
                    "x-unit": "seconds",
                    "x-group": "error"
                },
                "eps": {
                    "type": "number",
                    "description": "Estimated speed error.",
                    "x-unit": "meters/second",
                    "x-group": "error"
                },
                "epc": {
                    "type": "number",
                    "description": "Estimated climb error.",
                    "x-unit": "meters/second",
                    "x-group": "error"
                },
                "dop": {
                    "type": "number",
                    "description": "Dilution of precision.",
                    "x-group": "error"
                },
                "xdop": {
                    "type": "number",
                    "description": "Horizontal dilution of precision.",
                    "x-group": "error"
                },
                "ydop": {
                    "type": "number",
                    "description": "Vertical dilution of precision.",
                    "x-group": "error"
                },
                "hdop": {
                    "type": "number",
                    "description": "Horizontal dilution of precision.",
                    "x-group": "error"
                },
                "vdop": {
                    "type": "number",
                    "description": "Vertical dilution of precision.",
                    "x-group": "error"
                },
                "gdop": {
                    "type": "number",
                    "description": "Geometric dilution of precision.",
                    "x-group": "error"
                },
                "pdop": {
                    "type": "number",
                    "description": "Position dilution of precision.",
                    "x-group": "error"
                },
                "tdop": {
                    "type": "number",
                    "description": "Time dilution of precision.",
                    "x-group": "error"
                },
                "nsat": {
                    "type": "integer",
                    "description": "Number of visible satellites.",
                    "x-group": "info"
                },
                "usat": {
                    "type": "integer",
                    "description": "Number of satellites used in solution.",
                    "x-group": "info"
                }
            },
            "required": [
                "lat",
                "lon"
            ]
        })
    )

    @classmethod
    def add_args(cls, parser: argparse.ArgumentParser):
        """Add any additional arguments needed for this data type."""
        parser.add_argument(
            "--gpx-all-segments", default=False, action="store_true", help="Include points from all tracks and segments (default: first track/segment only).")

    @classmethod
    def get_metric_definitions(cls) -> List[Metric]:
        """Define the metrics that will be collected."""
        return [cls.metric_defn]

    # Parse the given file.
    #
    # Example format:
    #
    # Date,Time,Latitude,Longitude,Attenuation,Preamp 1,Preamp 2,DNC Preamp,Carrier No,Center Frequency (MHz),Bandwidth,SSB SCS,SSB Center Frequency (MHZ),L,PCI,SSB Index,Channel PWR (dBm),S-SS RSRP (dBm),P-SS RSRP (dBm),PBCH EVM (%),Time Error (us),Frequency Error (Hz),S-SS RSSI (dBm),PBCH DM-RS RSRP (dBm),PBCH DM-RS EVM (%),MCC,MNC,NCI,Country,Operator
    # "Friday, September 26, 2025",11:07:00.684 AM MDT,---,---,0,On,On,On,2,3365.76,20,0.03,3365.76,8,4,0,-84.3244,-118.78,-117.69,64.0101,0.545247,-97.4253,-97.7418,-115.893,82.9125,---,---,---,---,---
    # "Friday, September 26, 2025",11:07:01.213 AM MDT,---,---,0,On,On,On,1,3365.76,20,0.03,3365.76,8,---,---,---,---,---,---,---,---,---,---,---,999,99,---,-999,-999
    #
    def load_data(self):
        """Parse the given file."""
        with open(self.args.data_file, "r") as f:
            g = gpxpy.parse(f)
            ti = si = 0
            for trk in g.tracks:
                ti += 1
                for seg in trk.segments:
                    if si > 0 and not self.args.gpx_all_segments and len(seg.points) > 0:
                        LOG.warning("Skipping additional GPX track segment points!")
                        return
                    si += 1
                    LOG.info("Processing GPX track %d segment %d with %d points", ti, si, len(seg.points))
                    for pt in seg.points:
                        fd = dict(
                            lat=pt.latitude,
                            lon=pt.longitude
                        )
                        if pt.elevation is not None:
                            fd["alt"] = pt.elevation
                        if pt.speed is not None:
                            fd["speed"] = pt.speed
                        if pt.horizontal_dilution is not None:
                            fd["hdop"] = pt.horizontal_dilution
                        if pt.vertical_dilution is not None:
                            fd["vdop"] = pt.vertical_dilution
                        if pt.position_dilution is not None:
                            fd["pdop"] = pt.position_dilution
                        v = Value(
                            monitor_id=self.mh.monitor_id,
                            metric_id=str(uuid.UUID(int=0)),  # filled in later
                            fields=AnyObject.from_dict(fd)
                        )
                        if pt.time is not None:
                            v.created_at = pt.time
                        self.values.append(v)
                        LOG.debug("Parsed GPX point: %s", v)
            i = 0

    def prepare_values(self):
        """Prepare any values before uploading."""
        if len(self.values) == 0:
            LOG.warning("No values to prepare")
            return
        for v in self.values:
            m = self.mh.get_metric(self.metric_defn.name)
            if not m:
                raise ValueError(f"Metric {self.metric_defn.name} not found in MetricsHelper")
            v.metric_id = m.id
            m = self.mh.get_series(self.metric_defn.name)
            if not m:
                raise ValueError(f"Series for metric {self.metric_defn.name} not found in MetricsHelper")
            v.series_id = m.id