#!/bin/python

import time
import uuid
import os
import os.path
import sys
import signal
import argparse
import logging
import asyncio
import shlex
import subprocess
import datetime
import threading
import socket
import re
import enum
from typing import Optional, List, Any, cast

from zmsclient.zmc.client import ZmsZmcClient
from zmsclient.zmc.v1.models import (
    Subscription, EventFilter, Error,
    UpdateGrantOpStatus, GrantOpStatus,
    Grant, GrantConstraint, Constraint, GrantRadioPort,
    Radio, RadioList)    
from zmsclient.common.subscription import ZmsSubscriptionCallback
from zmsclient.common.manifest import parse_manifest
from zmsclient.common import DefaultDestEnvAction

HOSTNAME = socket.gethostname()
LOG = logging.getLogger(__name__)

# Not sure where to pick these up
EVENT_TYPE_REPLACED    =     1
EVENT_TYPE_CREATED     =     2
EVENT_SOURCETYPE_ZMC   =     2
EVENT_CODE_GRANT       =     2006

# Only when a daemon
LOGFILE = "/local/logs/tx-wrapper.log"

# Command to run.
TXCMDTEST = "/bin/sleep 1000"
if os.path.exists("/usr/lib/uhd/examples/tx_waveforms"):
    TXWAVEFORMS = "/usr/lib/uhd/examples/tx_waveforms"
elif os.path.exists("/usr/libexec/uhd/examples/tx_waveforms"):
    TXWAVEFORMS = "/usr/libexec/uhd/examples/tx_waveforms"
else:
    TXWAVEFORMS = "tx_waveforms"
TXCMD = TXWAVEFORMS + " --ant TX/RX --channel 0 " +\
    "--rate %r --gain 50 --freq %r "

class FreqRangeDesc:
    def __init__(self, min, max, eirp, width=None):
        self.min = min
        self.max = max
        self.eirp = eirp
        self.width = width

    @classmethod
    def from_string(cls, s):
        sa = s.split(",")
        if len(sa) != 3 and len(sa) != 4:
            raise ValueError("invalid frequency range descriptor")

        width = None
        if len(sa) == 4:
            width = float(sa[3])
            pass
        
        return FreqRangeDesc(*[float(x) for x in sa[0:3]], width=width)

    def __repr__(self):
        return f"FreqRangeDesc(min={self.min},max={self.max},eirp={self.eirp},width={self.width})"

class ViolationFlags(enum.IntFlag):
    NONE         = 0
    STOP         = 1 << 0
    START        = 1 << 1
    HEARTBEAT    = 1 << 2
    REPLACE      = 1 << 3
    ALL          = STOP | START | HEARTBEAT | REPLACE

    DEFAULT = STOP | START

    @classmethod
    def from_string(cls, s: str) -> 'ViolationFlags':
        # Default to STOP|START.  The old txwrapper effectively only 
        flags = cls.STOP
        for part in s.split(","):
            part = part.strip().lower()
            if part == "stop":
                flags |= cls.STOP
            elif part == "start":
                flags |= cls.START
            elif part == "replace":
                flags |= cls.REPLACE
            elif part == "heartbeat":
                flags |= cls.HEARTBEAT
            elif part == "all":
                flags = cls.ALL
            else:
                raise ValueError(f"invalid violation flag: {part}")
            pass
        return flags

#
# Wrap the ZMS grant.
#
class GrantWrapper():
    def __init__(self, zmcclient: ZmsZmcClient, args: argparse.Namespace,
                 lock: threading.RLock, condition: threading.Condition,
                 grant: Grant, should_delete: bool = False):
        self.lock = lock
        self.condition = condition
        self._grant: Grant = grant
        self.zmcclient = zmcclient
        self.args = args
        self.should_delete = should_delete
        pass

    @property
    def grant(self) -> Grant:
        return self._grant

    @grant.setter
    def grant(self, grant: Grant):
        self._grant = grant
        pass

    @property
    def constraints(self):
        return self.grant.constraints

    @property
    def status(self):
        return self.grant.status

    @property
    def id(self):
        return self.grant.id
    
    @property
    def replacement(self):
        return self.grant.replacement
    
    @property
    def expires_at(self):
        return self.grant.expires_at

    @property
    def status_ack_by(self):
        return self.grant.status_ack_by

    def revoked(self):
        return self.grant.status == "revoked"
    
    def deleted(self):
        return self.grant.deleted_at
    
    def terminal(self):
        return self.grant.status in ["revoked", "deleted", "denied"]

    #
    # If we have a radio_id but no radio_port_id, we need to find a radio_port_id
    # for the grant.  We special-case the POWDER RDZ-in-RDZ case, if we are
    # talking to an inner ZMC; we have to map the outer radio ID to the inner
    # radio ID.  Otherwise, we just load the radio from the target ZMC.
    #
    @classmethod
    def map_radio_to_port(cls, zmcclient: ZmsZmcClient, args: argparse.Namespace) -> str:
        if args.is_powder and zmcclient._base_url.find("rdz.powderwireless.net") < 0:
            LOG.info("Mapping outer radio ID %r to inner ID (%r)", args.radio_id, zmcclient._base_url)
        
            resp = zmcclient.list_radios(radio=args.radio_id, x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, RadioList):
                raise Exception("Could not map outer radio to inner radio: error: %r", resp)
            inner = resp.parsed
            if not inner or not inner.radios or len(inner.radios) == 0:
                raise Exception("Could not map outer radio to inner radio")

            radio = inner.radios[0]
            LOG.info("Mapped to inner radio: %r", radio.id)
            args.radio_id = radio.id
        else:
            LOG.info("Loading radio ID %r", args.radio_id)
            resp = zmcclient.get_radio(str(args.radio_id), x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, Radio):
                raise Exception("Could not load radio ID %r: error: %r" %
                                (args.radio_id, resp))
            radio = resp.parsed
        if not radio.ports or len(radio.ports) == 0:
            raise Exception("Radio has no ports: %r" % (radio.id,))
        for port in radio.ports:
            if port.tx:
                return cast(str, port.id)
        raise Exception("Radio has no TX ports: %r" % (radio.id,))

    @classmethod
    def create(cls, zmcclient: ZmsZmcClient, args: argparse.Namespace,
                    lock: threading.RLock, condition: threading.Condition,
                    extid: Optional[str] = None, **kwargs) -> 'GrantWrapper':
        starts = datetime.datetime.now(datetime.timezone.utc)
        expires = starts + datetime.timedelta(hours=2)
        constraint = Constraint(
            min_freq = int(args.band.min),
            max_freq = int(args.band.max),
            max_eirp = args.band.eirp)
        if not args.radio_port_id and not args.radio_id:
            constraint.exclusive = True
        elif not args.radio_port_id and args.radio_id:
            args.radio_port_id = cls.map_radio_to_port(zmcclient, args)
        if args.band.width:
            constraint.bandwidth = int(args.band.width)
        grantreq = Grant(
            element_id = args.element_id,
            name = HOSTNAME,
            description = "Powder Experiment " + HOSTNAME,
            starts_at = starts,
            expires_at = expires,
            constraints = [ GrantConstraint(constraint=constraint) ],
            allow_skip_acks = not args.heartbeat,
        )
        if args.radio_port_id:
            grantreq.radio_ports = [GrantRadioPort(radio_port_id=args.radio_port_id)]
        if extid:
            grantreq.ext_id = extid
            pass

        try:
            resp = zmcclient.create_grant(body=grantreq)
        except Exception as ex:
            LOG.exception(ex)
            LOG.error("Failed to create_grant at the ZMC")
            sys.exit(-1)
            pass
        if not resp.is_success or not isinstance(resp.parsed, Grant):
            LOG.error("Create Grant Error: %r", resp)
            sys.exit(-1)
        grant = resp.parsed
        LOG.debug("Created grant: %r", grant)

        #
        # There can be a delay, so lets poll for a bit until it goes
        # active or is denied/deleted.
        #
        LOG.info("Waiting for grant to be approved")
        for x in range(30):
            time.sleep(2)
            try:
                resp = zmcclient.get_grant(str(grant.id), x_api_elaborate="True")
            except Exception as ex:
                LOG.exception(ex)
                LOG.error("Failed to get_grant at the ZMC, waiting ...")
                continue
            if not resp.is_success or not isinstance(resp.parsed, Grant):
                LOG.error("Get Grant Error: %r", resp)
                continue
            grant = resp.parsed
            LOG.debug("Polled grant: %r", grant)
            if grant.status in ["active", "denied", "deleted", "pending"]:
                break
            LOG.debug("Waiting for grant %r to be approved ...", grant.id)
            pass
        if grant.status != "active":
            LOG.error("Failed to get initial approved grant %r", grant.id)
            sys.exit(-1)
            pass
        
        return cls(zmcclient, args, lock, condition, grant, should_delete=True)
    
    @classmethod
    def load(cls, zmcclient: ZmsZmcClient, args: argparse.Namespace,
             lock: threading.RLock, condition: threading.Condition,
             grant_id: str) -> 'GrantWrapper':
        #
        # Initial load of the grant, chasing the replacement chain to the
        # current (active) grant.
        #
        LOG.info("Loading initial grant %r", grant_id)
        resp = zmcclient.get_grant(grant_id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Grant):
            LOG.error("Could not get grant: %r " % (grant_id,))
            sys.exit(-1)
        grant = resp.parsed

        while grant.replacement:
            grant_id = grant.replacement.new_grant_id
            LOG.info("Loading next grant %r", grant_id)
            resp = zmcclient.get_grant(grant_id, x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, Grant):
                LOG.error("Could not get grant: %r " % (grant_id,))
                sys.exit(-1)
            grant = resp.parsed

        return cls(zmcclient, args, lock, condition, grant)

    def deleteGrant(self):
        if not self.should_delete:
            return
        
        with self.lock:
            if self.deleted():
                return
            LOG.info("Deleting grant %r", self.grant.id)
            resp = self.zmcclient.delete_grant(
                grant_id=str(self.grant.id))
            if not resp.is_success:
                LOG.error("Could not delete grant: %r", resp)
                return
            # XXX: forge this, because the API does not return the updated,
            # deleted grant object.
            self._grant.deleted_at = datetime.datetime.now(datetime.timezone.utc)
            pass
        pass
        
    def sendOpStatus(self):
        with self.lock:
            if self.terminal():
                return
            
            LOG.info("sendOpStatus: %r", self.grant.status)
            opstatus = self.grant.status
            if opstatus == "pending" or opstatus == "revoked" or opstatus == "deleted":
                opstatus = "paused"
                pass
            
            try:
                resp = self.zmcclient.update_grant_op_status(
                    grant_id=str(self.grant.id),
                    body=UpdateGrantOpStatus(GrantOpStatus(opstatus)))
            except Exception as ex:
                LOG.exception(ex)
                LOG.info("sendOpStatus failed")
                return
            
            if not resp.is_success or not isinstance(resp.parsed, Grant):
                LOG.error("Bad response to opstatus update: %r", resp)
                return
            ackby = resp.parsed.status_ack_by
            if ackby:
                LOG.info("sendOpStatus: next ackby: %s",
                         ackby.astimezone().strftime("%Y-%m-%d %H:%M:%S"))
                self.grant.status_ack_by = ackby
                pass
            pass
        pass

    def updateStatus(self, newstatus, force=False):
        with self.lock:
            oldstatus = self.grant.status
            self.grant.status = newstatus
            LOG.info("updateStatus: %r %r", oldstatus, newstatus)
            if force or oldstatus != newstatus:
                self.sendOpStatus()
                pass
            pass
        pass

    def replace(self):
        grantreq = Grant.from_dict(self.grant.to_dict())
        if not grantreq.constraints or len(grantreq.constraints) == 0 \
            or not grantreq.constraints[0].constraint:
            LOG.error(" No constraints in grant, cannot replace")
            return
        grantreq.constraints[0].constraint.min_freq = int(self.args.band.min)
        grantreq.constraints[0].constraint.max_freq = int(self.args.band.max)

        try:
            resp = self.zmcclient.replace_grant(
                grant_id=str(self.grant.id), body=grantreq)
        except Exception as ex:
            LOG.exception(ex)
            LOG.info("Replace grant failed")
            return

        if not resp.parsed or not isinstance(resp.parsed, Grant):
            LOG.error("Bad response to grant replace: %r", resp)
            return
        LOG.info("Replace grant: %r", resp.parsed)
        pass
    pass

#
# Optional heartbeat protocol handling with the ZMS
#
class HeartBeat():
    def __init__(self, grant: GrantWrapper):
        super(HeartBeat, self).__init__()
        self.grant = grant
        self.done = False
        pass
    
    async def start(self):
        LOG.info("Heartbeat starting")
        lastackby = 0

        self.grant.sendOpStatus()       
        while not self.done:
            # We can just slowly spin when nothing to do.
            if self.grant.revoked():
                await asyncio.sleep(5)
                lastackby = 0
                continue
            
            #
            # Count down till when the next heartbeat needs to go.
            #
            ackby    = self.grant.status_ack_by
            now      = datetime.datetime.now(datetime.timezone.utc)
            if not ackby:
                LOG.info("Heartbeat: no status_ack_by, sleeping 10 seconds")
                await asyncio.sleep(10)
                continue
            elif ackby == lastackby:
                # Something went wrong last update. Keep trying
                seconds = 10
            else:
                lastackby = ackby
                duration  = ackby - now
                seconds   = duration.total_seconds()
                pass

            if seconds >= 50:
                seconds = seconds - 30
                pass

            LOG.info("Heartbeat sleeping for %d seconds", seconds)
            while seconds > 0:
                await asyncio.sleep(2)
                if self.done:
                    break
                seconds -= 2
                pass
            LOG.info("Heartbeat done sleeping")
            self.grant.sendOpStatus()
            if self.done:
                break;
            pass
        LOG.info("Heartbeat is exiting")
        pass

    def stop(self):
        LOG.info("HeartBeat stopping")
        self.done = True
        pass

    pass

#
# Subclass ZmsSubscriptionCallback, to handle grant events.
#
class ZMCSubscriptionCallback(ZmsSubscriptionCallback):
    def __init__(self, zmcclient: ZmsZmcClient, element_id: str, 
                 args: argparse.Namespace, wrapper: GrantWrapper,
                 **kwargs):
        super(ZMCSubscriptionCallback, self).__init__(zmcclient, **kwargs)
        self.ZMC = zmcclient
        self.element_id = element_id
        self.args = args
        self.impotent = args.impotent
        self.wrapper = wrapper
        self.runstate = "stopped"
        self.child = None
        self.grant_id = wrapper.id
        self.vflags = ViolationFlags.NONE

    async def start(self):
        if self.wrapper.status == "active":
            await self.startTransmitter()
            if self.runstate == "running":
                self.wrapper.updateStatus(self.wrapper.status, force=True)
                pass
            pass
        try:
            await self.run_callbacks()
        except Exception as ex:
            LOG.exception(ex)
            await self.stopTransmitter()
            pass
        pass

    async def stop(self):
        if self.child:
            await self.stopTransmitter()
            pass
        self.wrapper.deleteGrant()
        pass

    async def on_event(self, ws, evt, message):
        if evt.header.source_type != EVENT_SOURCETYPE_ZMC:
            LOG.error("on_event: unexpected source type: %r (%r)",
                      evt.header.source_type, message)
            return

        # Since we are subscribed to all ZMC events for this element,
        # there will be chatter we do not care about.
        if (evt.header.code != EVENT_CODE_GRANT):
            return

        try:
            await self.handleEvent(evt.object_)
        except Exception as ex:
            LOG.exception(ex)
            pass
        pass

    async def startTransmitter(self):
        LOG.info("startTransmitter")
        if not self.wrapper.constraints or len(self.wrapper.constraints) == 0 \
            or not self.wrapper.constraints[0].constraint:
            LOG.error(" No constraints in grant, cannot start transmitter")
            return
        min_freq = self.wrapper.constraints[0].constraint.min_freq
        max_freq = self.wrapper.constraints[0].constraint.max_freq

        if self.args.impotent:
            command = TXCMDTEST
        elif self.args.txcommand:
            # For external script
            os.environ["TXMIN_FREQ"] = str(min_freq)
            os.environ["TXMAX_FREQ"] = str(max_freq)
            command = self.args.txcommand + " start"
        else:
            freq = min_freq + ((max_freq - min_freq) / 2)
            rate = (max_freq - min_freq) / 4
            command = TXCMD % (rate, freq)
            pass
        LOG.info(" %r", command)
        
        if self.child:
            LOG.info(" Stopping child that should not be running")
            self.child.terminate()
            await self.child.wait()
            self.child = None
            pass
        fp = open("/var/tmp/tx.log", "w")
        self.child = await asyncio.create_subprocess_exec(
            *(shlex.split(command)), stdout=fp, stderr=subprocess.STDOUT)
        if self.child:
            # Give the process time to start.
            try:
                await asyncio.wait_for(self.child.wait(), timeout=3)
            except asyncio.TimeoutError:
                pass
            
            if self.args.txcommand and self.child.returncode == 0:
                LOG.debug(" txcommand succeeded")
            elif self.child.returncode != None:
                LOG.error("Could not start TX process")
                self.child = None
                return
            else:
                LOG.debug(" TX process running")
                pass
            pass
        self.runstate = "running"
        pass

    async def stopTransmitter(self):
        LOG.info("stopTransmitter")
        if not self.child:
            LOG.error(" Child is not running")
            return

        if self.args.txcommand:
            process = await asyncio.create_subprocess_shell(
                self.args.txcommand + " stop", stderr=asyncio.subprocess.STDOUT)
            try:
                await asyncio.wait_for(process.wait(), timeout=5)
            except subprocess.TimeoutExpired:
                LOG.error("External transmitter stop command failed")
                raise asyncio.CancelledError

            if process.returncode:
                LOG.error(process.stdout)
                LOG.error("External script failed to stop transmitter. Exiting")
                raise asyncio.CancelledError
            LOG.info(process.stdout)
        else:
            try:
                self.child.terminate()
            except ProcessLookupError as ex:
                LOG.debug("TX process already stopped: %r", ex)
            except Exception as ex:
                LOG.error("Failed to terminate tx process: %r", ex)
                raise asyncio.CancelledError
            pass

        try:
            await asyncio.wait_for(self.child.wait(), timeout=5)
        except subprocess.TimeoutExpired:
            LOG.error("Failed to stop transmitter. Exiting")
            raise asyncio.CancelledError

        self.child = None
        self.runstate = "stopped"
        pass

    async def handleEvent(self, grant: Grant):
        # Check for an updated to our violation flags.
        new_vflags = self.vflags
        if os.path.exists("/tmp/violate"):
            new_vflags = ViolationFlags.DEFAULT
            with open("/tmp/violate", "r") as fp:
                line = fp.readline()
                if line:
                    try:
                        new_vflags = ViolationFlags.from_string(line)
                    except ValueError as ex:
                        LOG.error(" Invalid violate flags in /tmp/violate; defaulting to %r: %r", ViolationFlags.DEFAULT, ex)
                else:
                    LOG.debug(" Empty /tmp/violate file; defaulting to %r", ViolationFlags.DEFAULT)
        else:
            new_vflags = ViolationFlags.NONE

        # Maybe updated violation flags.
        if self.vflags != new_vflags:
            LOG.info(" Updating violation flags: %r (was %r)", new_vflags, self.vflags)
            self.vflags = new_vflags

        old_grant_id = grant.id
        LOG.info("Grant event: %r - %r", old_grant_id, grant.status)

        #
        # Look for stop/start not related to grant replacement.
        #
        if not grant.replacement or not grant.replacement.new_grant_id:
            LOG.info("  Not a replacement grant")

            # Has to be an event for our current grant.
            if self.wrapper.revoked() or old_grant_id != self.wrapper.id:
                LOG.info("  Not our current grant, ignoring")
                return
 
            # Ignore the "replacing" state, has no meaning here.
            if grant.status == "replacing":
                LOG.info("  Ignoring replacing state")
                return

            if self.vflags == ViolationFlags.ALL:
                LOG.error("  Forcing a violation (all)!")
                return

            if grant.status == "active" and self.runstate == "stopped":
                if self.vflags & ViolationFlags.START:
                    LOG.error("  Forcing a violation (start)!")
                else:
                    await self.startTransmitter()
                if self.runstate == "running":
                    if self.vflags & ViolationFlags.HEARTBEAT:
                        LOG.error("  Forcing a violation (start heartbeat)!")
                    else:
                        self.wrapper.updateStatus(grant.status)
            elif (grant.status in ["pending", "paused", "revoked", "deleted"] and
                  self.runstate == "running"):
                if self.vflags & ViolationFlags.STOP:
                    LOG.error("  Forcing a violation (stop)!")
                else:
                    await self.stopTransmitter()
                if self.runstate == "stopped" and not self.args.replace_on_pending:
                    if self.vflags & ViolationFlags.HEARTBEAT:
                        LOG.error("  Forcing a violation (stop heartbeat)!")
                    else:
                        self.wrapper.updateStatus(grant.status)
                if self.args.replace_on_pending:
                    if grant.status == "pending":
                        if self.vflags & ViolationFlags.REPLACE:
                            LOG.error("  Forcing a violation (replace)!")
                        else:
                            self.wrapper.replace()
                    else:
                        if self.vflags & ViolationFlags.HEARTBEAT:
                            LOG.error("  Forcing a violation (%r heartbeat)!", grant.status)
                        else:
                            self.wrapper.updateStatus(grant.status)
            return

        #
        # Grant replacement. Might be an extension or a change in spectrum.
        # Make sure it is for our grant.
        #
        if grant.replacement.grant_id != self.wrapper.id:
            LOG.info("Ignoring grant replacement for another grant: %r",
                     grant.replacement.grant_id)
            return
        
        new_grant_id = grant.replacement.new_grant_id
        LOG.info("Grant Replacement: %r -> %r", old_grant_id, new_grant_id)

        #
        # Since we loop for it to go active, ignore any events that
        # come in after that.
        #
        if new_grant_id == self.wrapper.id:
            LOG.info("  Our current grant, ignoring")
            return

        #
        # There can be a delay, so lets poll for a bit until it goes
        # active or is denied/deleted.
        #
        for x in range(30):
            time.sleep(2)
            resp = self.ZMC.get_grant(new_grant_id, x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, Grant):
                LOG.error("Get Grant Error: %r", resp)
                continue
            grant = resp.parsed
            LOG.debug("Polled grant: %r", grant)
            if grant.status in ["active", "denied", "deleted", "pending"]:
                break
            pass

        #
        # We get a new grant (replacement) even if the grant extension is
        # denied. We have to ignore that.
        #
        if grant.status != "active":
            LOG.error("  The replaced grant was not approved!")
            LOG.error("  Ignoring grant change")
            return

        LOG.info("  Expiration: %s", grant.expires_at)

        #
        # Lets see if the range changed. If no change, then just make
        # sure the transmitter is running. Otherwise, stop and restart
        # on new range.
        #
        if not grant.constraints or len(grant.constraints) == 0 \
            or not grant.constraints[0].constraint \
            or not self.wrapper.constraints or len(self.wrapper.constraints) == 0 \
            or not self.wrapper.constraints[0].constraint:
            LOG.error("  No constraints in grant, cannot handle replacement")
            await self.stopTransmitter()
            self.wrapper.updateStatus(GrantOpStatus.PAUSED)
            return
        old_min_freq = self.wrapper.constraints[0].constraint.min_freq
        old_max_freq = self.wrapper.constraints[0].constraint.max_freq
        new_min_freq = grant.constraints[0].constraint.min_freq
        new_max_freq = grant.constraints[0].constraint.max_freq
        LOG.info("  Old frequency min/max: %r/%r", old_min_freq, old_max_freq)
        LOG.info("  New frequency min/max: %r/%r", new_min_freq, new_max_freq)

        if self.vflags == ViolationFlags.ALL:
            LOG.error("  Forcing a violation (all)!")
            return

        # Update our grant since it has changed internally.
        self.wrapper.grant = grant
        if self.runstate == "stopped":
            if self.vflags & ViolationFlags.START:
                LOG.error("  Forcing a violation (start)!")
            else:
                await self.startTransmitter()
            if self.runstate == "running":
                if self.vflags & ViolationFlags.HEARTBEAT:
                    LOG.error("  Forcing a violation (start heartbeat)!")
                else:
                    self.wrapper.updateStatus(grant.status)
            return

        if new_min_freq != old_min_freq or new_max_freq != old_max_freq:
            if self.vflags & ViolationFlags.STOP:
                LOG.error("  Forcing a violation (stop/start)!")
            else:
                await self.stopTransmitter()
                await self.startTransmitter()
            if self.vflags & ViolationFlags.HEARTBEAT:
                LOG.error("  Forcing a violation (stop/start heartbeat)!")
            else:
                self.wrapper.updateStatus(grant.status)
        pass

# The hander has to be outside the async main.
def set_signal_handler(signum, task_to_cancel):
    def handler(_signum, _frame):
        asyncio.get_running_loop().call_soon_threadsafe(task_to_cancel.cancel)
    signal.signal(signum, handler)

def init_main() -> tuple[ZMCSubscriptionCallback, Optional[HeartBeat], bool, str]:
    parser = argparse.ArgumentParser(
        prog="tx_wrapper",
        description="Handle grant change events for a transmitter")
    parser.add_argument(
        "-b", "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Do not actually transmit, just play pretend.")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "--txcommand", type=str,
        help="Optional external program to start and stop the radio transmitter. minimum and maximum frequencies are supplied as the environment variables (TXMAX_FREQ,TXMAX_FREQ). The script should expect a single argument, 'start' or 'stop'. Otherwise a builtin Powder specific command line is used.")
    
    parser.add_argument(
        "--cli", default=False, action="store_true",
        help="CLI mode instead of relying on a Powder experiment")
    parser.add_argument(
        "--grant-id", action=DefaultDestEnvAction, type=str, required=False,
        help="Grant ID that needs to watched")
    parser.register("type", "FreqRangeDesc", FreqRangeDesc.from_string)
    parser.add_argument(
        "--band", type='FreqRangeDesc', default="3360e6,3550e6,-70.0,1e6",
        help="A frequency range descriptor that defines band characteristics for a created grant: 'minfreq,maxfreq,max_eirp,width' (width is optional provide an empty string).")
    parser.add_argument(
        "--radio-id", type=str, required=False,
        help="Radio ID to bind to the created grant; default to the first tx-enabled port unless --radio-port-id also specified.")
    parser.add_argument(
        "--radio-port-id", type=str, required=False,
        help="Radio Port ID to bind to the created grant.  Overrides --radio-id if both specified.")
    parser.add_argument(
        "--is-powder", default=False, action='store_true',
        help="Enable POWDER-RDZ mode (and check for RDZ-in-RDZ nested case)")
    parser.add_argument(
        "--element-id", action=DefaultDestEnvAction, type=str, required=False,
        help="Local element ID")
    parser.add_argument(
        "--element-token", action=DefaultDestEnvAction, type=str, required=False,
        help="Local element token")
    parser.add_argument(
        "--zmc-http", action=DefaultDestEnvAction, type=str, required=False,
        help="Local ZMC URL")
    parser.add_argument(
        "--element-userid", action=DefaultDestEnvAction, type=str, required=False,
        help="User id that is bound to the element token. "+
        "Not required if using an admin token")
    parser.add_argument(
        "--replace-on-pending", default=False, action="store_true",
        help="Try to replace current grant when paused")
    parser.add_argument(
        "--heartbeat", default=False, action="store_true",
        help="Enable ZMC heartbeat protocol")

    #
    # Need these from either the command line or the manifest
    #
    element_id = None
    element_token = None
    element_userid = None
    zmc_http = None
    grant_id = None

    args = parser.parse_args(sys.argv[1:])

    if args.cli:
        if not (args.element_id and args.element_token and args.zmc_http):
            sys.exit("Please supply all ZMC arguments")
            pass

        if args.replace_on_pending and not args.band:
            sys.exit("Must provide --band info when requesting replace-on-pending")
            pass

        element_id = args.element_id
        element_token = args.element_token
        zmc_http = args.zmc_http
        if args.element_userid:
            element_userid = args.element_userid
            pass
        if args.grant_id:
            grant_id = args.grant_id
        elif not args.band:
            sys.exit("Must supply a grant ID or --band description")
            pass
    else:
        # Grab everything we need from the manifest.
        manifest_dict = parse_manifest()

        if not "rdzinfo" in manifest_dict:
            sys.exit("No rdzinfo in the manifest")
            pass
        rdzinfo = manifest_dict["rdzinfo"]
        if not "element-id" in rdzinfo:
            sys.exit("No element-id in the manifest rdzinfo")
            pass
        element_id = rdzinfo["element-id"]
        if not "zmc-http" in rdzinfo:
            sys.exit("No zmc-http in the manifest rdzinfo")
            pass
        zmc_http = rdzinfo["zmc-http"]
        if not "element-token" in manifest_dict["passwords"]:
            sys.exit("No element-token in the manifest passwords")
            pass
        element_token = manifest_dict["passwords"]["element-token"]
        #
        # the zms user id is optional, if we do not get it, then we can assume
        # that the token is a privileged token or we got an admin token.
        #
        element_userid = None
        if "user-id" in rdzinfo:
            element_userid = rdzinfo["user-id"]
            pass
        elif "admin-token" in manifest_dict["passwords"]:
            element_token = manifest_dict["passwords"]["admin-token"]
            pass

        #
        # Grant are in the spectrum section. For now just one grant, or
        # none at all. 
        #
        if ("spectrum" in manifest_dict and
            len(manifest_dict["spectrum"].keys()) > 0):
            grant_keys = list(manifest_dict["spectrum"].keys())
            grant_id = grant_keys[0]
            pass

        args.element_id = element_id
        args.element_token = element_token
        args.element_userid = element_userid
        args.zmc_http = zmc_http
        pass

    #
    # Heartbeats do not make sense in the Powder context where a grant
    # is supplied.
    #
    if grant_id and args.heartbeat:
        sys.exit("Not allowed to use heartbeats when supplying a Powder grant")
        pass

    #
    # Going to need this when running in Powder and allocating a grant
    #
    extid = None
    if not grant_id and os.path.exists("/var/emulab"):
        result = subprocess.run(
            ["geni-get", 'slice_urn'], capture_output=True, text=True)
        if result.returncode:
            sys.exit("Could not get slice urn")
            pass

        matched = re.match(r"^[^\+]+\+([^\+]+)\+slice\+(.*)$", result.stdout)
        if not matched:
            sys.exit("Could not decode geni slice urn")
            pass
        tokens = matched.group(1).split(":")
        if len(tokens) != 2 and len(tokens) != 3:
            sys.exit("Bad domain:project in geni slice urn")
            pass

        extid = "smarttx:" + tokens[1] + ":" + matched.group(2)
        pass

    if args.debug:
        LOG.setLevel(logging.DEBUG)
        #logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
        logging.getLogger('zmsclient').setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)
        pass

    ZMC = ZmsZmcClient(zmc_http, element_token,
                       raise_on_unexpected_status=True)

    #
    # When doing heartbeats, we need thread locking.
    #
    lock = threading.RLock()
    condition = threading.Condition(lock=lock)

    if grant_id:
        # Need to load the grant we are watching
        LOG.info("Loading grant")
        wrapper = GrantWrapper.load(ZMC, args, lock, condition, grant_id)
        pass
    else:
        LOG.info("Requesting grant")
        wrapper = GrantWrapper.create(ZMC, args, lock, condition, extid=extid)
        pass
    
    # Then subscribe to the events
    if element_userid:
        filter = EventFilter(
            element_ids=[element_id], user_ids=[element_userid])
    else:
        filter = EventFilter(
            element_ids=[element_id])
        pass
    ZMCsubscription = ZMCSubscriptionCallback(
        ZMC, element_id, args, wrapper,
        subscription=Subscription(
            id=str(uuid.uuid4()), filters=[filter]),
        reconnect_on_error=True)
    
    beater = None
    if args.heartbeat:
        beater = HeartBeat(wrapper)
        pass

    return ZMCsubscription, beater, args.daemon, args.logfile

async def async_main(ZMCsubscription: ZMCSubscriptionCallback, beater: Optional[HeartBeat]):
    this_task = asyncio.current_task();
    set_signal_handler(signal.SIGINT, this_task)
    set_signal_handler(signal.SIGHUP, this_task)
    set_signal_handler(signal.SIGTERM, this_task)

    subs: List[Any] = [ZMCsubscription]
    if beater:
        subs.append(beater)
        pass

    try:
        runnable = [sub.start() for sub in subs]
        await asyncio.gather(*runnable)
    except asyncio.CancelledError:
        # Avoid noise on ^C
        pass
    finally:
        if beater:
            beater.stop()
            pass
        await ZMCsubscription.stop()
        pass
    pass

#
# The reason for this split of main, is cause we need to do the daemon
# fork() before heading into the asyncio main.
#
def main():
    ZMCsubscription, beater, daemonize, logfile = init_main()
    format = "%(levelname)s:%(asctime)s:%(message)s"

    if daemonize:
        try:
            fp = open(logfile, "a");
            sys.stdout = fp
            sys.stderr = fp
            sys.stdin.close();
            logging.basicConfig(stream=fp, format=format)
            pass
        except:
            print("Could not open log file for append")
            sys.exit(1);
            pass
        pid = os.fork()
        if pid:
            sys.exit(0)
        os.setsid();
    else:
        logging.basicConfig(format=format)

    asyncio.run(async_main(ZMCsubscription, beater))


if __name__ == "__main__":
    main()
