import argparse
import csv
import datetime
import logging
import uuid
from typing import (List, Dict)

from zmsclient.dst.v1.models import (Metric, Value, AnyObject)

from . import (BaseDataType, MetricsHelper)

LOG = logging.getLogger(__name__)

class NemoCellCsvDataType(BaseDataType):
    name: str = "nemo-cell-csv"
    metric_schema = {
        "$id": "https://gitlab.flux.utah.edu/openzms/zms-dst-schemas/schemas/metrics/keysight/nemo-cell-csv/v1/schema.json",
        "type": "object",
        "title": "Keysight Nemo Cell Metrics CSV v1",
        "$schema": "https://json-schema.org/draft/2020-12/schema",
        "x-label-template": "pci=${tag_pci}",
        "x-tag-filter": {
            "cell": "serving"
        },
        "properties": {
            "rsrp": {
                "type": "number",
                "description": "Reference Signal Received Power (RSRP)",
                "x-unit": "dBm",
                "x-group": "cellular"
            },
            "rsrq": {
                "type": "number",
                "description": "Reference Signal Received Quality (RSRQ).",
                "x-unit": "dB",
                "x-group": "cellular-quality"
            },
            "sinr": {
                "type": "number",
                "description": "Signal to Interference plus Noise Ratio (SINR).",
                "x-unit": "dB",
                "x-group": "cellular-quality"
            },
            "pci": {
                "type": "string",
                "description": "Physical Cell ID (PCI).",
                "x-tag": True
            },
            "system": {
                "type": "string",
                "description": "Measured System (e.g NR).",
                "x-tag": True
            },
            "arfcn": {
                "type": "integer",
                "description": "Absolute Radio Frequency Channel Number (ARFCN).",
                "x-tag": True
            },
            "cell": {
                "type": "string",
                "description": "Cell type: serving or neighbor.",
                "x-tag": True
            }
        }
    }
    metric_defn: Metric = Metric(
        name="keysight.nemo-cell-csv.v1",
        element_id=str(uuid.UUID(int=0)),
        description="Keysight Nemo Cell Metrics CSV v1.",
        schema=AnyObject.from_dict(metric_schema)
    )
    field_map: Dict[str,str] = dict(
        rsrp="RSRP (NR SpCell)",
        rsrq="RSRQ (NR SpCell)",
        sinr="SINR (NR SpCell)",
    )
    tag_map: Dict[str,str] = dict(
        pci="Physical cell identity (NR SpCell)",
        system="Measured System",
        arfcn="NR-ARFCN (NR SpCell)",
    )

    def __init__(self, args: argparse.Namespace, mh: "MetricsHelper"):
        super().__init__(args, mh)
        if self.args.nemo_cell_date is None:
            raise ValueError("Nemo data files do not include a date; --nemo-cell-date is required.")
        try:
            datetime.datetime.strptime(self.args.nemo_cell_date, "%Y-%m-%d").date()
        except Exception as e:
            raise ValueError(f"Could not parse --nemo-cell-date {self.args.nemo_cell_date} as %Y-%m-%d: {e}")

    @classmethod
    def add_args(cls, parser: argparse.ArgumentParser):
        """Add any additional arguments needed for this data type."""
        parser.add_argument(
            "--nemo-cell-date", type=str, required=False, default=None,
            help="Nemo timestamps do not include a date; specify as %Y-%m-%d (e.g. 2025-09-31).")

    @classmethod
    def get_metric_definitions(cls) -> List[Metric]:
        """Define the metrics that will be collected."""
        return [cls.metric_defn]

    # Parse the given file.
    def load_data(self):
        """Parse the given file."""
        with open(self.args.data_file, "r") as f:
            csvreader = csv.DictReader(f)
            i = 0
            for row in csvreader:
                i += 1
                try:
                    t = self.args.nemo_cell_date + " " + row["Time"]
                    dt = datetime.datetime.strptime(t, "%Y-%m-%d %H:%M:%S.%f")
                    dt = dt.astimezone(datetime.timezone.utc)
                except Exception as e:
                    msg = f"Cannot parse row {i} with bad date/time ({e}): {row}"
                    if self.args.continue_on_error:
                        LOG.warning(msg)
                        continue
                    else:
                        raise ValueError(msg)
                # First assemble the values and tags for the serving cell.
                svd = dict()
                for fn, cn in self.field_map.items():
                    v = row.get(cn, None)
                    if v is None or v == "" or v == "n/a":
                        continue
                    try:
                        vt = self.metric_schema["properties"][fn]["type"]
                        if vt == "number":
                            svd[fn] = float(v)
                        elif vt == "integer":
                            svd[fn] = int(v)
                        else:
                            svd[fn] = v
                        # One-off: convert speed in km/h to m/s.
                        if fn == "speed":
                            svd[fn] = svd[fn] * (1000.0 / (60.0 * 60.0))
                    except Exception as e:
                        msg = "Could not parse field %s with value %s as type %s: %s" % (fn, v, vt, e)
                        if self.args.continue_on_error:
                            LOG.warning(msg)
                            continue
                        else:
                            raise ValueError(msg)
                if len(svd) == 0:
                    LOG.debug("Skipping empty serving cell on line %d", i)
                    continue
                std = dict()
                for tn, cn in self.tag_map.items():
                    v = row.get(cn, None)
                    if v is None or v == "" or v == "n/a":
                        continue
                    try:
                        vt = self.metric_schema["properties"][tn]["type"]
                        if vt == "number":
                            std[tn] = float(v)
                        elif vt == "integer":
                            std[tn] = int(v)
                        else:
                            std[tn] = v
                    except Exception as e:
                        msg = "Could not parse tag %s with value %s as type %s: %s" % (tn, v, vt, e)
                        if self.args.continue_on_error:
                            LOG.warning(msg)
                            continue
                        else:
                            raise ValueError(msg)
                std["cell"] = "serving"
                v = Value(
                    monitor_id=self.mh.monitor_id,
                    metric_id=str(uuid.UUID(int=0)),  # filled in later
                    created_at=dt,
                    fields=AnyObject.from_dict(svd),
                    tags=AnyObject.from_dict(std)
                )
                self.values.append(v)
                LOG.debug("Parsed row %d serving cell: %s", i, v)

    def prepare_values(self):
        """Prepare any values before uploading."""
        if len(self.values) == 0:
            LOG.warning("No values to prepare")
            return
        for v in self.values:
            m = self.mh.get_metric(self.metric_defn.name)
            if not m:
                raise ValueError(f"Metric {self.metric_defn.name} not found in MetricsHelper")
            v.metric_id = m.id
            m = self.mh.get_series(self.metric_defn.name)
            if not m:
                raise ValueError(f"Series for metric {self.metric_defn.name} not found in MetricsHelper")
            v.series_id = m.id

class NemoGpsCsvDataType(BaseDataType):
    name: str = "nemo-gps-csv"
    metric_schema = {
        "$id": "https://gitlab.flux.utah.edu/openzms/zms-dst-schemas/schemas/metrics/zms/location/gps/v1/schema.json",
        "type": "object",
        "title": "OpenZMS GPS Location Data v1",
        "$schema": "https://json-schema.org/draft/2020-12/schema",
        "properties": {
            "systemid": {
                "type": "integer",
                "description": "NMEA 4.11 GPS system identifier (1=GPS, 2=GLONASS, 3=Galileo, 4=BDS).",
                "x-tag": True
            },
            "signalid": {
                "type": "integer",
                "description": "NMEA 4.11 Signal identifier.",
                "x-tag": True
            },
            "device": {
                "type": "string",
                "description": "Measuring device.",
                "x-tag": True
            },
            "mode": {
                "type": "integer",
                "description": "Mode (0=unknown, 1=no fix, 2=2D fix, 3=3D fix).",
                "x-group": "system",
                "x-tag": True
            },
            "lat": {
                "type": "number",
                "description": "Latitude.",
                "x-unit": "degrees",
                "x-group": "coordinates"
            },
            "lon": {
                "type": "number",
                "description": "Longitude.",
                "x-unit": "degrees",
                "x-group": "coordinates"
            },
            "alt": {
                "type": "number",
                "description": "Altitude.",
                "x-unit": "meters",
                "x-group": "elevation"
            },
            "altmsl": {
                "type": "number",
                "description": "Altitude (mean sea level).",
                "x-unit": "meters",
                "x-group": "elevation"
            },
            "althae": {
                "type": "number",
                "description": "Altitude (height above ellipsoid).",
                "x-unit": "meters",
                "x-group": "elevation"
            },
            "track": {
                "type": "number",
                "description": "Track.",
                "x-unit": "degrees",
                "x-group": "heading"
            },
            "magtrack": {
                "type": "number",
                "description": "Magnetic track.",
                "x-unit": "degrees",
                "x-group": "heading"
            },
            "magvar": {
                "type": "number",
                "description": "Magnetic variation.",
                "x-unit": "degrees",
                "x-group": "heading"
            },
            "speed": {
                "type": "number",
                "description": "Speed.",
                "x-unit": "meters/second",
                "x-group": "rate"
            },
            "climb": {
                "type": "number",
                "description": "Climb rate.",
                "x-unit": "meters/second",
                "x-group": "rate"
            },
            "geoidsep": {
                "type": "number",
                "description": "Geoid separation.",
                "x-unit": "meters",
                "x-group": "elevation"
            },
            "sep": {
                "type": "number",
                "description": "Estimated vertical error.",
                "x-unit": "meters",
                "x-group": "error"
            },
            "eph": {
                "type": "number",
                "description": "Estimated horizontal error.",
                "x-unit": "meters",
                "x-group": "error"
            },
            "epv": {
                "type": "number",
                "description": "Estimated vertical error.",
                "x-unit": "meters",
                "x-group": "error"
            },
            "epx": {
                "type": "number",
                "description": "Estimated longitude error.",
                "x-unit": "meters",
                "x-group": "error"
            },
            "epy": {
                "type": "number",
                "description": "Estimated latitude error.",
                "x-unit": "meters",
                "x-group": "error"
            },
            "ept": {
                "type": "number",
                "description": "Estimated timestamp error.",
                "x-unit": "seconds",
                "x-group": "error"
            },
            "eps": {
                "type": "number",
                "description": "Estimated speed error.",
                "x-unit": "meters/second",
                "x-group": "error"
            },
            "epc": {
                "type": "number",
                "description": "Estimated climb error.",
                "x-unit": "meters/second",
                "x-group": "error"
            },
            "dop": {
                "type": "number",
                "description": "Dilution of precision.",
                "x-group": "error"
            },
            "xdop": {
                "type": "number",
                "description": "Horizontal dilution of precision.",
                "x-group": "error"
            },
            "ydop": {
                "type": "number",
                "description": "Vertical dilution of precision.",
                "x-group": "error"
            },
            "hdop": {
                "type": "number",
                "description": "Horizontal dilution of precision.",
                "x-group": "error"
            },
            "vdop": {
                "type": "number",
                "description": "Vertical dilution of precision.",
                "x-group": "error"
            },
            "gdop": {
                "type": "number",
                "description": "Geometric dilution of precision.",
                "x-group": "error"
            },
            "pdop": {
                "type": "number",
                "description": "Position dilution of precision.",
                "x-group": "error"
            },
            "tdop": {
                "type": "number",
                "description": "Time dilution of precision.",
                "x-group": "error"
            },
            "nsat": {
                "type": "integer",
                "description": "Number of visible satellites.",
                "x-group": "info"
            },
            "usat": {
                "type": "integer",
                "description": "Number of satellites used in solution.",
                "x-group": "info"
            }
        },
        "required": [
            "lat",
            "lon"
        ]
    }
    metric_defn: Metric = Metric(
        name="zms.location.gps.v1",
        element_id=str(uuid.UUID(int=0)),
        description="OpenZMS GPS Location Data v1",
        schema=AnyObject.from_dict(metric_schema)
    )
    field_map: Dict[str,str] = dict(
        lon="Longitude",
        lat="Latitude",
        alt="Height",
        nsat="GPS satellites",
        hdop="Horizontal dilution of precision",
        pdop="Position dilution of precision",
        vdop="Vertical dilution of precision",
        speed="Velocity",
    )
    tag_map: Dict[str,str] = dict()

    def __init__(self, args: argparse.Namespace, mh: "MetricsHelper"):
        super().__init__(args, mh)
        if self.args.nemo_cell_date is None:
            raise ValueError("Nemo data files do not include a date; --nemo-cell-date is required.")
        try:
            datetime.datetime.strptime(self.args.nemo_cell_date, "%Y-%m-%d").date()
        except Exception as e:
            raise ValueError(f"Could not parse --nemo-cell-date {self.args.nemo_cell_date} as %Y-%m-%d: {e}")

    @classmethod
    def add_args(cls, parser: argparse.ArgumentParser):
        """Add any additional arguments needed for this data type."""
        pass

    @classmethod
    def get_metric_definitions(cls) -> List[Metric]:
        """Define the metrics that will be collected."""
        return [cls.metric_defn]

    # Parse the given file.
    def load_data(self):
        """Parse the given file."""
        with open(self.args.data_file, "r") as f:
            csvreader = csv.DictReader(f)
            i = 0
            for row in csvreader:
                i += 1
                try:
                    t = self.args.nemo_cell_date + " " + row["Time"]
                    dt = datetime.datetime.strptime(t, "%Y-%m-%d %H:%M:%S.%f")
                    dt = dt.astimezone(datetime.timezone.utc)
                except Exception as e:
                    msg = f"Cannot parse row {i} with bad date/time ({e}): {row}"
                    if self.args.continue_on_error:
                        LOG.warning(msg)
                        continue
                    else:
                        raise ValueError(msg)
                # Assemble the values and tags for the GPS data.
                svd = dict()
                for fn, cn in self.field_map.items():
                    v = row.get(cn, None)
                    if v is None or v == "" or v == "n/a":
                        continue
                    vt = None
                    try:
                        vt = self.metric_schema["properties"][fn]["type"]
                        if vt == "number":
                            svd[fn] = float(v)
                        elif vt == "integer":
                            svd[fn] = int(v)
                        else:
                            svd[fn] = v
                        # One-off: convert speed in km/h to m/s.
                        if fn == "speed":
                            svd[fn] = svd[fn] * (1000.0 / (60.0 * 60.0))
                    except Exception as e:
                        msg = "Could not parse field %s with value %s as type %s: %s" % (fn, v, vt, e)
                        if self.args.continue_on_error:
                            LOG.warning(msg)
                            continue
                        else:
                            raise ValueError(msg)
                std = dict()
                for tn, cn in self.tag_map.items():
                    v = row.get(cn, None)
                    if v is None or v == "" or v == "n/a":
                        continue
                    vt = None
                    try:
                        vt = self.metric_schema["properties"][tn]["type"]
                        if vt == "number":
                            std[tn] = float(v)
                        elif vt == "integer":
                            std[tn] = int(v)
                        else:
                            std[tn] = v
                    except Exception as e:
                        msg = "Could not parse tag %s with value %s as type %s: %s" % (tn, v, vt, e)
                        if self.args.continue_on_error:
                            LOG.warning(msg)
                            continue
                        else:
                            raise ValueError(msg)
                v = Value(
                    monitor_id=self.mh.monitor_id,
                    metric_id=str(uuid.UUID(int=0)),  # filled in later
                    created_at=dt,
                    fields=AnyObject.from_dict(svd),
                    tags=AnyObject.from_dict(std)
                )
                self.values.append(v)
                LOG.debug("Parsed row %d: %s", i, v)

    def prepare_values(self):
        """Prepare any values before uploading."""
        if len(self.values) == 0:
            LOG.warning("No values to prepare")
            return
        for v in self.values:
            m = self.mh.get_metric(self.metric_defn.name)
            if not m:
                raise ValueError(f"Metric {self.metric_defn.name} not found in MetricsHelper")
            v.metric_id = m.id
            m = self.mh.get_series(self.metric_defn.name)
            if not m:
                raise ValueError(f"Series for metric {self.metric_defn.name} not found in MetricsHelper")
            v.series_id = m.id