#!/bin/python

import sys
import argparse
import logging
from typing import cast

from zmsclient.zmc.client import ZmsZmcClient
from zmsclient.dst.v1.models import MetricList
from zmsclient.zmc.v1.models import Monitor, Radio
from zmsclient.zmc.v1.models import MonitorList, ActionList, RadioList, AntennaList, LocationList, SpectrumList
from zmsclient.common import DefaultDestEnvAction

LOG = logging.getLogger(__name__)

def fatal(response):
    LOG.error("ERROR: %r", response)
    exit(-1)
    pass

#
# Delete all Monitors
#
def deleteMetrics(inner_zmc_client: ZmsZmcClient, impotent: bool = False):
    resp = inner_zmc_client.list_metrics(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MetricList):
        fatal(resp)
    metrics = cast(MetricList, resp.parsed)
    for metric in metrics.metrics:
        if impotent:
            LOG.info("Not deleting metric: %r", metric.id)
            continue
        LOG.debug("Deleting: %r", metric)
        response = inner_zmc_client.delete_metric(metric.id)
        if not response.is_success:
            fatal(response)
            pass
        LOG.info("Deleted metric: %r", metric.id)
        LOG.debug("Deleted metrics: %r", response)
        pass
    pass

#
# Delete all Monitors
#
def deleteMonitors(inner_zmc_client: ZmsZmcClient, impotent: bool = False):
    resp = inner_zmc_client.list_monitors(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MonitorList):
        fatal(resp)
    monitors = cast(MonitorList, resp.parsed)
    for monitor in monitors.monitors:
        if impotent:
            LOG.info("Not deleting monitor: %r", monitor.id)
            continue
        LOG.debug("Deleting: %r", monitor)
        response = inner_zmc_client.delete_monitor(str(monitor.id))
        if not response.is_success:
            fatal(response)
            pass
        LOG.info("Deleted monitor: %r", monitor.id)
        LOG.debug("Deleted monitor: %r", response)
        pass
    pass

#
# Delete all Actions
#
def deleteActions(inner_zmc_client: ZmsZmcClient, impotent: bool = False):
    resp = inner_zmc_client.list_actions(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, ActionList):
        fatal(resp)
    actions = cast(ActionList, resp.parsed)
    for action in actions.actions:
        if impotent:
            LOG.info("Not deleting action: %r", action.id)
            continue
        LOG.debug("Deleting: %r", action)
        response = inner_zmc_client.delete_action(str(action.id))
        if not response.is_success:
            fatal(response)
            pass
        LOG.info("Deleted action: %r", action.id)
        LOG.debug("Deleted action: %r", response)
        pass
    pass

#
# Delete all Radios
#
def deleteRadios(inner_zmc_client: ZmsZmcClient, impotent=False):
    resp = inner_zmc_client.list_radios(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, RadioList):
        fatal(resp)
    radios = cast(RadioList, resp.parsed)
    for radio in radios.radios:
        # Reload with elaborate so we get the ports.
        resp = inner_zmc_client.get_radio(str(radio.id), x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Radio):
            fatal("Could not reload radio: %r" % (radio.id))
        radio = cast(Radio, resp.parsed)
        if radio.ports and len(radio.ports) > 0:
            for port in radio.ports:
                if impotent:
                    LOG.info("Not deleting radio port: %r", port.id)
                    continue
                LOG.debug("Deleting: %r", port)
                response = inner_zmc_client.delete_radio_port(str(radio.id), str(port.id))
                if not response.is_success:
                    fatal(response)
                    pass
                LOG.info("Deleted radio port: %r", port.id)
                LOG.debug("Deleted radio port: %r", response)
                pass
        if impotent:
            LOG.info("Not deleting radio: %r", radio.id)
            continue
        LOG.debug("Deleting radio: %r", radio)
        response = inner_zmc_client.delete_radio(str(radio.id))
        if not response.is_success:
            fatal(response)
            pass
        LOG.info("Deleted radio: %r", radio.id)
        LOG.debug("Deleted radio: %r", response)
        pass
    pass

#
# Delete all Antennas
#
def deleteAntennas(inner_zmc_client: ZmsZmcClient, impotent: bool = False):
    resp = inner_zmc_client.list_antennas(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, AntennaList):
        fatal(resp)
    antennas = cast(AntennaList, resp.parsed)
    for antenna in antennas.antennas:
        if impotent:
            LOG.info("Not deleting antenna: %r", antenna.id)
            continue
        LOG.debug("Deleting antenna: %r", antenna)
        response = inner_zmc_client.delete_antenna(str(antenna.id))
        if not response.is_success:
            fatal(response)
            pass
        LOG.info("Deleted antenna: %r", antenna.id)
        LOG.debug("Deleted antenna: %r", response)
        pass
    pass

#
# Delete all locations
#
def deleteLocations(inner_zmc_client: ZmsZmcClient, impotent: bool = False):
    resp = inner_zmc_client.list_locations(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, LocationList):
        fatal(resp)
    locations = cast(LocationList, resp.parsed)
    for location in locations.locations:
        if impotent:
            LOG.info("Not deleting location: %r", location.id)
            continue
        LOG.debug("Deleting location: %r", location)
        response = inner_zmc_client.delete_location(str(location.id))
        if not response.is_success:
            fatal(response)
            pass
        LOG.info("Deleted location: %r", location.id)
        LOG.debug("Deleted location: %r", response)
        pass
    pass

#
# Delete all spectrum
#
def deleteSpectrum(inner_zmc_client: ZmsZmcClient, impotent: bool = False):
    resp = inner_zmc_client.list_spectrum(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, SpectrumList):
        fatal(resp)
    spectrumlist = cast(SpectrumList, resp.parsed)
    for spectrum in spectrumlist.spectrum:
        if impotent:
            LOG.info("Not deleting spectrum: %r", spectrum.id)
            continue
        LOG.debug("Deleting spectrum: %r", spectrum)
        response = inner_zmc_client.delete_spectrum(str(spectrum.id))
        if not response.is_success:
            fatal(response)
            pass
        LOG.info("Deleted spectrum: %r", spectrum.id)
        LOG.debug("Deleted spectrum: %r", response)
        pass
    pass

def main():
    parser = argparse.ArgumentParser(
        prog="loadzmc",
        description="Delete all objects (radios, antennas, monitors, locations) from a ZMC.")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Impotent: do not inject events")
    parser.add_argument(
        "--element-token", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) element token")
    parser.add_argument(
        "--zmc-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) ZMC URL")

    args = parser.parse_args(sys.argv[1:])

    logging.basicConfig()
    if args.debug:
        LOG.setLevel(logging.DEBUG)
        logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)

    inner_zmc_client = ZmsZmcClient(
        args.zmc_http, args.element_token,
        raise_on_unexpected_status=True)

    #deleteMetrics(inner_zmc_client, impotent=args.impotent)
    deleteMonitors(inner_zmc_client, impotent=args.impotent)
    deleteActions(inner_zmc_client, impotent=args.impotent)
    deleteRadios(inner_zmc_client, impotent=args.impotent)
    deleteAntennas(inner_zmc_client, impotent=args.impotent)
    deleteLocations(inner_zmc_client, impotent=args.impotent)
    deleteSpectrum(inner_zmc_client, impotent=args.impotent)

if __name__ == "__main__":
    main()
