from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.metric import Metric
from ...types import UNSET, Response, Unset


def _get_kwargs(
    *,
    body: Metric,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    _kwargs: Dict[str, Any] = {
        "method": "post",
        "url": "/metrics",
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, Metric]]:
    if response.status_code == HTTPStatus.CREATED:
        try:
            response_201 = Metric.from_dict(response.json())

            return response_201
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, Metric]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    client: AuthenticatedClient,
    *,
    body: Metric,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, Metric]]:
    """Create a new metric.

    Args:
        x_api_elaborate (Union[Unset, str]):
        body (Metric): Description of a Metric whose Values may be reported from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Metric]]
    """

    kwargs = _get_kwargs(
        body=body,
        x_api_elaborate=x_api_elaborate,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    client: AuthenticatedClient,
    *,
    body: Metric,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, Metric]]:
    """Create a new metric.

    Args:
        x_api_elaborate (Union[Unset, str]):
        body (Metric): Description of a Metric whose Values may be reported from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Metric]
    """

    return sync_detailed(
        client=client,
        body=body,
        x_api_elaborate=x_api_elaborate,
    ).parsed


async def asyncio_detailed(
    client: AuthenticatedClient,
    *,
    body: Metric,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, Metric]]:
    """Create a new metric.

    Args:
        x_api_elaborate (Union[Unset, str]):
        body (Metric): Description of a Metric whose Values may be reported from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Metric]]
    """

    kwargs = _get_kwargs(
        body=body,
        x_api_elaborate=x_api_elaborate,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    client: AuthenticatedClient,
    *,
    body: Metric,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, Metric]]:
    """Create a new metric.

    Args:
        x_api_elaborate (Union[Unset, str]):
        body (Metric): Description of a Metric whose Values may be reported from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Metric]
    """

    return (
        await asyncio_detailed(
            client=client,
            body=body,
            x_api_elaborate=x_api_elaborate,
        )
    ).parsed
