from io import BytesIO
from typing import Any, Dict, List, Type, TypeVar, Union, cast

from attrs import define as _attrs_define
from attrs import field as _attrs_field

from ..types import UNSET, File, FileJsonType, Unset

T = TypeVar("T", bound="SpectrumAuth")


@_attrs_define
class SpectrumAuth:
    """Associates authorization information from an external allocation system for a Spectrum.

    Attributes:
        spectrum_id (str): The associated spectrum id.
        type (str): The type of authorization (e.g., `spectracert`).
        ext_id (str): An external ID associated with this spectrum auth by the delegating user; opaque to OpenZMS.
        id (Union[Unset, str]): The spectrum auth id.
        url (Union[None, Unset, str]): A URL describing this spectrum auth, establishing permission/legality to operate
            in this spectrum, etc.
        data (Union[File, None, Unset]): The raw authorization data, if any.
        descriptor (Union[File, None, Unset]): A human-readable descriptor (e.g. JSON) that describes the authorization.
    """

    spectrum_id: str
    type: str
    ext_id: str
    id: Union[Unset, str] = UNSET
    url: Union[None, Unset, str] = UNSET
    data: Union[File, None, Unset] = UNSET
    descriptor: Union[File, None, Unset] = UNSET
    additional_properties: Dict[str, Any] = _attrs_field(init=False, factory=dict)
    _immutable_on = {
        "id": {"put": True, "post": True},
        "spectrum_id": {"put": True, "post": None},
        "type": {"put": True, "post": None},
        "ext_id": {"put": True, "post": None},
        "url": {"put": True, "post": None},
        "data": {"put": True, "post": None},
        "descriptor": {"put": True, "post": None},
    }

    def to_dict(self) -> Dict[str, Any]:
        spectrum_id = self.spectrum_id

        type = self.type

        ext_id = self.ext_id

        id = self.id

        url: Union[None, Unset, str]
        if isinstance(self.url, Unset):
            url = UNSET
        else:
            url = self.url

        data: Union[FileJsonType, None, Unset]
        if isinstance(self.data, Unset):
            data = UNSET
        elif isinstance(self.data, File):
            data = self.data.to_tuple()

        else:
            data = self.data

        descriptor: Union[FileJsonType, None, Unset]
        if isinstance(self.descriptor, Unset):
            descriptor = UNSET
        elif isinstance(self.descriptor, File):
            descriptor = self.descriptor.to_tuple()

        else:
            descriptor = self.descriptor

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "spectrum_id": spectrum_id,
                "type": type,
                "ext_id": ext_id,
            }
        )
        if id is not UNSET:
            field_dict["id"] = id
        if url is not UNSET:
            field_dict["url"] = url
        if data is not UNSET:
            field_dict["data"] = data
        if descriptor is not UNSET:
            field_dict["descriptor"] = descriptor

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        d = src_dict.copy()
        spectrum_id = d.pop("spectrum_id")

        type = d.pop("type")

        ext_id = d.pop("ext_id")

        id = d.pop("id", UNSET)

        def _parse_url(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        url = _parse_url(d.pop("url", UNSET))

        def _parse_data(data: object) -> Union[File, None, Unset]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, bytes):
                    raise TypeError()
                data_type_0 = File(payload=BytesIO(data))

                return data_type_0
            except:  # noqa: E722
                pass
            return cast(Union[File, None, Unset], data)

        data = _parse_data(d.pop("data", UNSET))

        def _parse_descriptor(data: object) -> Union[File, None, Unset]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, bytes):
                    raise TypeError()
                descriptor_type_0 = File(payload=BytesIO(data))

                return descriptor_type_0
            except:  # noqa: E722
                pass
            return cast(Union[File, None, Unset], data)

        descriptor = _parse_descriptor(d.pop("descriptor", UNSET))

        spectrum_auth = cls(
            spectrum_id=spectrum_id,
            type=type,
            ext_id=ext_id,
            id=id,
            url=url,
            data=data,
            descriptor=descriptor,
        )

        spectrum_auth.additional_properties = d
        return spectrum_auth

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties


def _load_lazy_imports(ns=globals()):
    pass
