from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.event import Event
from ...types import UNSET, Response, Unset


def _get_kwargs(
    subscription_id: str,
    *,
    x_api_elaborate: Union[Unset, str] = UNSET,
    connection: str,
    upgrade: str,
    sec_web_socket_version: str,
    sec_web_socket_key: str,
    x_delete_on_close: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    headers["Connection"] = connection

    headers["Upgrade"] = upgrade

    headers["Sec-WebSocket-Version"] = sec_web_socket_version

    headers["Sec-WebSocket-Key"] = sec_web_socket_key

    if not isinstance(x_delete_on_close, Unset):
        headers["X-Delete-On-Close"] = x_delete_on_close

    _kwargs: Dict[str, Any] = {
        "method": "get",
        "url": "/subscriptions/{subscription_id}/events".format(
            subscription_id=subscription_id,
        ),
    }

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, Event]]:
    if response.status_code == HTTPStatus.OK:
        try:
            response_200 = Event.from_dict(response.json())

            return response_200
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.INTERNAL_SERVER_ERROR:
        try:
            response_500 = Error.from_dict(response.json())

            return response_500
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, Event]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    client: AuthenticatedClient,
    subscription_id: str,
    *,
    x_api_elaborate: Union[Unset, str] = UNSET,
    connection: str,
    upgrade: str,
    sec_web_socket_version: str,
    sec_web_socket_key: str,
    x_delete_on_close: Union[Unset, str] = UNSET,
) -> Response[Union[Error, Event]]:
    """Stream events corresponding to a subscription over a WebSocket upgrade of the invoking HTTP
    connection.  Only one event stream may be associated with each subscription.  Matching events are
    returned as JSON-formatted `Event` objects, as described in this schema.

    Args:
        subscription_id (str): The target subscription.
        x_api_elaborate (Union[Unset, str]):
        connection (str): You must provide the `Connection: Upgrade` header. Example: Upgrade.
        upgrade (str): You must provide the `Upgrade: websocket` header. Example: websocket.
        sec_web_socket_version (str): You must provide the `Sec-WebSocket-Version: 3` header.
            Example: 13.
        sec_web_socket_key (str): You must provide the `Sec-WebSocket-Key: <base64>` header.  The
            value should be a 16-character string, encoded in base64, according to the WebSocket
            specification. Example: MDAwMDAwMDAwMDAwMDAwMA==.
        x_delete_on_close (Union[Unset, str]): Set to `true` to persist the associated
            subscription even if this WebSocket connection is closed.  By default, or if not set to
            `true`, when this WebSocket is closed, the subscription will be automatically deleted.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Event]]
    """

    kwargs = _get_kwargs(
        subscription_id=subscription_id,
        x_api_elaborate=x_api_elaborate,
        connection=connection,
        upgrade=upgrade,
        sec_web_socket_version=sec_web_socket_version,
        sec_web_socket_key=sec_web_socket_key,
        x_delete_on_close=x_delete_on_close,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    client: AuthenticatedClient,
    subscription_id: str,
    *,
    x_api_elaborate: Union[Unset, str] = UNSET,
    connection: str,
    upgrade: str,
    sec_web_socket_version: str,
    sec_web_socket_key: str,
    x_delete_on_close: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, Event]]:
    """Stream events corresponding to a subscription over a WebSocket upgrade of the invoking HTTP
    connection.  Only one event stream may be associated with each subscription.  Matching events are
    returned as JSON-formatted `Event` objects, as described in this schema.

    Args:
        subscription_id (str): The target subscription.
        x_api_elaborate (Union[Unset, str]):
        connection (str): You must provide the `Connection: Upgrade` header. Example: Upgrade.
        upgrade (str): You must provide the `Upgrade: websocket` header. Example: websocket.
        sec_web_socket_version (str): You must provide the `Sec-WebSocket-Version: 3` header.
            Example: 13.
        sec_web_socket_key (str): You must provide the `Sec-WebSocket-Key: <base64>` header.  The
            value should be a 16-character string, encoded in base64, according to the WebSocket
            specification. Example: MDAwMDAwMDAwMDAwMDAwMA==.
        x_delete_on_close (Union[Unset, str]): Set to `true` to persist the associated
            subscription even if this WebSocket connection is closed.  By default, or if not set to
            `true`, when this WebSocket is closed, the subscription will be automatically deleted.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Event]
    """

    return sync_detailed(
        subscription_id=subscription_id,
        client=client,
        x_api_elaborate=x_api_elaborate,
        connection=connection,
        upgrade=upgrade,
        sec_web_socket_version=sec_web_socket_version,
        sec_web_socket_key=sec_web_socket_key,
        x_delete_on_close=x_delete_on_close,
    ).parsed


async def asyncio_detailed(
    client: AuthenticatedClient,
    subscription_id: str,
    *,
    x_api_elaborate: Union[Unset, str] = UNSET,
    connection: str,
    upgrade: str,
    sec_web_socket_version: str,
    sec_web_socket_key: str,
    x_delete_on_close: Union[Unset, str] = UNSET,
) -> Response[Union[Error, Event]]:
    """Stream events corresponding to a subscription over a WebSocket upgrade of the invoking HTTP
    connection.  Only one event stream may be associated with each subscription.  Matching events are
    returned as JSON-formatted `Event` objects, as described in this schema.

    Args:
        subscription_id (str): The target subscription.
        x_api_elaborate (Union[Unset, str]):
        connection (str): You must provide the `Connection: Upgrade` header. Example: Upgrade.
        upgrade (str): You must provide the `Upgrade: websocket` header. Example: websocket.
        sec_web_socket_version (str): You must provide the `Sec-WebSocket-Version: 3` header.
            Example: 13.
        sec_web_socket_key (str): You must provide the `Sec-WebSocket-Key: <base64>` header.  The
            value should be a 16-character string, encoded in base64, according to the WebSocket
            specification. Example: MDAwMDAwMDAwMDAwMDAwMA==.
        x_delete_on_close (Union[Unset, str]): Set to `true` to persist the associated
            subscription even if this WebSocket connection is closed.  By default, or if not set to
            `true`, when this WebSocket is closed, the subscription will be automatically deleted.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Event]]
    """

    kwargs = _get_kwargs(
        subscription_id=subscription_id,
        x_api_elaborate=x_api_elaborate,
        connection=connection,
        upgrade=upgrade,
        sec_web_socket_version=sec_web_socket_version,
        sec_web_socket_key=sec_web_socket_key,
        x_delete_on_close=x_delete_on_close,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    client: AuthenticatedClient,
    subscription_id: str,
    *,
    x_api_elaborate: Union[Unset, str] = UNSET,
    connection: str,
    upgrade: str,
    sec_web_socket_version: str,
    sec_web_socket_key: str,
    x_delete_on_close: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, Event]]:
    """Stream events corresponding to a subscription over a WebSocket upgrade of the invoking HTTP
    connection.  Only one event stream may be associated with each subscription.  Matching events are
    returned as JSON-formatted `Event` objects, as described in this schema.

    Args:
        subscription_id (str): The target subscription.
        x_api_elaborate (Union[Unset, str]):
        connection (str): You must provide the `Connection: Upgrade` header. Example: Upgrade.
        upgrade (str): You must provide the `Upgrade: websocket` header. Example: websocket.
        sec_web_socket_version (str): You must provide the `Sec-WebSocket-Version: 3` header.
            Example: 13.
        sec_web_socket_key (str): You must provide the `Sec-WebSocket-Key: <base64>` header.  The
            value should be a 16-character string, encoded in base64, according to the WebSocket
            specification. Example: MDAwMDAwMDAwMDAwMDAwMA==.
        x_delete_on_close (Union[Unset, str]): Set to `true` to persist the associated
            subscription even if this WebSocket connection is closed.  By default, or if not set to
            `true`, when this WebSocket is closed, the subscription will be automatically deleted.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Event]
    """

    return (
        await asyncio_detailed(
            subscription_id=subscription_id,
            client=client,
            x_api_elaborate=x_api_elaborate,
            connection=connection,
            upgrade=upgrade,
            sec_web_socket_version=sec_web_socket_version,
            sec_web_socket_key=sec_web_socket_key,
            x_delete_on_close=x_delete_on_close,
        )
    ).parsed
