#!/bin/bash

#
# Start a tranmission. This script is specific to the Powder
# environment and is provided as a example for environments that do
# not use UHD tx_waveforms and need a custom control script.
#

# The smart transmitter sets these instead of using command line args
TXMIN_FREQ=${TXMIN_FREQ}
TXMAX_FREQ=${TXMAX_FREQ}

# Record process pid here so we can kill it when the stop order comes in
PIDFILE="/var/tmp/tx.pid"
# Debugging
impotent=0

function usage
{
   echo "Usage: `basename $0` [--min_freq <hz>] [--max_freq <hz>] {start|stop}" >&2
   exit 1
}

#
# Must get at least one argument, start or stop. 
#
if [ $# -eq 0 ]; then
    usage
fi
if [ ${!#} != "start" -a ${!#} != "stop" ]; then
    usage
fi
   
#
# Allow for command line override.
#
while [[ $# -gt 0 ]]; do
  case $1 in
    --min_freq)
      TXMIN_FREQ="$2"
      shift
      shift
      ;;
    --max_freq)
      TXMAX_FREQ="$2"
      shift
      shift
      ;;
    --impotent)
      impotent=1
      shift
      ;;
    *)
      op="$1"
      break
  esac
done

if [ $op = "stop" ]; then
    if [ ! -f $PIDFILE ]; then
	echo "$PIDFILE does not exist"
	exit 1
    fi
    pid=`/bin/cat $PIDFILE`
    echo "Killing pid $pid"
    kill $pid
    rm -f $PIDFILE
    exit 0
fi

if [ "${TXMIN_FREQ}" = "" ]; then
    echo "Must provide minimum frequency (--min_freq)"
    exit 1
fi
if [ "${TXMAX_FREQ}" = "" ]; then
    echo "Must provide maximum frequency (--max_freq)"
    exit 1
fi

#
# Convert to center frequency and rate,
#
center=$((TXMIN_FREQ + ((TXMAX_FREQ - TXMIN_FREQ) / 2)))
rate=$(((TXMAX_FREQ - TXMIN_FREQ) / 4))
#echo "$center, $rate"

#
# Different versions of UHD
#
if [ -e "/usr/lib/uhd/examples/tx_waveforms" ]; then
    TXWAVEFORMS="/usr/lib/uhd/examples/tx_waveforms"
elif [ -e "/usr/libexec/uhd/examples/tx_waveforms" ]; then
    TXWAVEFORMS="/usr/libexec/uhd/examples/tx_waveforms"
else
    TXWAVEFORMS="tx_waveforms"
fi
TXCMD="$TXWAVEFORMS --ant TX/RX --channel 0 --rate $rate --gain 85 --freq $center"
echo $TXCMD

if [ $impotent -eq 1 ]; then
    TXCMD="/bin/sleep 1000"
    echo $TXCMD
fi

echo $$ > $PIDFILE
exec $TXCMD

