#!/bin/python

import time
import uuid
import os
import sys
import signal
import traceback
import argparse
import logging
import asyncio
import datetime
import base64
import json
from typing import cast, Optional, Dict

from zmsclient.zmc.client import ZmsZmcClient
from zmsclient.dst.client import ZmsDstClient
from zmsclient.dst.v1.models import Observation, Value, Metric, MetricList
from zmsclient.zmc.v1.models import Subscription, EventFilter
from zmsclient.zmc.v1.models import Claim, Grant, Constraint, GrantConstraint
from zmsclient.zmc.v1.models import Spectrum, ClaimList, SpectrumList
from zmsclient.zmc.v1.models import MonitorList
from zmsclient.common.subscription import ZmsSubscriptionCallback
from zmsclient.common import DefaultDestEnvAction

LOG = logging.getLogger(__name__)

# Not sure where to pick these up
EVENT_TYPE_REPLACED    =     1
EVENT_TYPE_CREATED     =     2
EVENT_TYPE_VIOLATION   =     20
EVENT_TYPE_INTERFERENCE=     21
EVENT_SOURCETYPE_ZMC   =     2
EVENT_SOURCETYPE_DST   =     3
EVENT_CODE_GRANT       =     2006
EVENT_CODE_OBSERVATION =     3001
EVENT_CODE_VALUE       =     3011

# Only when a daemon
LOGFILE = "/local/logs/zms-observations-relay.log"

#
# Need a map of outer monitor IDs to inner monitor IDs.
#
def loadMonitorMap(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient,
                   monitor: Optional[str] = None):
    monitorMap = {}
    kwargs = dict()
    if monitor:
        kwargs["monitor"] = monitor
    resp = inner_zmc_client.list_monitors(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MonitorList):
        LOG.error("Could not list inner monitors: %r", resp)
        return {}
    inner_monitors = resp.parsed
    resp = outer_zmc_client.list_monitors(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MonitorList):
        LOG.error("Could not list outer monitors: %r", resp)
        return {}
    outer_monitors = resp.parsed

    for inner_monitor in inner_monitors.monitors:
        if not inner_monitor.enabled:
            continue

        for outer_monitor in outer_monitors.monitors:
            if outer_monitor.id == inner_monitor.device_id:
                monitorMap[outer_monitor.id] = inner_monitor.id;
                break

    LOG.info("Monitor map: %r", monitorMap)
    return monitorMap

#
# Also need a map of inner spectrum IDs to outer grants IDs
#
def loadSpectrumMap(inner_zmc_client: ZmsZmcClient, outer_zmc_client: ZmsZmcClient) -> Dict[str, str]:
    spectrumMap: Dict[str, str] = {}

    resp = inner_zmc_client.list_spectrum(items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, SpectrumList):
        LOG.error("Could not list inner spectrum: %r", resp)
        return {}
    inner_spectrum_list = resp.parsed
    for spectrum in inner_spectrum_list.spectrum:
        if not spectrum.ext_id or spectrum.ext_id == "":
            continue

        #
        # Need to chase back the grant chain in case there was a replacement
        # during the off season. At the moment, just worrying about the
        # expiration (extension) of the most recent grant in the chain. 
        #
        outer_id = spectrum.ext_id
        
        resp = outer_zmc_client.get_grant(outer_id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Grant):
            LOG.error("Could not get grant: %r " % (outer_id,))
            return {}
        grant = cast(Grant, resp.parsed)

        current_outer_id = outer_id
        while grant.replacement:
            current_outer_id = grant.replacement.new_grant_id
            resp = outer_zmc_client.get_grant(current_outer_id, x_api_elaborate="True")
            if not resp.is_success or not isinstance(resp.parsed, Grant):
                LOG.error("Could not get grant: %r ", current_outer_id)
                return {}
            grant = cast(Grant, resp.parsed)

        #
        # if the grant has been replaced, lets update the expiration. 
        #
        if outer_id != current_outer_id:
            LOG.info("Updating spectrum expiration: %r", grant.expires_at)
            spectrum.ext_id = current_outer_id
            spectrum.expires_at = grant.expires_at
            resp = inner_zmc_client.update_spectrum(
                spectrum_id=str(spectrum.id), body=spectrum)
            if not resp.is_success or not isinstance(resp.parsed, SpectrumList):
                LOG.error("Could not update spectrum: %r", resp)
                return {}
            LOG.info("updated spectrum: %r -> %r", outer_id, current_outer_id )
        
        spectrumMap[spectrum.ext_id] = str(spectrum.id)

        #
        # Watch for starting up and being out of sync wrt paused/resumed.
        #
        resp = inner_zmc_client.list_claims(
            element_id=spectrum.element_id, ext_id=spectrum.id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, ClaimList):
            LOG.error("Could not list claims for spectrum %r: %r",
                      spectrum.id, resp)
            return {}
        claimlist = cast(ClaimList, resp.parsed)

        if grant.status != "active" and claimlist.total == 0:
            LOG.info("Creating claim on spectrum: %r", spectrum.id)
            createClaim(inner_zmc_client, str(spectrum.id))
        elif grant.status == "active" and claimlist.total != 0:
            claim = claimlist.claims[0]
            LOG.info("Deleting claim on spectrum: %r", spectrum.id)
            inner_zmc_client.delete_claim(claim_id=str(claim.id))
            pass
        pass
    
    LOG.info("Spectrum map: %r", spectrumMap)
    return spectrumMap

#
# Metrics map (which will also update dynamically) since they change
# more often,
#
def loadMetricMap(inner_dst_client: ZmsDstClient, outer_dst_client: ZmsDstClient,
                  metric: Optional[str] = None) -> Dict[str, str]:
    metricMap: Dict[str, str] = {}
    kwargs = dict()
    if metric:
        kwargs["metric"] = metric
    resp = inner_dst_client.list_metrics(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MetricList):
        LOG.error("Could not list inner metrics: %r", resp)
        return {}
    inner_metrics = cast(MetricList, resp.parsed)
    resp = outer_dst_client.list_metrics(**kwargs, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MetricList):
        LOG.error("Could not list outer metrics: %r", resp)
        return {}
    outer_metrics = cast(MetricList, resp.parsed)

    for inner_metric in inner_metrics.metrics:
        for outer_metric in outer_metrics.metrics:
            if str(outer_metric.id) in str(inner_metric.description):
                metricMap[str(outer_metric.id)] = str(inner_metric.id)
                break

    LOG.info("Metric map: %r", metricMap)
    return metricMap

#
# Load one missing (probably new) Metric and save.
#
def loadNewMetric(inner_dst_client: ZmsDstClient, outer_dst_client: ZmsDstClient,
                  element_id: str, outer_metric_id: str, metric_map: Dict[str, str]) -> Optional[str]:
    resp = outer_dst_client.get_metric(outer_metric_id, x_api_elaborate="True")
    if not resp.is_success or not isinstance(resp.parsed, Metric):
        LOG.error("Could not get outer Metric: %r " % (outer_metric_id,))
        return None
    outer_metric = cast(Metric, resp.parsed)
    outer_metric_id = cast(str, outer_metric.id)

    resp = inner_dst_client.list_metrics(
        element_id=element_id, metric=outer_metric.name, items_per_page=1000)
    if not resp.is_success or not isinstance(resp.parsed, MetricList):
        LOG.error("Could not list inner metrics: %r", resp)
        return None
    inner_metrics = cast(MetricList, resp.parsed)
        
    if inner_metrics and inner_metrics.metrics and len(inner_metrics.metrics):
        inner_metric = inner_metrics.metrics[0]
        inner_metric_id = cast(str, inner_metric.id)
        metric_map[outer_metric_id] = inner_metric_id
        return inner_metric_id

    if outer_metric.description:
        description = outer_metric.description + " (" + outer_metric_id + ")"
    else:
        description = outer_metric_id
        pass
    new = Metric(name=outer_metric.name,
                 element_id=element_id,
                 description=description,
                 unit=outer_metric.unit,
                 schema=outer_metric.schema)
    LOG.debug("Creating new metric: %r", new)

    resp = inner_dst_client.create_metric(body=new)
    if not resp.is_success or not isinstance(resp.parsed, Metric):
        LOG.error("Could not create metric: " + outer_metric_id)
        return None
    inner_metric = cast(Metric, resp.parsed)
    inner_metric_id = cast(str, inner_metric.id)
    LOG.debug("Created new metric: %r", inner_metric)
    metric_map[outer_metric_id] = inner_metric_id
    return inner_metric_id

#
# Create a claim on the entire spectrum object.
#
def createClaim(zmc_client: ZmsZmcClient, spectrum_id: str):
    resp = zmc_client.get_spectrum(spectrum_id, x_api_elaborate="True")    
    if not resp.is_success or not isinstance(resp.parsed, Spectrum):
        LOG.error("Could not get spectrum: %r", resp)
        return
    spectrum = cast(Spectrum, resp.parsed)
    if not spectrum.constraints or len(spectrum.constraints) == 0 \
        or not spectrum.constraints[0].constraint:
        LOG.error("Spectrum has no constraints: %r", spectrum)
        return
    min_freq = spectrum.constraints[0].constraint.min_freq
    max_freq = spectrum.constraints[0].constraint.max_freq
    LOG.info("Creating a claim for %r,%r", min_freq, max_freq)
    
    claim = Claim(
        element_id=spectrum.element_id,
        ext_id=str(spectrum.id),
        type="claim",
        source="rdzinrdz",
        name="Paused",
        description="Paused by outer RDZ",
        grant=Grant(
            element_id=spectrum.element_id,
            spectrum_id=spectrum.id,
            priority=1023,
            name="Paused",
            ext_id=spectrum_id,
            description="Paused by outer RDZ",
            starts_at=datetime.datetime.now(datetime.timezone.utc),
            constraints=[
                GrantConstraint(
                    constraint=Constraint(
                        min_freq=min_freq,
                        max_freq=max_freq,
                        max_eirp=30,
                        exclusive=True
                    )
                )
            ]
        )
    )
    resp = zmc_client.create_claim(body=claim)
    if not resp.is_success or not isinstance(resp.parsed, Claim):
        LOG.error("Could not create new claim: %r", claim)
        return
    claim = cast(Claim, resp.parsed)
    LOG.info("Created claim: %r", claim.id)
    pass

#
# Inject an observation from the parent, into the child.
#
def injectObservation(outer_dst_client: ZmsDstClient, inner_dst_client: ZmsDstClient,
                      id: str, element_id: str,
                      monitor_map: Dict[str, str] = dict(), metric_map: Dict[str, str] = dict(),
                      impotent: bool = False):
    resp = outer_dst_client.get_observation(
        id, data_inline=True, metadata_inline=True)
    if not resp.is_success or not isinstance(resp.parsed, Observation):
        LOG.error("Could not get observation: %r", resp)
        return
    observation = cast(Observation, resp.parsed)
    inner_monitor_id = monitor_map.get(observation.monitor_id, None)
    LOG.info("Observation: %r", id)
    LOG.info("  Outer Monitor: %r", observation.monitor_id)
    LOG.info("  Inner Monitor: %r", inner_monitor_id)
    if not inner_monitor_id:
        LOG.info(" Skipping observation from outer monitor %s: no matching inner monitor", observation.monitor_id)
        return

    #
    # If a sigmf observation then need to process the metadata.
    #
    metadata = observation.metadata
    if observation.format_ == "sigmf" and metadata:
        try:
            decoded = base64.b64decode(metadata).decode('utf-8')
            mdata = json.loads(decoded)
        except Exception as ex:
            LOG.exception(ex)
            return

        if "openzms-core:values" in mdata["global"]:
            for value in mdata["global"]["openzms-core:values"]:
                outer_metric_id = value["metric_id"]
                inner_metric_id = metric_map.get(outer_metric_id, None)
                if not inner_metric_id:
                    LOG.info(" Loading missing metric")
                    inner_metric_id = loadNewMetric(
                        inner_dst_client, outer_dst_client, element_id,
                        outer_metric_id, metric_map)
                    if not inner_metric_id:
                        LOG.error(" Skipping observation: " +
                                  "cannot map outer metric %s to inner metric",
                                  outer_metric_id)

                        return
                    pass
                value["metric_id"] = inner_metric_id
                value["monitor_id"] = inner_monitor_id
                pass

            encoded = base64.b64encode(
                json.dumps(mdata).encode('utf-8')).decode('utf-8')
            metadata = encoded
            pass
        pass

    new = Observation(element_id=element_id,
                      monitor_id=inner_monitor_id,
                      kind=observation.kind,
                      types=observation.types,
                      format_=observation.format_,
                      description=observation.description,
                      min_freq=observation.min_freq,
                      max_freq=observation.max_freq,
                      starts_at=observation.starts_at,
                      violation=observation.violation)

    LOG.debug("Creating observation: %r", new)
    if impotent:
        return

    new.data = observation.data
    new.metadata = metadata
    response = inner_dst_client.create_observation(body=new)
    LOG.debug("created observation: %r", response)

#
# Inject an Value from the parent, into the child.
#
def injectValue(outer_dst_client: ZmsDstClient, inner_dst_client: ZmsDstClient,
                value: Value, element_id: str,
                monitor_map: Dict[str, str] = dict(), metric_map: Dict[str, str] = dict(),
                impotent: bool = False):
    value_metric_id = str(value.metric_id)
    inner_monitor_id = monitor_map.get(value.monitor_id, None)
    inner_metric_id = metric_map.get(value_metric_id, None)
    LOG.info("Value: %r", value.id)
    LOG.info("  Outer Monitor: %r", value.monitor_id)
    LOG.info("  Inner Monitor: %r", inner_monitor_id)
    LOG.info("  Inner Metric: %r", inner_metric_id)
    if not inner_monitor_id:
        LOG.info(" Skipping value from outer monitor %s: no matching inner monitor",
                 value.monitor_id)
        return
    if not inner_metric_id:
        LOG.info(" Loading missing metric")
        inner_metric_id = loadNewMetric(
            inner_dst_client, outer_dst_client, element_id,
            value_metric_id, metric_map)
        if not inner_metric_id:
            LOG.info(" Skipping value from outer metric %s: no matching inner metric",
                     value.metric_id)
            return
        pass

    new = Value(monitor_id=inner_monitor_id,
                metric_id=inner_metric_id,
                value=value.value,
                fields=value.fields,
                tags=value.tags)

    LOG.debug("Creating Value: %r", new)
    if impotent:
        return

    response = inner_dst_client.create_value(body=new)
    LOG.debug("created value: %r", response)
    pass

#
# Watching for several cases;
# 1) Outer grant is paused/resumed; This is converted to a high priority
#    Claim on the entire spectrum,
# 2) Outer grant is replaced, as for extension; update the expires_at of
#    the corresponding spectrum object.
# 3) Outer grant is replaced, because the spectrum range had to be
#    moved to a different place on the dial. Update the spectrum range,
#
def updateSpectrum(outer_zmc_client: ZmsZmcClient, inner_zmc_client: ZmsZmcClient,
                   grant: Grant, element_id: str, spectrum_map: dict,
                   impotent: bool = False):
    old_grant_id = grant.id
    LOG.info("Grant event: %r - %r", old_grant_id, grant.status)

    #
    # Watch for the grant being revoked or paused and disable the local
    # corresponding spectrum object
    #
    if not grant.replacement or not grant.replacement.new_grant_id:
        LOG.info("  Not a replacement")

        if not old_grant_id in spectrum_map:
            LOG.error("  Cannot find the old_grant_id in spectrum_map!")
            LOG.error("  Ignoring grant change")
            return

        # Ignore the "replacing" state, has no meaning here.
        if grant.status == "replacing":
            LOG.info("  Ignoring replacing state")
            return

        # Check for an existing claim on this spectrum.
        spectrum_id = spectrum_map[old_grant_id];
        resp = inner_zmc_client.list_claims(
            element_id=element_id, ext_id=spectrum_id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, ClaimList):
            LOG.error("  Could not list claims for spectrum %r: %r",
                      spectrum_id, resp)
            return
        claimlist = resp.parsed
        if grant.status != "active" and grant.status != "approved" and claimlist.total == 0:
            LOG.info("  Creating claim on spectrum: %r", spectrum_id)
            createClaim(inner_zmc_client, spectrum_id)
        elif grant.status == "active" and claimlist.total != 0:
            claim = claimlist.claims[0]
            LOG.info("  Deleting claim on spectrum: %r", spectrum_id)
            inner_zmc_client.delete_claim(claim_id=str(claim.id))
            pass
        return

    new_grant_id = grant.replacement.new_grant_id
    LOG.info("Grant Replacement: %r -> %r", old_grant_id, new_grant_id)

    #
    # There can be a delay, so lets poll for a bit until it goes
    # active or is denied/deleted.
    #
    newgrant = None
    for x in range(30):
        time.sleep(2)
        resp = outer_zmc_client.get_grant(new_grant_id, x_api_elaborate="True")
        if not resp.is_success or not isinstance(resp.parsed, Grant):
            LOG.error("  Could not get new grant: %r", resp)
            continue
        newgrant = cast(Grant, resp.parsed)
        if newgrant.status in ["active", "denied", "deleted", "pending"]:
            break
        pass

    #
    # We get a new grant (replacement) even if the grant extension is
    # denied. We have to ignore that.
    #
    if grant.status != "active":
        LOG.error("  The replaced grant was not approved!")
        LOG.error("  Ignoring grant change")
        return

    #
    # I have a feeling that we will eventually get out of sync, if we
    # miss an event. 
    #
    if not old_grant_id in spectrum_map:
        LOG.error("  Cannot find the old_grant_id in spectrum_map!")
        LOG.error("  Ignoring grant change")
        return
    
    spectrum_id = spectrum_map[old_grant_id];
    resp = inner_zmc_client.get_spectrum(spectrum_id, x_api_elaborate="True")
    if not resp.is_success or not isinstance(resp.parsed, Spectrum):
        LOG.error("Could not get spectrum: %r", resp)
        return
    spectrum = cast(Spectrum, resp.parsed)
    LOG.info("  Spectrum %r", spectrum_id)
    LOG.info("  Old expiration: %s", spectrum.expires_at or "")
    LOG.info("  New expiration: %s", grant.expires_at)

    if impotent:
        return

    # Update the spectrum for the new expiration and the replacement grant
    spectrum.ext_id = new_grant_id
    spectrum.expires_at = grant.expires_at
    # Lets set the URL to the outer grant page.
    spectrum.url = outer_zmc_client._base_url + "/grants/" + new_grant_id

    resp = inner_zmc_client.update_spectrum(
        spectrum_id=spectrum_id, body=spectrum)
    if not resp.is_success or not isinstance(resp.parsed, Spectrum):
        LOG.error("Could not update spectrum: %r", resp)
        return
    spectrum = cast(Spectrum, resp.parsed)
    LOG.info("  Updated spectrum expiration: %r", spectrum_id)
    LOG.debug("updated spectrum: %r", spectrum)

    #
    # Watch for modified spectrum constraint.
    #
    # XXX There is currently no way to match a grant having multiple
    # constraints, to the corresponding spectrum constraint. But at
    # the moment Powder created grants are single constraint.
    #
    if not spectrum.constraints or len(spectrum.constraints) == 0 \
        or not spectrum.constraints[0].constraint:
        LOG.error("Spectrum has no constraints: %r", spectrum)
        return
    if not grant.constraints or len(grant.constraints) == 0 \
        or not grant.constraints[0].constraint:
        LOG.error("Grant has no constraints: %r", grant)
        return
    spectrum_constraint = spectrum.constraints[0].constraint
    grant_constraint = grant.constraints[0].constraint
    if (grant_constraint.min_freq != spectrum_constraint.min_freq or
        grant_constraint.max_freq != spectrum_constraint.max_freq):
        spectrum_constraint.min_freq = grant_constraint.min_freq
        spectrum_constraint.max_freq = grant_constraint.max_freq

        response = inner_zmc_client.update_spectrum_constraint(
            spectrum_id=spectrum_id, constraint_id=str(spectrum_constraint.id),
            body=spectrum_constraint, x_api_force_update="True")
        LOG.info("  Updated spectrum constraint: %r - %r,%r", spectrum_id,
                 grant_constraint.min_freq, grant_constraint.max_freq)
        LOG.debug("updated spectrum constraint: %r", response)
        pass

    # This is in leu of restarting the subscription and web socket. 
    spectrum_map[new_grant_id] = spectrum_id
    del(spectrum_map[old_grant_id])
    pass

#
# Subclass ZmsSubscriptionCallback, to inject observations from the outer ZMS,
# into the inner ZMS
#
class DSTSubscriptionCallback(ZmsSubscriptionCallback):
    def __init__(self, outer_dstclient: ZmsDstClient, inner_dstclient: ZmsDstClient,
                 inner_element_id: str,
                 monitor_map: dict = dict(), metric_map: dict = dict(),
                 impotent: bool = False, **kwargs):
        super(DSTSubscriptionCallback, self).__init__(outer_dstclient, **kwargs)
        self.outerDST = outer_dstclient
        self.innerDST = inner_dstclient
        self.inner_element_id = inner_element_id
        self.monitor_map = monitor_map
        self.metric_map = metric_map
        self.impotent = impotent

    def on_event(self, ws, evt, message):
        # For now, just observations
        if evt.header.source_type != EVENT_SOURCETYPE_DST:
            LOG.error("on_event: unexpected source type: %r (%r)",
                      evt.header.source_type, message)
            return
        if evt.header.type not in [EVENT_TYPE_CREATED, EVENT_TYPE_VIOLATION, EVENT_TYPE_INTERFERENCE]:
            LOG.error("on_event: unexpected type: %r (%r)",
                      evt.header.type, message)
            return
        if (evt.header.code != EVENT_CODE_OBSERVATION and
            evt.header.code != EVENT_CODE_VALUE):
            LOG.error("on_event: unexpected code: %r (%r)",
                      evt.header.code, message)
            return

        if evt.header.code == EVENT_CODE_OBSERVATION:
            try:
                injectObservation(
                    self.outerDST, self.innerDST, evt.header.object_id,
                    self.inner_element_id, monitor_map=self.monitor_map,
                    metric_map=self.metric_map,
                    impotent=self.impotent)
            except Exception as ex:
                LOG.exception(ex)
                return
        else:
            try:
                injectValue(
                    self.outerDST, self.innerDST, evt.object_,
                    self.inner_element_id, monitor_map=self.monitor_map,
                    metric_map=self.metric_map,
                    impotent=self.impotent)
            except Exception as ex:
                LOG.exception(ex)
                return
        pass

class ZMCSubscriptionCallback(ZmsSubscriptionCallback):
    def __init__(self, outer_zmcclient: ZmsZmcClient, inner_zmcclient: ZmsZmcClient,
                 inner_element_id: str, spectrum_map: Optional[dict] = None,
                 impotent: bool = False, **kwargs):
        super(ZMCSubscriptionCallback, self).__init__(outer_zmcclient, **kwargs)
        self.outerZMC = outer_zmcclient
        self.innerZMC = inner_zmcclient
        self.inner_element_id = inner_element_id
        self.spectrum_map = spectrum_map or dict()
        self.impotent = impotent

    def on_event(self, ws, evt, message):
        if evt.header.source_type != EVENT_SOURCETYPE_ZMC:
            LOG.error("on_event: unexpected source type: %r (%r)",
                      evt.header.source_type, message)
            return

        # Since we are subscribed to all ZMC events for this element,
        # there will be chatter we do not care about.
        if (evt.header.code != EVENT_CODE_GRANT):
            return

        try:
            updateSpectrum(self.outerZMC, self.innerZMC, evt.object_,
                           self.inner_element_id, self.spectrum_map,
                           impotent=self.impotent)
        except Exception as ex:
            LOG.exception(ex)

# The hander has to be outside the async main.
def set_signal_handler(signum, task_to_cancel):
    def handler(_signum, _frame):
        asyncio.get_running_loop().call_soon_threadsafe(task_to_cancel.cancel)
    signal.signal(signum, handler)

def init_main() -> tuple[Optional[DSTSubscriptionCallback], Optional[ZMCSubscriptionCallback], bool, str]:
    parser = argparse.ArgumentParser(
        prog="relayevents",
        description="Relay observation events from a 'parent' OpenZMS to another.")
    parser.add_argument(
        "-b", "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Impotent: do not inject events")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "--element-id", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) element ID")
    parser.add_argument(
        "--element-token", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) element token")
    parser.add_argument(
        "--zmc-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) ZMC URL")
    parser.add_argument(
        "--dst-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Local (inner) DST URL")
    parser.add_argument(
        "--parent-zmc-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) ZMC URL")
    parser.add_argument(
        "--parent-dst-http", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) DST URL")
    parser.add_argument(
        "--parent-element-token", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) element token")
    parser.add_argument(
        "--parent-element-userid", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) user id that is bound to the upstream element token")
    parser.add_argument(
        "--parent-element-id", action=DefaultDestEnvAction, type=str, required=True,
        help="Upstream (outer) token id")
    parser.add_argument(
        "--monitor", default=None, type=str,
        help="Filter by monitor name")
    parser.add_argument(
        "--no-zmc", default=False, action="store_true",
        help="Disable ZMC subscription")
    parser.add_argument(
        "--no-dst", default=False, action="store_true",
        help="Disable DST subscription")

    args = parser.parse_args(sys.argv[1:])

    if args.debug:
        LOG.setLevel(logging.DEBUG)
        logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
        logging.getLogger('zmsclient').setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)

    innerZMC   = ZmsZmcClient(args.zmc_http, args.element_token,
                              raise_on_unexpected_status=True)
    innerDST   = ZmsDstClient(args.dst_http, args.element_token,
                              raise_on_unexpected_status=True)
    outerZMC   = ZmsZmcClient(args.parent_zmc_http, args.parent_element_token,
                              raise_on_unexpected_status=True)
    outerDST   = ZmsDstClient(args.parent_dst_http, args.parent_element_token,
                              raise_on_unexpected_status=True)

    # Map outer monitors to inner monitors
    monitor_map = loadMonitorMap(innerZMC, outerZMC, monitor=args.monitor)
    # Ditto inner spectrum to outer grants
    spectrum_map = loadSpectrumMap(innerZMC, outerZMC)
    # And metrics
    metric_map = loadMetricMap(innerDST, outerDST)

    # And create the subscription (which includes the websocket)
    DSTsubscription = None
    if not args.no_dst:
        DSTsubscription = DSTSubscriptionCallback(
            outerDST, innerDST, args.element_id,
            subscription=Subscription(id=str(uuid.uuid4())),
            monitor_map=monitor_map, metric_map=metric_map,
            impotent=args.impotent, reconnect_on_error=True)

    ZMCsubscription = None
    if not args.no_zmc:
        ZMCsubscription = ZMCSubscriptionCallback(
            outerZMC, innerZMC, args.element_id,
            subscription=Subscription(
                id=str(uuid.uuid4()),
                filters=[EventFilter(element_ids=[args.parent_element_id],
                                     user_ids=[args.parent_element_userid])]),
            spectrum_map=spectrum_map,
            impotent=args.impotent, reconnect_on_error=True)

    return DSTsubscription, ZMCsubscription, args.daemon, args.logfile

async def async_main(*args):
    this_task = asyncio.current_task();
    set_signal_handler(signal.SIGINT, this_task)
    set_signal_handler(signal.SIGHUP, this_task)
    set_signal_handler(signal.SIGTERM, this_task)

    try:
        runnable = [sub.run_callbacks() for sub in args]
        await asyncio.gather(*runnable)
    except asyncio.CancelledError:
        for sub in args:
            if sub.id:
                sub.unsubscribe()

#
# The reason for this split of main, is cause we need to do the daemon
# fork() before heading into the asyncio main.
#
def main():
    DSTsubscription, ZMCsubscription, daemonize, logfile = init_main()
    format = "%(levelname)s:%(asctime)s:%(message)s"

    if daemonize:
        try:
            fp = open(logfile, "a");
            sys.stdout = fp
            sys.stderr = fp
            sys.stdin.close();
            logging.basicConfig(stream=fp, format=format)
            pass
        except:
            print("Could not open log file for append")
            sys.exit(1);
            pass
        pid = os.fork()
        if pid:
            sys.exit(0)
        os.setsid();
    else:
        logging.basicConfig(format=format)

    subs = []
    if DSTsubscription:
        subs.append(DSTsubscription)
    if ZMCsubscription:
        subs.append(ZMCsubscription)
    if not subs:
        sys.exit(0)
    asyncio.run(async_main(*subs))

if __name__ == "__main__":
    main()
