import abc
import argparse
import copy
import logging
from typing import (List, Optional, Union)

from zmsclient.dst.client import ZmsDstClient
from zmsclient.dst.v1.models import Error as DstError
from zmsclient.dst.v1.models import (Metric, MetricList, Value, Series)

LOG = logging.getLogger(__name__)
    
class MetricsHelper:
    """A metrics API helper class."""
    def __init__(self, client: ZmsDstClient, monitor_id: str, element_id: str,
                 impotent: bool = False):
        self.client = client
        self.metrics = dict()  # metric_name -> Metric
        self.series = dict()   # metric_name -> Series
        self.monitor_id = monitor_id
        self.element_id = element_id
        self.impotent = impotent

    def upload_value(self, v: Value) -> Union[DstError, Value]:
        v.monitor_id = self.monitor_id
        if self.impotent:
            LOG.info("Impotent mode: not uploading value: %r", v)
            return v
        try:
            resp = self.client.create_value(body=v)
            if not resp.is_success or not isinstance(resp.parsed, Value):
                LOG.error("Error creating value: %s", resp)
                return resp.parsed
            else:
                LOG.debug("created value: %r", resp.parsed)
            return resp.parsed
        except Exception as e:
            LOG.error("Could not create value %r: %r", v, e)
            raise e

    def get_metric(self, metric_name: str) -> Optional[Metric]:
        return self.metrics.get(metric_name, None)
    
    def get_series(self, metric_name: str) -> Optional[Series]:
        return self.series.get(metric_name, None)

    def _init_metric(self, metric: Metric, create_if_missing: bool = True) -> Optional[Union[Metric,DstError]]:
        # Look for a public metric with this name first.
        try:
            resp = self.client.list_metrics(metric=metric.name, is_public=True)
            if isinstance(resp.parsed, MetricList) and len(resp.parsed.metrics) > 0:
                return resp.parsed.metrics[0]
            else:
                LOG.debug("No public metric found for %s (%s)", metric.name, resp)
        except Exception as e:
            LOG.error("Could not list metric %s: %s", metric.name, e)
            raise e
        # Look for a per-element metric next.
        try:
            resp = self.client.list_metrics(metric=metric.name, element_id=self.element_id)
            if isinstance(resp.parsed, MetricList) and len(resp.parsed.metrics) > 0:
                return resp.parsed.metrics[0]
            else:
                LOG.debug("No private metric found for %s (%s)", metric.name, resp)
        except Exception as e:
            LOG.error("Could not list metric %s: %s", metric.name, e)
            raise e
        if create_if_missing:
            # Create a per-element metric.
            try:
                metric.element_id = self.element_id
                LOG.info("Creating metric %s: %s", metric.name, metric)
                resp = self.client.create_metric(body=metric)
                return resp.parsed
            except Exception as e:
                LOG.error("Could not create metric %s: %s", metric.name, e)
                raise e
        return None

    def init_metrics(self, metrics: List[Metric], create_if_missing: bool = True):
        """Load (or create) the metrics we need in the DST."""
        for m in metrics:
            LOG.debug("Metric definition: %s", m)
            # NB: just do a shallow copy; we will be changing the element_id
            # and possibly other fields.  We cannot do a deepcopy because the
            # generated models check to see if fields are UNSET when JSONifying,
            # and a deepcopy will change UNSET to a new Unset instance!
            nm = copy.copy(m)
            nm.element_id = self.element_id
            if self.impotent:
                LOG.info("Impotent mode: not initializing metric %s: %s", m.name, nm)
                self.metrics[m.name] = nm
            else:
                ret = self._init_metric(nm, create_if_missing)
                if isinstance(ret, Metric):
                    LOG.debug("Initialized metric %s: %s", nm.name, ret)
                    self.metrics[nm.name] = ret
                else:
                    LOG.error("Could not initialize metric %s: %s", nm.name, ret)
                    raise RuntimeError(f"Could not initialize metric {nm.name}: {ret}")
    
    def _create_series(self, series: Series) -> Union[Series, DstError]:
        """Create a new series in the DST."""
        if self.impotent:
            LOG.info("Impotent mode: not creating series: %r", series)
            return series
        try:
            response = self.client.create_series(body=series)
            LOG.debug("created series: %r", response)
            return response.parsed
        except Exception as e:
            LOG.error("Could not create series %r: %r", series, e)
            raise e
    
    def fetch_series(self, series_id: str) -> Union[Series, DstError]:
        """Get an existing series in the DST."""
        if self.impotent:
            LOG.info("Impotent mode: not getting series: %r", series_id)
            return None
        try:
            response = self.client.get_series(series_id=series_id)
            LOG.debug("fetched series: %r", response)
            return response.parsed
        except Exception as e:
            LOG.error("Could not fetch series %r: %r", series_id, e)
            raise e
                
    def init_series(self, name_prefix: str = "", description_prefix: str = "",
                    create_if_missing: bool = True):
        """Create a series for each metric we need in the DST."""
        for mname, metric in self.metrics.items():
            s = Series(
                metric_id=metric.id,
                monitor_id=self.monitor_id,
                element_id=self.element_id)
            if name_prefix:
                s.name = name_prefix
                if len(self.metrics) > 1:
                    s.name += " (" + mname + ")"
            if description_prefix:
                s.description = description_prefix
                if len(self.metrics) > 1:
                    s.description += " (" + mname + ")"
            if self.impotent:
                LOG.info("Impotent mode: not creating series for metric %s: %s", mname, s)
                self.series[mname] = s
            else:
                ret = self._create_series(s)
                if isinstance(ret, Series):
                    LOG.debug("Created series for metric %s: %s", mname, ret)
                    self.series[mname] = ret
                else:
                    LOG.error("Could not create series for metric %s: %s", mname, ret)
                    raise RuntimeError(f"Could not create series for metric {mname}: {ret}")

class BaseDataType(abc.ABC):
    """Base class for data types to be uploaded."""
    name: str = "datatype"

    def __init__(self, args: argparse.Namespace, mh: MetricsHelper):
        """Initialize the data type with the given arguments."""
        self.args = args
        self.mh = mh
        self.values = []  # type: List[Value]

    @classmethod
    @abc.abstractmethod
    def add_args(cls, parser: argparse.ArgumentParser):
        """Add any additional arguments needed for this data type."""
        pass

    @classmethod
    @abc.abstractmethod
    def get_metric_definitions(cls) -> List[Metric]:
        """Define the metrics that will be collected."""
        raise NotImplementedError("Subclasses must implement the get_metric_definitions method")

    @abc.abstractmethod
    def load_data(self):
        """Parse the given file."""
        raise NotImplementedError("Subclasses must implement the load_data method")
    
    @abc.abstractmethod
    def prepare_values(self):
        """Prepare the values to be uploaded."""
        raise NotImplementedError("Subclasses must implement the prepare_values method")
    
    def init_metrics(self, create_if_missing: bool = True):
        """Initialize the metrics in the DST."""
        mlist = self.get_metric_definitions()
        self.mh.init_metrics(mlist, create_if_missing=True)
    
    def init_series(self, create_if_missing: bool = True):
        """Initialize the series in the DST."""
        self.mh.init_series(
            self.args.series_name, self.args.series_description,
            create_if_missing=True)

    def upload_data(self) -> Optional[DstError]:
        """Upload the parsed data."""
        if not self.values or len(self.values) == 0:
            LOG.warning("No values to upload")
            return
        for v in self.values:
            err = self.mh.upload_value(v)
            if isinstance(err, DstError):
                if self.args.continue_on_error:
                    LOG.warning("Error uploading value %s: %s", v, err)
                else:
                    LOG.error("Error uploading value %s: %s", v, err)
                    return err
            else:
                LOG.info("Uploaded value: %s", err)
        return None
