import datetime
from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.element_list import ElementList
from ...models.error import Error
from ...types import UNSET, Response, Unset


def _get_kwargs(
    user_id: str,
    *,
    approved: Union[Unset, bool] = UNSET,
    deleted: Union[Unset, bool] = UNSET,
    include_deleted: Union[Unset, bool] = UNSET,
    include_deleted_after: Union[Unset, datetime.datetime] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    params: Dict[str, Any] = {}

    params["approved"] = approved

    params["deleted"] = deleted

    params["include_deleted"] = include_deleted

    json_include_deleted_after: Union[Unset, str] = UNSET
    if not isinstance(include_deleted_after, Unset):
        json_include_deleted_after = include_deleted_after.isoformat()
    params["include_deleted_after"] = json_include_deleted_after

    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    _kwargs: Dict[str, Any] = {
        "method": "get",
        "url": "/users/{user_id}/elements".format(
            user_id=user_id,
        ),
        "params": params,
    }

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[ElementList, Error]]:
    if response.status_code == HTTPStatus.OK:
        try:
            response_200 = ElementList.from_dict(response.json())

            return response_200
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[ElementList, Error]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    client: AuthenticatedClient,
    user_id: str,
    *,
    approved: Union[Unset, bool] = UNSET,
    deleted: Union[Unset, bool] = UNSET,
    include_deleted: Union[Unset, bool] = UNSET,
    include_deleted_after: Union[Unset, datetime.datetime] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[ElementList, Error]]:
    """Retrieve a list of elements in which this user has roles.

    Args:
        user_id (str): The target user.
        approved (Union[Unset, bool]): If omitted or `true`, show only approved element
            rolebindings; if `false`, show unapproved element rolebindings.
        deleted (Union[Unset, bool]): If `true`, show only deleted element rolebindings.
        include_deleted (Union[Unset, bool]):
        include_deleted_after (Union[Unset, datetime.datetime]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[ElementList, Error]]
    """

    kwargs = _get_kwargs(
        user_id=user_id,
        approved=approved,
        deleted=deleted,
        include_deleted=include_deleted,
        include_deleted_after=include_deleted_after,
        x_api_elaborate=x_api_elaborate,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    client: AuthenticatedClient,
    user_id: str,
    *,
    approved: Union[Unset, bool] = UNSET,
    deleted: Union[Unset, bool] = UNSET,
    include_deleted: Union[Unset, bool] = UNSET,
    include_deleted_after: Union[Unset, datetime.datetime] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[ElementList, Error]]:
    """Retrieve a list of elements in which this user has roles.

    Args:
        user_id (str): The target user.
        approved (Union[Unset, bool]): If omitted or `true`, show only approved element
            rolebindings; if `false`, show unapproved element rolebindings.
        deleted (Union[Unset, bool]): If `true`, show only deleted element rolebindings.
        include_deleted (Union[Unset, bool]):
        include_deleted_after (Union[Unset, datetime.datetime]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[ElementList, Error]
    """

    return sync_detailed(
        user_id=user_id,
        client=client,
        approved=approved,
        deleted=deleted,
        include_deleted=include_deleted,
        include_deleted_after=include_deleted_after,
        x_api_elaborate=x_api_elaborate,
    ).parsed


async def asyncio_detailed(
    client: AuthenticatedClient,
    user_id: str,
    *,
    approved: Union[Unset, bool] = UNSET,
    deleted: Union[Unset, bool] = UNSET,
    include_deleted: Union[Unset, bool] = UNSET,
    include_deleted_after: Union[Unset, datetime.datetime] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[ElementList, Error]]:
    """Retrieve a list of elements in which this user has roles.

    Args:
        user_id (str): The target user.
        approved (Union[Unset, bool]): If omitted or `true`, show only approved element
            rolebindings; if `false`, show unapproved element rolebindings.
        deleted (Union[Unset, bool]): If `true`, show only deleted element rolebindings.
        include_deleted (Union[Unset, bool]):
        include_deleted_after (Union[Unset, datetime.datetime]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[ElementList, Error]]
    """

    kwargs = _get_kwargs(
        user_id=user_id,
        approved=approved,
        deleted=deleted,
        include_deleted=include_deleted,
        include_deleted_after=include_deleted_after,
        x_api_elaborate=x_api_elaborate,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    client: AuthenticatedClient,
    user_id: str,
    *,
    approved: Union[Unset, bool] = UNSET,
    deleted: Union[Unset, bool] = UNSET,
    include_deleted: Union[Unset, bool] = UNSET,
    include_deleted_after: Union[Unset, datetime.datetime] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[ElementList, Error]]:
    """Retrieve a list of elements in which this user has roles.

    Args:
        user_id (str): The target user.
        approved (Union[Unset, bool]): If omitted or `true`, show only approved element
            rolebindings; if `false`, show unapproved element rolebindings.
        deleted (Union[Unset, bool]): If `true`, show only deleted element rolebindings.
        include_deleted (Union[Unset, bool]):
        include_deleted_after (Union[Unset, datetime.datetime]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[ElementList, Error]
    """

    return (
        await asyncio_detailed(
            user_id=user_id,
            client=client,
            approved=approved,
            deleted=deleted,
            include_deleted=include_deleted,
            include_deleted_after=include_deleted_after,
            x_api_elaborate=x_api_elaborate,
        )
    ).parsed
