import datetime
from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.influx_d_bv_1_query_response import InfluxDBv1QueryResponse
from ...types import UNSET, Response, Unset


def _get_kwargs(
    series_id: str,
    *,
    from_: Union[Unset, datetime.datetime] = UNSET,
    to: Union[Unset, datetime.datetime] = UNSET,
    q: Union[Unset, str] = UNSET,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    params: Dict[str, Any] = {}

    json_from_: Union[Unset, str] = UNSET
    if not isinstance(from_, Unset):
        json_from_ = from_.isoformat()
    params["from"] = json_from_

    json_to: Union[Unset, str] = UNSET
    if not isinstance(to, Unset):
        json_to = to.isoformat()
    params["to"] = json_to

    params["q"] = q

    params["page"] = page

    params["items_per_page"] = items_per_page

    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    _kwargs: Dict[str, Any] = {
        "method": "get",
        "url": "/series/{series_id}/values/influx".format(
            series_id=series_id,
        ),
        "params": params,
    }

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, InfluxDBv1QueryResponse]]:
    if response.status_code == HTTPStatus.OK:
        try:
            response_200 = InfluxDBv1QueryResponse.from_dict(response.text)

            return response_200
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, InfluxDBv1QueryResponse]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    client: AuthenticatedClient,
    series_id: str,
    *,
    from_: Union[Unset, datetime.datetime] = UNSET,
    to: Union[Unset, datetime.datetime] = UNSET,
    q: Union[Unset, str] = UNSET,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, InfluxDBv1QueryResponse]]:
    """Retrieve a list of values for a specific series in InfluxDB v1 JSON query protocol format.  The
    series type must be `influxdb`, otherwise an error is returned.

    Args:
        series_id (str): The target series.
        from_ (Union[Unset, datetime.datetime]): Filters values from this timestamp (inclusive).
        to (Union[Unset, datetime.datetime]): Filters values up to this timestamp (exclusive).
        q (Union[Unset, str]): An InfluxDB v1 query string to filter the results.  See
            https://docs.influxdata.com/influxdb/v1/tools/api/#query-http-endpoint for more
            information.  Only the read-only GET form is supported (e.g. no `INTO` clause).
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, InfluxDBv1QueryResponse]]
    """

    kwargs = _get_kwargs(
        series_id=series_id,
        from_=from_,
        to=to,
        q=q,
        page=page,
        items_per_page=items_per_page,
        x_api_elaborate=x_api_elaborate,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    client: AuthenticatedClient,
    series_id: str,
    *,
    from_: Union[Unset, datetime.datetime] = UNSET,
    to: Union[Unset, datetime.datetime] = UNSET,
    q: Union[Unset, str] = UNSET,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, InfluxDBv1QueryResponse]]:
    """Retrieve a list of values for a specific series in InfluxDB v1 JSON query protocol format.  The
    series type must be `influxdb`, otherwise an error is returned.

    Args:
        series_id (str): The target series.
        from_ (Union[Unset, datetime.datetime]): Filters values from this timestamp (inclusive).
        to (Union[Unset, datetime.datetime]): Filters values up to this timestamp (exclusive).
        q (Union[Unset, str]): An InfluxDB v1 query string to filter the results.  See
            https://docs.influxdata.com/influxdb/v1/tools/api/#query-http-endpoint for more
            information.  Only the read-only GET form is supported (e.g. no `INTO` clause).
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, InfluxDBv1QueryResponse]
    """

    return sync_detailed(
        series_id=series_id,
        client=client,
        from_=from_,
        to=to,
        q=q,
        page=page,
        items_per_page=items_per_page,
        x_api_elaborate=x_api_elaborate,
    ).parsed


async def asyncio_detailed(
    client: AuthenticatedClient,
    series_id: str,
    *,
    from_: Union[Unset, datetime.datetime] = UNSET,
    to: Union[Unset, datetime.datetime] = UNSET,
    q: Union[Unset, str] = UNSET,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, InfluxDBv1QueryResponse]]:
    """Retrieve a list of values for a specific series in InfluxDB v1 JSON query protocol format.  The
    series type must be `influxdb`, otherwise an error is returned.

    Args:
        series_id (str): The target series.
        from_ (Union[Unset, datetime.datetime]): Filters values from this timestamp (inclusive).
        to (Union[Unset, datetime.datetime]): Filters values up to this timestamp (exclusive).
        q (Union[Unset, str]): An InfluxDB v1 query string to filter the results.  See
            https://docs.influxdata.com/influxdb/v1/tools/api/#query-http-endpoint for more
            information.  Only the read-only GET form is supported (e.g. no `INTO` clause).
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, InfluxDBv1QueryResponse]]
    """

    kwargs = _get_kwargs(
        series_id=series_id,
        from_=from_,
        to=to,
        q=q,
        page=page,
        items_per_page=items_per_page,
        x_api_elaborate=x_api_elaborate,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    client: AuthenticatedClient,
    series_id: str,
    *,
    from_: Union[Unset, datetime.datetime] = UNSET,
    to: Union[Unset, datetime.datetime] = UNSET,
    q: Union[Unset, str] = UNSET,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, InfluxDBv1QueryResponse]]:
    """Retrieve a list of values for a specific series in InfluxDB v1 JSON query protocol format.  The
    series type must be `influxdb`, otherwise an error is returned.

    Args:
        series_id (str): The target series.
        from_ (Union[Unset, datetime.datetime]): Filters values from this timestamp (inclusive).
        to (Union[Unset, datetime.datetime]): Filters values up to this timestamp (exclusive).
        q (Union[Unset, str]): An InfluxDB v1 query string to filter the results.  See
            https://docs.influxdata.com/influxdb/v1/tools/api/#query-http-endpoint for more
            information.  Only the read-only GET form is supported (e.g. no `INTO` clause).
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, InfluxDBv1QueryResponse]
    """

    return (
        await asyncio_detailed(
            series_id=series_id,
            client=client,
            from_=from_,
            to=to,
            q=q,
            page=page,
            items_per_page=items_per_page,
            x_api_elaborate=x_api_elaborate,
        )
    ).parsed
