import hashlib
import secrets
import binascii

#
# OpenZMS token generation helper routines:
#
#   token = prefix + "_" + jxskiss/base62(sha256(32) + uvarint(crc32(4)))
#
# where jxskiss/base62 encodes as in b62_encode_compact, as below;
# and unsigned varint encoding is as specified in
# https://pkg.go.dev/encoding/binary#PutUvarint which points to
# https://protobuf.dev/programming-guides/encoding/
#

#
# Encode an unsigned integer as a Protobuf Uvarint.
#
def varint_encode(x):
    enc = bytearray()
    while True:
        byte = x & 0x7f
        x >>= 7
        if x == 0:
            enc.append(byte)
            break
        enc.append(byte | 0x80)
    return bytes(enc)

#
# Encode a bytes array in base62 using a customized, optimal encoding scheme.
#
# See https://github.com/jxskiss/base62 ; this is not standard base62:
#
#  "We define a compactMask as 0b_0001_1110.
#   When encoding, for each 6 bits integer x, if x & compactMask == compactMask
#   is true, it must be one of 30, 31, 62 or 63, we just encode the lower 5
#   bits, which are 0b_0001_1110 (30) and 0b_0001_1111 (31), we leave the 6-th
#   bit to next byte."
#
# What follows is a port of the v2 encoding goo.
#
CONV = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"
COMPACT_MASK = 0x1e
def b62_encode_compact(data):
    if len(data) == 0 or data[0] == 0:
        return CONV[0]
    pos = len(data) * 8
    def next6bits():
        r = pos & 0x7
        i = pos >> 3
        if r == 0:
            i -= 1
            r = 8
        b = data[i] >> (8 - r)
        if r < 6 and i > 0:
            b |= data[i-1] << r
        return b & 0x3f
    a = []
    while pos > 0:
        x = next6bits()
        ssize = 6
        if x & COMPACT_MASK == COMPACT_MASK:
            a.append(CONV[x & 0x1f])
            if pos > 6 or x > 0x1f:
                ssize = 5
        else:
            a.append(CONV[x])
        pos -= ssize
    return "".join(a)

#
# Generate a token using the helpers.
#
def generate_token(token_type="rpu"):
    b = secrets.token_bytes(nbytes=32)
    h = hashlib.sha256(b)
    d = h.digest()
    s = binascii.crc32(d) & 0xffffffff
    v = varint_encode(s)
    b62 = b62_encode_compact(d + v)
    return token_type + "_" + b62

def main():
    import argparse
    import sys

    parser = argparse.ArgumentParser(
        prog=sys.argv[0],
        description="OpenZMS token utility.")
    subparsers = parser.add_subparsers(
        dest="subcommand", help="Available subcommands")

    subparser_generate = subparsers.add_parser(
        "generate", help="Generate an OpenZMS token")
    subparser_generate.add_argument(
        "-t", "--token-type", default="rpp", choices=["rpp", "rpu", "rps"],
        help="The token type to generate: 'rpu' (user-login), 'rpp' (personal access token), 'rps' (service token)")
    subparser_generate.set_defaults(
        func=lambda args: print(generate_token(token_type=args.token_type)))

    args = parser.parse_args()
    if hasattr(args, 'func'):
        args.func(args)
        exit(0)
    else:
        parser.print_help()
        exit(-1)

if __name__ == "__main__":
    main()
