import argparse
import os


class DefaultDestEnvAction(argparse.Action):
    def __init__(self, option_strings, dest, required=True, default=None, **kwargs):
        """An argparse.Action that initializes the default value of the arg from an environment variable named `dest.upper()` (where dest is the storage location of the value post-parse, e.g. `args.dest`); and, if the arg was required, *unsets* it from being required, so that argparse does not fail the parse if the argument is not supplied.  This is certainly a bit unfortunate since it changes the helptext behavior, but nothing to do about that."""
        dest_upper = dest.upper()
        if dest_upper in os.environ:
            default = os.environ[dest_upper]
        if required and default:
            required = False
        super(DefaultDestEnvAction, self).__init__(
            option_strings, dest, default=default, required=required, **kwargs)

    def __call__(self, parser, namespace, values, option_string=None):
        setattr(namespace, self.dest, values)

class FreqRangeDesc:
    def __init__(self, min: float, max: float, power: float,
                 power_factor: float | None = None, grant_id: str | None = None,
                 lat: float | None = None, lon: float | None = None,
                 dist: float | None = None, enabled: bool = True):
        self.min = min
        self.max = max
        self.power = power
        self.power_factor = power_factor
        self.grant_id = grant_id
        self.lat = lat
        self.lon = lon
        self.dist = dist
        self.enabled = enabled

    @classmethod
    def from_string(cls, s):
        sa = s.split(",")
        if len(sa) != 4:
            raise ValueError("invalid frequency range descriptor")
        return FreqRangeDesc(*[float(x) for x in sa[0:3]], power_factor=float(sa[3]))

    def __repr__(self):
        return (
            f"FreqRangeDesc(min={self.min},max={self.max},power={self.power},"
            f"power_factor={self.power_factor},grant_id={self.grant_id},"
            f"lat={self.lat},lon={self.lon},dist={self.dist},"
            f"enabled={self.enabled})")

    def violate(self, shift: float, gain: float) -> 'FreqRangeDesc':
        return FreqRangeDesc(
            self.min + shift, self.max + shift, self.power + gain,
            power_factor=self.power_factor, grant_id=self.grant_id,
            lat=self.lat, lon=self.lon, dist=self.dist, enabled=self.enabled)
