from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.grant_alarm_log_list import GrantAlarmLogList
from ...types import UNSET, Response, Unset


def _get_kwargs(
    alarm_grant_id: str,
    *,
    limit: Union[Unset, int] = 100,
) -> Dict[str, Any]:
    params: Dict[str, Any] = {}

    params["limit"] = limit

    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    _kwargs: Dict[str, Any] = {
        "method": "get",
        "url": "/alarms/grants/{alarm_grant_id}/logs".format(
            alarm_grant_id=alarm_grant_id,
        ),
        "params": params,
    }

    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, GrantAlarmLogList]]:
    if response.status_code == HTTPStatus.OK:
        try:
            response_200 = GrantAlarmLogList.from_dict(response.json())

            return response_200
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, GrantAlarmLogList]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    client: AuthenticatedClient,
    alarm_grant_id: str,
    *,
    limit: Union[Unset, int] = 100,
) -> Response[Union[Error, GrantAlarmLogList]]:
    """Retrieve a list of recent alarm log entries for a specific grant alarm; admin only.

    Args:
        alarm_grant_id (str): The target alarm grant.
        limit (Union[Unset, int]): The maximum number of log entries to return. Default: 100.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, GrantAlarmLogList]]
    """

    kwargs = _get_kwargs(
        alarm_grant_id=alarm_grant_id,
        limit=limit,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    client: AuthenticatedClient,
    alarm_grant_id: str,
    *,
    limit: Union[Unset, int] = 100,
) -> Optional[Union[Error, GrantAlarmLogList]]:
    """Retrieve a list of recent alarm log entries for a specific grant alarm; admin only.

    Args:
        alarm_grant_id (str): The target alarm grant.
        limit (Union[Unset, int]): The maximum number of log entries to return. Default: 100.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, GrantAlarmLogList]
    """

    return sync_detailed(
        alarm_grant_id=alarm_grant_id,
        client=client,
        limit=limit,
    ).parsed


async def asyncio_detailed(
    client: AuthenticatedClient,
    alarm_grant_id: str,
    *,
    limit: Union[Unset, int] = 100,
) -> Response[Union[Error, GrantAlarmLogList]]:
    """Retrieve a list of recent alarm log entries for a specific grant alarm; admin only.

    Args:
        alarm_grant_id (str): The target alarm grant.
        limit (Union[Unset, int]): The maximum number of log entries to return. Default: 100.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, GrantAlarmLogList]]
    """

    kwargs = _get_kwargs(
        alarm_grant_id=alarm_grant_id,
        limit=limit,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    client: AuthenticatedClient,
    alarm_grant_id: str,
    *,
    limit: Union[Unset, int] = 100,
) -> Optional[Union[Error, GrantAlarmLogList]]:
    """Retrieve a list of recent alarm log entries for a specific grant alarm; admin only.

    Args:
        alarm_grant_id (str): The target alarm grant.
        limit (Union[Unset, int]): The maximum number of log entries to return. Default: 100.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, GrantAlarmLogList]
    """

    return (
        await asyncio_detailed(
            alarm_grant_id=alarm_grant_id,
            client=client,
            limit=limit,
        )
    ).parsed
