import argparse
import logging
import datetime
import os
import uuid
from typing import (List, Dict)

from zmsclient.dst.v1.models import (Metric, Value, AnyObject)

from . import ModemBase

LOG = logging.getLogger(__name__)

class SierraModem(ModemBase):
    name: str = "sierra"
    metric_defn: Metric = Metric(
        name="powder.ue-lte-cell.v2",
        element_id=uuid.UUID(int=0),
        description="POWDER LTE UE Cell Metrics v2",
        schema=AnyObject.from_dict({
            "$id": "https://gitlab.flux.utah.edu/openzms/zms-dst-schemas/schemas/metrics/powder/ue-lte-cell/v2/schema.json",
            "type": "object",
            "title": "POWDER LTE UE Cell Metrics",
            "$schema": "https://json-schema.org/draft/2020-12/schema",
            "x-tag-filter": {
                "cell": "serving"
            },
            "x-label-template": "pci=${tag_pci}",
            "required": [
                "rssi",
                "rsrp",
                "rsrq",
            ],
            "properties": {
                "mcc": {
                    "type": "integer",
                    "description": "Mobile country code.",
                    "x-tag": True
                },
                "mnc": {
                    "type": "integer",
                    "description": "Mobile network code.",
                    "x-tag": True
                },
                "tac": {
                    "type": "integer",
                    "description": "Tracking area code.",
                    "x-tag": True
                },
                "cid": {
                    "type": "integer",
                    "description": "Cell Global ID.",
                    "x-tag": True
                },
                "cell": {
                    "type": "string",
                    "enum": ["serving", "neighbor"],
                    "description": "Type of cell: serving or neighbor.",
                    "x-tag": True
                },
                "pci": {
                    "type": "integer",
                    "description": "Physical cell ID.",
                    "x-tag": True
                },
                "earfcn": {
                    "type": "integer",
                    "description": "E-UTRA Absolute Radio Frequency Channel Number.",
                    "x-tag": True
                },
                "band": {
                    "type": "integer",
                    "description": "E-UTRA band.",
                    "x-tag": True
                },
                "sinr": {
                    "type": "number",
                    "description": "Signal to interference plus noise ratio.",
                    "x-unit": "dB",
                    "x-group": "relative"
                },
                "snr": {
                    "type": "number",
                    "description": "Signal to noise ratio.",
                    "x-unit": "dB",
                    "x-group": "relative"
                },
                "rssi": {
                    "type": "number",
                    "maximum": 0,
                    "description": "Received signal strength indicator.",
                    "x-unit": "dBm",
                    "x-group": "power"
                },
                "rsrp": {
                    "type": "number",
                    "maximum": 0,
                    "description": "Received signal received power.",
                    "x-unit": "dBm",
                    "x-group": "power"
                },
                "rsrq": {
                    "type": "number",
                    "maximum": 0,
                    "description": "Received signal received quality.",
                    "x-unit": "dB",
                    "x-group": "relative"
                },
                "rssnr": {
                    "type": "number",
                    "description": "Reference signal strength to noise ratio.",
                    "x-unit": "dB",
                    "x-group": "relative"
                },
                "latitude": {
                    "type": "number",
                    "description": "Latitude.",
                    "x-unit": "degrees",
                    "x-group": "geospatial"
                },
                "longitude": {
                    "type": "number",
                    "description": "Longitude.",
                    "x-unit": "degrees",
                    "x-group": "geospatial"
                },
                "altitude": {
                    "type": "number",
                    "description": "Altitude.",
                    "x-unit": "meters",
                    "x-group": "geospatial"
                },
                "speed": {
                    "type": "number",
                    "description": "Speed.",
                    "x-unit": "meters/second",
                    "x-group": "geospatial"
                },
                "heading": {
                    "type": "number",
                    "description": "Heading.",
                    "x-unit": "degrees",
                    "x-group": "geospatial"
                },
                "dop": {
                    "type": "number",
                    "description": "Dilution of precision.",
                    "x-group": "geospatial"
                }
            }
        })
    )

    def __init__(self, args: argparse.Namespace):
        super(SierraModem, self).__init__(args)
        self._last_file_mtime = 0

    @classmethod
    def add_args(kls, parser: argparse.ArgumentParser):
        return

    def init(self):
        """Initialize the modem, if necessary."""
        super(SierraModem, self).init()
        if self.file is None:
            raise ValueError("SierraModem requires --file argument")
        LOG.info("Using Sierra modem data file %s", self.file)
        try:
            f = open(self.file, "r")
            f.close()
        except Exception as e:
            LOG.error("Could not open Sierra modem data file %s: %s", self.file, e)
            raise e

    def get_metric_definitions(self) -> List[Metric]:
        return [self.__class__.metric_defn]
    
    #
    # Parse the output of AT!LTEINFO? command.  For instance:
    #
    # !LTEINFO:
    # 
    # Serving:   EARFCN MCC MNC   TAC      CID Bd D U SNR PCI  RSRQ   RSRP   RSSI RXLV
    #
    #              1100 310 410 39179 06E21B08  2 5 5   0 218 -13.8  -95.7  -59.3 --
    #
    # IntraFreq:                                          PCI  RSRQ   RSRP   RSSI RXLV
    #
    #                                                     218 -13.8  -95.7  -59.3 --
    #
    #                                                     503 -20.0 -101.5  -69.7 --
    #
    #                                                     500 -18.3  -99.1  -70.5 --
    #
    #                                                     252 -20.0  -99.6  -69.8 --
    #
    #                                                      99 -16.7  -96.1  -70.5 --
    #
    def _parse_lteinfo(self, lines: List[str]) -> List[Dict]:
        values = []
        state = "start"
        # We record the serving cell's PCI because serving
        # cells are also listed in the neighbor list.
        serving_pci = None
        mcc = mnc = tac = cid = band = earfcn = sinr = None
        for l in lines:
            l = l.strip()
            if l.startswith("!LTEINFO:"):
                state = "header"
                continue
            if state == "header":
                if l.startswith("Serving:"):
                    state = "serving"
                    continue
                if l.startswith("IntraFreq:"):
                    state = "intrafreq"
                    continue
            elif state == "serving":
                if len(l) == 0:
                    continue
                elif l.startswith("IntraFreq:"):
                    state = "intrafreq"
                    continue
                f = l.split()
                if len(f) < 13:
                    LOG.warning("Could not parse Sierra modem serving cell line: %s", l)
                    continue
                try:
                    serving_pci = int(f[9])
                    mcc = int(f[1])
                    mnc = int(f[2])
                    tac = int(f[3], 16)
                    cid = int(f[4], 16)
                    band = int(f[6])
                    earfcn = int(f[5])
                    sinr = float(f[7])
                    v = dict(
                        sinr=sinr,
                        rsrq=float(f[10]),
                        rsrp=float(f[11]),
                        rssi=float(f[12])
                    )
                    t = dict(
                        mcc=mcc,
                        mnc=mnc,
                        tac=tac,
                        cid=cid,
                        pci=serving_pci,
                        band=band,
                        earfcn=earfcn,
                        cell="serving"
                    )
                    values.append((v,t))
                except Exception as e:
                    LOG.warning("Could not parse Sierra modem serving cell line: %s: %s", l, e)
                    continue
            elif state == "intrafreq":
                if len(l) == 0:
                    continue
                elif l.startswith("InterFreq:"):
                    state = "interfreq"
                    continue
                f = l.split()
                if len(f) < 5:
                    LOG.warning("Could not parse Sierra modem intrafreq cell line: %s", l)
                    continue
                try:
                    pci = int(f[0])
                    if pci == serving_pci:
                        # Already recorded as the serving cell.
                        continue
                    v = dict(
                        rsrq=float(f[1]),
                        rsrp=float(f[2]),
                        rssi=float(f[3])
                    )
                    t = dict(
                        mcc=mcc,
                        mnc=mnc,
                        tac=tac,
                        cid=cid,
                        band=band,
                        earfcn=earfcn,
                        pci=pci,
                        cell="neighbor"
                    )
                    values.append((v,t))
                except Exception as e:
                    LOG.warning("Could not parse Sierra modem intrafreq cell line: %s: %s", l, e)
                    continue
            elif state == "interfreq":
                break
        return values

    def get_values(self, metric_name: str) -> List[Value]:
        if metric_name != self.__class__.metric_defn.name:
            return []   
        res = os.stat(self.file)
        if res.st_mtime == self._last_file_mtime:
            if not self.when_unchanged:
                return []
        self._last_file_mtime = res.st_mtime
        try:
            f = open(self.file, "r")
            lines = f.readlines()
            f.close()
        except Exception as e:
            LOG.error("Could not read Sierra modem data file %s: %s", self.file, e)
            raise e
        values = self._parse_lteinfo(lines)
        if len(values) == 0:
            LOG.warning("No cell data found in Sierra modem data file %s", self.file)
            return []
        ret = []
        for (v,t) in values:
            vv = Value(
                metric_id=str(uuid.UUID(int=0)),
                monitor_id=str(uuid.UUID(int=0)),
                created_at=datetime.datetime.fromtimestamp(self._last_file_mtime, tz=datetime.timezone.utc),
                fields=AnyObject.from_dict(v),
                tags=AnyObject.from_dict(t)
            )
            ret.append(vv)
        return ret
    