#!/bin/python

import time
import uuid
import os
import sys
import signal
import traceback
import argparse
import logging
import datetime
import datetime
from typing import (Dict)

from zmsclient.tools import DefaultDestEnvAction
from zmsclient.dst.client import ZmsDstClient
from zmsclient.dst.v1.models import (Metric, Value)
from zmsclient.dst.v1.models import Error as DstError

from . import (Monitor, ModemBase)
from .sierra import SierraModem

LOG = logging.getLogger(__name__)

LOGFILE = "/var/tmp/zmsclient-modem-monitor.log"

MODEMS = {
    "sierra": SierraModem,
}

def create_argparser(modems: Dict[str, ModemBase]) -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        prog="zmsclient-monitor-modem",
        description="An OpenZMS monitor for COTS modems.")
    parser.add_argument(
        "-D", "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Impotent: do not create observations in DST")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "--token", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="OpenZMS token")
    parser.add_argument(
        "--monitor-id", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Monitor ID to associate with observation")
    parser.add_argument(
        "--element-id", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Element ID containing monitor")
    parser.add_argument(
        "--dst-http", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="DST URL")
    # Modem type
    parser.add_argument(
        "--modem", default="sierra", type=str, choices=list(modems.keys()),
         required=True, help="Modem type")
    # Generic modem args
    parser.add_argument(
        "--device", type=str, help="Modem serial device source (e.g. /dev/ttyUSB2)")
    parser.add_argument(
        "--file", type=str, help="Modem data file source")
    parser.add_argument(
        "--interval", type=float, default=10.0,
        help="Interval (seconds) between value collection.")
    parser.add_argument(
        "--when-unchanged", default=False, action="store_true",
        help="Create values each interval, even if data source has not updated.")
    
    for m in modems.values():
        m.add_args(parser)
    
    return parser

def make_sigh(monitor: Monitor) -> callable:
    def sigh(signalnum, frame):
        LOG.info("Exiting on signal %d", signalnum)
        monitor.stop()
        exit(0)
    return sigh

def main():
    parser = create_argparser(MODEMS)
    
    args = parser.parse_args(sys.argv[1:])

    logging.basicConfig()

    if args.debug:
        LOG.setLevel(logging.DEBUG)
        logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
        logging.getLogger('zmsclient').setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)

    LOG.debug(f"args: {args}")

    if not args.impotent:
        if not args.token:
            raise ValueError("A --token must be specified")
        if not args.dst_http:
            raise ValueError("A --dst-http must be specified")
        if not args.monitor_id:
            raise ValueError("A --monitor-id must be specified")
        if not args.element_id:
            raise ValueError("A --element-id must be specified")
    else:
        if not args.monitor_id:
            args.monitor_id = str(uuid.uuid4())
            LOG.info(f"Using random monitor ID {args.monitor_id}")
        if not args.element_id:
            args.element_id = str(uuid.uuid4())
            LOG.info(f"Using random element ID {args.element_id}")

    if not args.device and not args.file:
        raise ValueError("either --device or --file must be specified")

    dst_client = ZmsDstClient(args.dst_http, args.token,
                              detailed=False, raise_on_unexpected_status=True)

    modem = MODEMS[args.modem](args)

    monitor = Monitor(
        dst_client, modem,
        monitor_id=args.monitor_id,
        element_id=args.element_id,
        interval=args.interval, impotent=args.impotent)
    monitor.init_metrics(create_if_missing=not args.impotent)

    if args.daemon:
        try:
            fp = open(args.logfile, "a");
            sys.stdout = fp
            sys.stderr = fp
            sys.stdin.close()
            logging.basicConfig(stream=fp)
        except:
            LOG.error(f"Could not open log file ({args.logfile}); aborting.")
            sys.exit(1)
        pid = os.fork()
        if pid:
            sys.exit(0)
        os.setsid()

    sigh_exit = make_sigh(monitor)

    signal.signal(signal.SIGINT, sigh_exit)
    signal.signal(signal.SIGTERM, sigh_exit)
    signal.signal(signal.SIGHUP, sigh_exit)

    return monitor.run()

if __name__ == "__main__":
    main()
