import datetime
from typing import TYPE_CHECKING, Any, Dict, List, Type, TypeVar, Union, cast

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.any_object import AnyObject


T = TypeVar("T", bound="Action")


@_attrs_define
class Action:
    """An action resource provided by monitors.  Defines inputs and outputs to be used in MonitorTasks.

    Attributes:
        name (str): The name of the action.
        kind (str): The kind of data: sample, fft, psd, spectrogram, other.  This should match the kind field in any
            Observations reported to zms-dst as a result of this action. Example: psd.
        type (str): A fully-qualified type for the output data.  For instance, POWDER SDR monitors use type
            `powder.rfmonitor-v2`.  This should match the type field in any Observations reported to zms-dst as a result of
            this action. Example: powder.rfmonitor-v2.
        format_ (str): The data format (e.g. `sigmf`, `csv`, `binary`).  This should match the format field in any
            Observations reported to zms-dst as a result of this action. Example: sigmf.
        element_id (str): The id of the element that created this action.
        id (Union[Unset, str]): The id of the action resource.
        description (Union[None, Unset, str]): The description of the action.
        html_url (Union[None, Unset, str]): A descriptive URL for the action.
        parameter_defs (Union[None, Unset, AnyObject]):
        is_public (Union[Unset, bool]): True if this action is available to all elements; false if only available to the
            creator's element.
        creator_id (Union[Unset, str]): The user id of the creator.
        updater_id (Union[None, Unset, str]): The user id of the updater.
        created_at (Union[Unset, datetime.datetime]): Creation time.
        updated_at (Union[None, Unset, datetime.datetime]): Last modification time.
        deleted_at (Union[None, Unset, datetime.datetime]): Deletion time.
    """

    name: str
    kind: str
    type: str
    format_: str
    element_id: str
    id: Union[Unset, str] = UNSET
    description: Union[None, Unset, str] = UNSET
    html_url: Union[None, Unset, str] = UNSET
    parameter_defs: Union[None, Unset, "AnyObject"] = UNSET
    is_public: Union[Unset, bool] = UNSET
    creator_id: Union[Unset, str] = UNSET
    updater_id: Union[None, Unset, str] = UNSET
    created_at: Union[Unset, datetime.datetime] = UNSET
    updated_at: Union[None, Unset, datetime.datetime] = UNSET
    deleted_at: Union[None, Unset, datetime.datetime] = UNSET
    additional_properties: Dict[str, Any] = _attrs_field(init=False, factory=dict)
    _immutable_on = {
        "id": {"put": True, "post": True},
        "name": {"put": None, "post": None},
        "description": {"put": None, "post": None},
        "html_url": {"put": None, "post": None},
        "kind": {"put": None, "post": None},
        "type": {"put": None, "post": None},
        "format": {"put": None, "post": None},
        "parameter_defs": {"put": None, "post": None},
        "is_public": {"put": None, "post": None},
        "element_id": {"put": True, "post": None},
        "creator_id": {"put": True, "post": True},
        "updater_id": {"put": True, "post": True},
        "created_at": {"put": True, "post": True},
        "updated_at": {"put": True, "post": True},
        "deleted_at": {"put": True, "post": True},
    }

    def to_dict(self) -> Dict[str, Any]:
        name = self.name

        kind = self.kind

        type = self.type

        format_ = self.format_

        element_id = self.element_id

        id = self.id

        description: Union[None, Unset, str]
        if isinstance(self.description, Unset):
            description = UNSET
        else:
            description = self.description

        html_url: Union[None, Unset, str]
        if isinstance(self.html_url, Unset):
            html_url = UNSET
        else:
            html_url = self.html_url

        parameter_defs: Union[None, Unset, Dict[str, Any]] = UNSET
        if self.parameter_defs is None:
            parameter_defs = None
        elif not isinstance(self.parameter_defs, Unset):
            parameter_defs = self.parameter_defs.to_dict()

        is_public = self.is_public

        creator_id = self.creator_id

        updater_id: Union[None, Unset, str]
        if isinstance(self.updater_id, Unset):
            updater_id = UNSET
        else:
            updater_id = self.updater_id

        created_at: Union[Unset, str] = UNSET
        if not isinstance(self.created_at, Unset):
            created_at = self.created_at.isoformat()

        updated_at: Union[None, Unset, str]
        if isinstance(self.updated_at, Unset):
            updated_at = UNSET
        elif isinstance(self.updated_at, datetime.datetime):
            updated_at = self.updated_at.isoformat()
        else:
            updated_at = self.updated_at

        deleted_at: Union[None, Unset, str]
        if isinstance(self.deleted_at, Unset):
            deleted_at = UNSET
        elif isinstance(self.deleted_at, datetime.datetime):
            deleted_at = self.deleted_at.isoformat()
        else:
            deleted_at = self.deleted_at

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "name": name,
                "kind": kind,
                "type": type,
                "format": format_,
                "element_id": element_id,
            }
        )
        if id is not UNSET:
            field_dict["id"] = id
        if description is not UNSET:
            field_dict["description"] = description
        if html_url is not UNSET:
            field_dict["html_url"] = html_url
        if parameter_defs is not UNSET:
            field_dict["parameter_defs"] = parameter_defs
        if is_public is not UNSET:
            field_dict["is_public"] = is_public
        if creator_id is not UNSET:
            field_dict["creator_id"] = creator_id
        if updater_id is not UNSET:
            field_dict["updater_id"] = updater_id
        if created_at is not UNSET:
            field_dict["created_at"] = created_at
        if updated_at is not UNSET:
            field_dict["updated_at"] = updated_at
        if deleted_at is not UNSET:
            field_dict["deleted_at"] = deleted_at

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        from ..models.any_object import AnyObject

        d = src_dict.copy()
        name = d.pop("name")

        kind = d.pop("kind")

        type = d.pop("type")

        format_ = d.pop("format")

        element_id = d.pop("element_id")

        id = d.pop("id", UNSET)

        def _parse_description(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        description = _parse_description(d.pop("description", UNSET))

        def _parse_html_url(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        html_url = _parse_html_url(d.pop("html_url", UNSET))

        _parameter_defs = d.pop("parameter_defs", UNSET)
        parameter_defs: Union[None, Unset, AnyObject]
        if isinstance(_parameter_defs, Unset):
            parameter_defs = UNSET
        elif _parameter_defs is None:
            parameter_defs = None
        else:
            parameter_defs = AnyObject.from_dict(_parameter_defs)

        is_public = d.pop("is_public", UNSET)

        creator_id = d.pop("creator_id", UNSET)

        def _parse_updater_id(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        updater_id = _parse_updater_id(d.pop("updater_id", UNSET))

        _created_at = d.pop("created_at", UNSET)
        created_at: Union[Unset, datetime.datetime]
        if isinstance(_created_at, Unset):
            created_at = UNSET
        else:
            created_at = isoparse(_created_at)

        def _parse_updated_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                updated_at_type_0 = isoparse(data)

                return updated_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        updated_at = _parse_updated_at(d.pop("updated_at", UNSET))

        def _parse_deleted_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                deleted_at_type_0 = isoparse(data)

                return deleted_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        deleted_at = _parse_deleted_at(d.pop("deleted_at", UNSET))

        action = cls(
            name=name,
            kind=kind,
            type=type,
            format_=format_,
            element_id=element_id,
            id=id,
            description=description,
            html_url=html_url,
            parameter_defs=parameter_defs,
            is_public=is_public,
            creator_id=creator_id,
            updater_id=updater_id,
            created_at=created_at,
            updated_at=updated_at,
            deleted_at=deleted_at,
        )

        action.additional_properties = d
        return action

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties


def _load_lazy_imports(ns=globals()):
    from ..models.any_object import AnyObject

    ns["AnyObject"] = AnyObject
