#!/bin/python

import uuid
import os
import sys
import signal
import argparse
import logging
import asyncio

from . import DefaultDestEnvAction
import zmsclient.identity.v1
import zmsclient.identity.client
import zmsclient.identity.v1.models
import zmsclient.zmc.client
import zmsclient.zmc.v1.models
import zmsclient.dst.client
import zmsclient.dst.v1.models
import zmsclient.alarm.client
import zmsclient.alarm.v1.models
from zmsclient.common.subscription import ZmsSubscriptionCallback

LOG = logging.getLogger(__name__)

# Only when a daemon
LOGFILE = "/local/logs/zms-watcher.log"


class SubscriptionCallback(ZmsSubscriptionCallback):
    def on_event(self, ws, evt, message):
        LOG.info("event: %r", evt)

def init_main():
    parser = argparse.ArgumentParser(
        prog="watcher",
        description="Watch events from one or more OpenZMS services.")
    parser.add_argument(
        "-b", "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "--no-verify-ssl", default=False, action="store_true",
        help="Disable SSL verification"
    )
    parser.add_argument(
        "--token", action=DefaultDestEnvAction, type=str, required=True,
        help="OpenZMS token")
    parser.add_argument(
        "--element-id", type=str, default=None,
        help="Element ID to use in subscription filters")
    parser.add_argument(
        "--user-id", type=str, required=False,
        help="User ID to use in subscription filters")
    parser.add_argument(
        "--identity-http", action=DefaultDestEnvAction, type=str, required=False,
        help="identity service URL")
    parser.add_argument(
        "--zmc-http", action=DefaultDestEnvAction, type=str, required=False,
        help="zmc service URL")
    parser.add_argument(
        "--dst-http", action=DefaultDestEnvAction, type=str, required=False,
        help="dst service URL")
    parser.add_argument(
        "--alarm-http", action=DefaultDestEnvAction, type=str, required=False,
        help="alarm service URL")

    args = parser.parse_args(sys.argv[1:])

    if args.debug:
        LOG.setLevel(logging.DEBUG)
        logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
        logging.getLogger('zmsclient').setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)

    return args

def sigh_exit(signalnum, frame):
    exit(0)

# The hander has to be outside the async main.
def set_signal_handler(signum, task_to_cancel):
    def handler(_signum, _frame):
        asyncio.get_running_loop().call_soon_threadsafe(task_to_cancel.cancel)
    signal.signal(signum, handler)

async def async_main(*args, tasks=[], subs=[]):
    this_task = asyncio.current_task()
    set_signal_handler(signal.SIGINT, this_task)
    set_signal_handler(signal.SIGHUP, this_task)
    set_signal_handler(signal.SIGTERM, this_task)

    try:
        runnable = [*[task.run() for task in tasks], *[sub.run_callbacks() for sub in subs]]
        await asyncio.gather(*runnable)
    except asyncio.CancelledError:
        for sub in subs:
            if sub.id:
                sub.unsubscribe()

def main():
    args = init_main()

    if args.daemon:
        try:
            fp = open(args.logfile, "a");
            sys.stdout = fp
            sys.stderr = fp
            sys.stdin.close()
            logging.basicConfig(stream=fp)
        except:
            print(f"Could not open log file ({args.logfile}); aborting.")
            sys.exit(1)
        pid = os.fork()
        if pid:
            sys.exit(0)
        os.setsid()
    else:
        logging.basicConfig()

    signal.signal(signal.SIGINT, sigh_exit)
    signal.signal(signal.SIGTERM, sigh_exit)
    signal.signal(signal.SIGHUP, sigh_exit)

    subs = []

    if args.identity_http:
        identity_client = zmsclient.identity.client.ZmsIdentityClient(
            args.identity_http, args.token,
            detailed=False, raise_on_unexpected_status=True,
            verify_ssl=not args.no_verify_ssl)
        subscription = zmsclient.identity.v1.models.Subscription(
            id=str(uuid.uuid4()))
        filters = None
        if args.element_id or args.user_id:
            kwargs = {}
            if args.element_id:
                kwargs['element_ids'] = [args.element_id]
            if args.user_id:
                kwargs['user_ids'] = [args.user_id]
            filters = [zmsclient.identity.v1.models.EventFilter(**kwargs)]
            subscription.filters = filters
        identitySubscription = SubscriptionCallback(
            identity_client, subscription=subscription, reconnect_on_error=True,
            verify_ssl=not args.no_verify_ssl)
        subs.append(identitySubscription)

    if args.zmc_http:
        zmc_client = zmsclient.zmc.client.ZmsZmcClient(
            args.zmc_http, args.token,
            detailed=False, raise_on_unexpected_status=True,
            verify_ssl=not args.no_verify_ssl)
        subscription = zmsclient.zmc.v1.models.Subscription(
            id=str(uuid.uuid4()))
        filters = None
        if args.element_id or args.user_id:
            kwargs = {}
            if args.element_id:
                kwargs['element_ids'] = [args.element_id]
            if args.user_id:
                kwargs['user_ids'] = [args.user_id]
            filters = [zmsclient.zmc.v1.models.EventFilter(**kwargs)]
            subscription.filters = filters
        zmcSubscription = SubscriptionCallback(
            zmc_client, subscription=subscription, reconnect_on_error=True,
            verify_ssl=not args.no_verify_ssl)
        subs.append(zmcSubscription)

    if args.dst_http:
        dst_client = zmsclient.dst.client.ZmsDstClient(
            args.dst_http, args.token,
            detailed=False, raise_on_unexpected_status=True,
            verify_ssl=not args.no_verify_ssl)
        subscription = zmsclient.dst.v1.models.Subscription(
            id=str(uuid.uuid4()))
        filters = None
        if args.element_id or args.user_id:
            kwargs = {}
            if args.element_id:
                kwargs['element_ids'] = [args.element_id]
            if args.user_id:
                kwargs['user_ids'] = [args.user_id]
            filters = [zmsclient.dst.v1.models.EventFilter(**kwargs)]
            subscription.filters = filters
        dstSubscription = SubscriptionCallback(
            dst_client, subscription=subscription, reconnect_on_error=True,
            verify_ssl=not args.no_verify_ssl)
        subs.append(dstSubscription)

    if args.alarm_http:
        alarm_client = zmsclient.alarm.client.ZmsAlarmClient(
            args.alarm_http, args.token,
            detailed=False, raise_on_unexpected_status=True,
            verify_ssl=not args.no_verify_ssl)
        subscription = zmsclient.alarm.v1.models.Subscription(
            id=str(uuid.uuid4()))
        filters = None
        if args.element_id or args.user_id:
            kwargs = {}
            if args.element_id:
                kwargs['element_ids'] = [args.element_id]
            if args.user_id:
                kwargs['user_ids'] = [args.user_id]
            filters = [zmsclient.alarm.v1.models.EventFilter(**kwargs)]
            subscription.filters = filters
        alarmSubscription = SubscriptionCallback(
            alarm_client, subscription=subscription, reconnect_on_error=True,
            verify_ssl=not args.no_verify_ssl)
        subs.append(alarmSubscription)
        
    if not subs:
        print("No subscriptions configured; exiting.")
        sys.exit(0)

    asyncio.run(async_main(tasks=[], subs=subs))

if __name__ == "__main__":
    main()
