import argparse
import csv
import datetime
import logging
import uuid
from typing import (List, Dict)

from zmsclient.dst.v1.models import (Metric, Value, AnyObject)

from . import BaseDataType

LOG = logging.getLogger(__name__)

class GNetTrackProCsvDataType(BaseDataType):
    name: str = "gnettrackpro-csv"
    metric_schema = {
        "$id": "https://gitlab.flux.utah.edu/openzms/zms-dst-schemas/schemas/metrics/gyokovsolutions/gnettrackpro-csv/v1/schema.json",
        "type": "object",
        "title": "G-NetTrack Pro CSV Metrics v1",
        "$schema": "https://json-schema.org/draft/2020-12/schema",
        "x-label-template": "cgi=${tag_cgi}-pci=${tag_pci}",
        "x-tag-filter": {
            "cell": "serving"
        },
        "properties": {
            "latitude": {
                "type": "number",
                "description": "Latitude.",
                "x-unit": "degrees",
                "x-group": "geospatial"
            },
            "longitude": {
                "type": "number",
                "description": "Longitude.",
                "x-unit": "degrees",
                "x-group": "geospatial"
            },
            "altitude": {
                "type": "number",
                "description": "Altitude.",
                "x-unit": "meters",
                "x-group": "geospatial"
            },
            "accuracy": {
                "type": "number",
                "description": "Accuracy.",
                "x-unit": "meters",
                "x-group": "geospatial"
            },
            "speed": {
                "type": "number",
                "description": "Speed.",
                "x-unit": "m/s",
                "x-group": "geospatial"
            },
            "rsrp": {
                "type": "number",
                "description": "Reference Signal Received Power (RSRP)",
                "x-unit": "dBm",
                "x-group": "cellular"
            },
            "rsrq": {
                "type": "number",
                "description": "Reference Signal Received Quality (RSRQ).",
                "x-unit": "dB",
                "x-group": "cellular-quality"
            },
            "snr": {
                "type": "number",
                "description": "Signal to Noise Ratio (SNR).",
                "x-unit": "dB",
                "x-group": "cellular-quality"
            },
            "cqi": {
                "type": "number",
                "description": "Channel Quality Indicator (CQI).",
                "x-unit": "index",
                "x-group": "cellular-quality"
            },
            "dlbitrate": {
                "type": "number",
                "description": "Downlink Bitrate.",
                "x-unit": "bps",
                "x-group": "throughput"
            },
            "ulbitrate": {
                "type": "number",
                "description": "Uplink Bitrate.",
                "x-unit": "bps",
                "x-group": "throughput"
            },
            "servingtime": {
                "type": "number",
                "description": "Time in serving cell.",
                "x-unit": "seconds",
                "x-group": "cellular-duration"
            },
            "cell": {
                "type": "string",
                "enum": ["serving", "neighbor"],
                "description": "Type of cell: serving or neighbor.",
                "x-tag": True
            },
            "operatorname": {
                "type": "string",
                "description": "Operator Name.",
                "x-tag": True
            },
            "operator": {
                "type": "string",
                "description": "Operator PLMN.",
                "x-tag": True
            },
            "cgi": {
                "type": "string",
                "description": "Cell Global Identity (CGI).",
                "x-tag": True
            },
            "cellid": {
                "type": "string",
                "description": "Cell ID.",
                "x-tag": True
            },
            "pci": {
                "type": "string",
                "description": "Physical Cell ID (PCI).",
                "x-tag": True
            },
            "lac": {
                "type": "string",
                "description": "Location Area Code (LAC).",
                "x-tag": True
            },
            "ntech": {
                "type": "string",
                "description": "Network Technology (e.g 4G or 5G).",
                "x-tag": True
            },
            "nmode": {
                "type": "string",
                "description": "Network Mode (e.g. 5G NSA or 5G SA).",
                "x-tag": True
            },
            "arfcn": {
                "type": "integer",
                "description": "Absolute Radio Frequency Channel Number (ARFCN).",
                "x-tag": True
            },
            "location": {
                "type": "string",
                "description": "Location source (G=GPS, N=network).",
                "x-tag": True
            },
            "state": {
                "type": "string",
                "description": "State of the device (e.g. I=idle, V=voice, D=data).",
                "x-tag": True
            },
            "event": {
                "type": "string",
                "description": "Event (e.g. HANDOVER...).",
                "x-tag": True
            },
            "band": {
                "type": "string",
                "description": "Band (e.g. B5, N78...).",
                "x-tag": True
            },
            "bandwidth": {
                "type": "integer",
                "description": "Bandwidth in MHz (e.g. 20, 100).",
                "x-tag": True
            }
        }
    }
    metric_defn: Metric = Metric(
        name="gyokovsolutions.gnettrackpro-csv.v1",
        element_id=uuid.UUID(int=0),
        description="G-NetTrack Pro CSV Metrics v1.  See https://www.gyokovsolutions.com/manuals/gnettrackpro_manualcontent.html#logfile for details.",
        schema=AnyObject.from_dict(metric_schema)
    )
    field_map: Dict[str,str] = dict(
        longitude="Longitude",
        latitude="Latitude",
        altitude="Altitude",
        accuracy="Accuracy",
        speed="Speed",
        rsrp="Level",
        rsrq="Qual",
        snr="SNR",
        cqi="CQI",
        dlbitrate="DL_bitrate",
        ulbitrate="UL_bitrate",
        servingtime="SERVINGTIME",
    )
    tag_map: Dict[str,str] = dict(
        operatorname="Operatorname",
        operator="Operator",
        cgi="CGI",
        cellid="CellID",
        pci="PSC",
        lac="LAC",
        ntech="NetworkTech",
        nmode="NetworkMode",
        arfcn="ARFCN",
        location="Location",
        state="State",
        event="EVENT",
        band="BAND",
        bandwidth="BANDWIDTH",
    )
    # Append 1..18 to the neighbor field names to extract from CSV columns.
    ncell_field_map: Dict[str,str] = dict(
        rsrp="NRxLev",
        rsrq="NQual",
    )
    # Copy some values and optionally location info to neighbor cells.
    ncell_field_copies = []
    ncell_field_location_copies = [ "longitude", "latitude", "altitude", "accuracy", "speed" ]
    ncell_tag_map: Dict[str,str] = dict(
        ntech="NTech",
        cellid="NCellid",
        lac="NLAC",
        pci="NCell",
        arfcn="NARFCN",
    )
    # Copy some tags and optionally location info to neighbor cells.
    ncell_tag_copies = [ "operatorname", "operator", "cgi", "state", "band" ]
    ncell_tag_location_copies = [ "location" ]

    @classmethod
    def add_args(kls, parser: argparse.ArgumentParser):
        """Add any additional arguments needed for this data type."""
        parser.add_argument(
            "--gnettrackpro-operator", type=str, required=False, default=None,
            help="Only include operators whose PLMN matches this string.")
        parser.add_argument(
            "--gnettrackpro-serving-only", default=False, action="store_true",
            help="Only include serving cell entries.")
        parser.add_argument(
            "--gnettrackpro-no-duplicate-gps", default=False, action="store_true",
            help="Do not include duplicate GPS information for neighbor cells.")

    @classmethod
    def get_metric_definitions(self) -> List[Metric]:
        """Define the metrics that will be collected."""
        return [self.metric_defn]

    # Parse the given file.
    def load_data(self):
        """Parse the given file."""
        with open(self.args.data_file, "r") as f:
            csvreader = csv.DictReader(f, delimiter="\t")
            i = 0
            for row in csvreader:
                i += 1
                try:
                    # 2025.09.12_11.47.59
                    dt = datetime.datetime.strptime(row["Timestamp"], "%Y.%m.%d_%H.%M.%S")
                    dt = dt.astimezone(datetime.timezone.utc)
                except Exception as e:
                    msg = f"Cannot parse row {i} with bad date/time ({e}): {row}"
                    if self.args.continue_on_error:
                        LOG.warning(msg)
                        continue
                    else:
                        raise ValueError(msg)
                if self.args.gnettrackpro_operator != row.get("Operator", None):
                    LOG.debug("Skipping row %d with operator %s not matching filter %s", i, row.get("Operator", None), self.args.gnettrackpro_operator)
                    continue
                # First assemble the values and tags for the serving cell.
                svd = dict()
                for fn, cn in self.field_map.items():
                    v = row.get(cn, None)
                    if v is None or v == "" or v == "-":
                        continue
                    try:
                        vt = self.metric_schema["properties"][fn]["type"]
                        if vt == "number":
                            svd[fn] = float(v)
                        elif vt == "integer":
                            svd[fn] = int(v)
                        else:
                            svd[fn] = v
                        # One-off: convert speed in km/h to m/s.
                        if fn == "speed":
                            svd[fn] = svd[fn] * (1000.0 / (60.0 * 60.0))
                    except Exception as e:
                        msg = "Could not parse field %s with value %s as type %s: %s" % (fn, v, vt, e)
                        if self.args.continue_on_error:
                            LOG.warning(msg)
                            continue
                        else:
                            raise ValueError(msg)
                std = dict()
                for tn, cn in self.tag_map.items():
                    v = row.get(cn, None)
                    if v is None or v == "" or v == "-":
                        continue
                    try:
                        vt = self.metric_schema["properties"][tn]["type"]
                        if vt == "number":
                            std[tn] = float(v)
                        elif vt == "integer":
                            std[tn] = int(v)
                        else:
                            std[tn] = v
                    except Exception as e:
                        msg = "Could not parse tag %s with value %s as type %s: %s" % (tn, v, vt, e)
                        if self.args.continue_on_error:
                            LOG.warning(msg)
                            continue
                        else:
                            raise ValueError(msg)
                std["cell"] = "serving"
                v = Value(
                    monitor_id=self.mh.monitor_id,
                    metric_id=str(uuid.UUID(int=0)),  # filled in later
                    created_at=dt,
                    fields=AnyObject.from_dict(svd),
                    tags=AnyObject.from_dict(std)
                )
                self.values.append(v)
                LOG.debug("Parsed row %d serving cell: %s", i, v)

                # If requested, skip neighbor cell info.
                if self.args.gnettrackpro_serving_only:
                    continue

                # Now assemble the values and tags for each neighbor cell.
                spci = std.get("pci", None)
                for j in range(1,19):
                    nvd = dict()
                    ntd = dict()
                    for fn, cn in self.ncell_field_map.items():
                        v = row.get(f"{cn}{j}", None)
                        if v is None or v == "" or v == "-":
                            continue
                        try:
                            vt = self.metric_schema["properties"][fn]["type"]
                            if vt == "number":
                                nvd[fn] = float(v)
                            elif vt == "integer":
                                nvd[fn] = int(v)
                            else:
                                nvd[fn] = v
                        except Exception as e:
                            msg = "Could not parse neighbor field %d on line %d: %s with value %s as type %s: %r" % (j, i, fn, v, vt, e)
                            if self.args.continue_on_error:
                                LOG.warning(msg)
                                continue
                            else:
                                raise ValueError(msg)
                    # Skip empty neighbor cell columns.
                    if len(nvd) == 0:
                        LOG.debug("Skipping empty neighbor cell %d on line %d", j, i)
                        continue
                    for tn, cn in self.ncell_tag_map.items():
                        v = row.get(f"{cn}{j}", None)
                        if v is None or v == "" or v == "-":
                            continue
                        try:
                            vt = self.metric_schema["properties"][tn]["type"]
                            if vt == "number":
                                ntd[tn] = float(v)
                            elif vt == "integer":
                                ntd[tn] = int(v)
                            else:
                                ntd[tn] = v
                        except Exception as e:
                            msg = "Could not parse neighbor tag %d on line %d: %s with value %s as type %s: %r" % (j, i, fn, v, vt, e)
                            if self.args.continue_on_error:
                                LOG.warning(msg)
                                continue
                            else:
                                raise ValueError(msg)
                    # Copy over some fields and tags from the serving cell.
                    for fn in self.ncell_field_copies:
                        if fn in svd:
                            nvd[fn] = svd[fn]
                    if not self.args.gnettrackpro_no_duplicate_gps:
                        for fn in self.ncell_field_location_copies:
                            if fn in svd:
                                nvd[fn] = svd[fn]
                    for tn in self.ncell_tag_copies:
                        if tn in std:
                            ntd[tn] = std[tn]
                    if not self.args.gnettrackpro_no_duplicate_gps:
                        for tn in self.ncell_tag_location_copies:
                            if tn in std:
                                ntd[tn] = std[tn]
                    # If the neighbor cell PCI matches the serving cell PCI, skip it.
                    if spci == ntd.get("pci", None):
                        LOG.debug("Skipping neighbor cell %d on line %d: neighbor has same PCI %s as serving cell", j, i, spci)
                        continue
                    ntd["cell"] = "neighbor"
                    nv = Value(
                        monitor_id=self.mh.monitor_id,
                        metric_id=str(uuid.UUID(int=0)),  # filled in later
                        created_at=dt,
                        fields=AnyObject.from_dict(nvd),
                        tags=AnyObject.from_dict(ntd)
                    )
                    self.values.append(nv)
                    LOG.debug("Parsed row %d neighbor cell %d: %s", i, j, nv)

    def prepare_values(self):
        """Prepare any values before uploading."""
        if len(self.values) == 0:
            LOG.warning("No values to prepare")
            return
        for v in self.values:
            v.metric_id = self.mh.get_metric(self.metric_defn.name).id
            v.series_id = self.mh.get_series(self.metric_defn.name).id