import datetime
from typing import (
    TYPE_CHECKING,
    Any,
    Dict,
    List,
    Type,
    TypeVar,
    Union,
    cast,
)

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.series import Series


T = TypeVar("T", bound="DatasetSeries")


@_attrs_define
class DatasetSeries:
    """A DatasetSeries records that a Series is part of a Dataset.

    Attributes:
        dataset_id (str): The id of the dataset.
        series_id (str): The id of the series.
        id (Union[Unset, str]): The id of the dataset series.
        creator_id (Union[Unset, str]): The user id of the creator.
        created_at (Union[Unset, datetime.datetime]): Creation time.
        deleted_at (Union[None, Unset, datetime.datetime]): Deletion time.
        series (Union[None, Unset, Series]): A series of metric values reported from a monitor.
    """

    dataset_id: str
    series_id: str
    id: Union[Unset, str] = UNSET
    creator_id: Union[Unset, str] = UNSET
    created_at: Union[Unset, datetime.datetime] = UNSET
    deleted_at: Union[None, Unset, datetime.datetime] = UNSET
    series: Union[None, Unset, "Series"] = UNSET
    additional_properties: Dict[str, Any] = _attrs_field(init=False, factory=dict)
    _immutable_on = {
        "id": {"put": True, "post": True},
        "dataset_id": {"put": True, "post": None},
        "series_id": {"put": True, "post": None},
        "creator_id": {"put": True, "post": True},
        "created_at": {"put": True, "post": True},
        "deleted_at": {"put": True, "post": True},
        "series": {"put": True, "post": True},
    }

    def to_dict(self) -> Dict[str, Any]:
        dataset_id = self.dataset_id

        series_id = self.series_id

        id = self.id

        creator_id = self.creator_id

        created_at: Union[Unset, str] = UNSET
        if not isinstance(self.created_at, Unset):
            created_at = self.created_at.isoformat()

        deleted_at: Union[None, Unset, str]
        if isinstance(self.deleted_at, Unset):
            deleted_at = UNSET
        elif isinstance(self.deleted_at, datetime.datetime):
            deleted_at = self.deleted_at.isoformat()
        else:
            deleted_at = self.deleted_at

        series: Union[None, Unset, Dict[str, Any]] = UNSET
        if self.series is None:
            series = None
        elif not isinstance(self.series, Unset):
            series = self.series.to_dict()

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "dataset_id": dataset_id,
                "series_id": series_id,
            }
        )
        if id is not UNSET:
            field_dict["id"] = id
        if creator_id is not UNSET:
            field_dict["creator_id"] = creator_id
        if created_at is not UNSET:
            field_dict["created_at"] = created_at
        if deleted_at is not UNSET:
            field_dict["deleted_at"] = deleted_at
        if series is not UNSET:
            field_dict["series"] = series

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        from ..models.series import Series

        d = src_dict.copy()
        dataset_id = d.pop("dataset_id")

        series_id = d.pop("series_id")

        id = d.pop("id", UNSET)

        creator_id = d.pop("creator_id", UNSET)

        _created_at = d.pop("created_at", UNSET)
        created_at: Union[Unset, datetime.datetime]
        if isinstance(_created_at, Unset):
            created_at = UNSET
        else:
            created_at = isoparse(_created_at)

        def _parse_deleted_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                deleted_at_type_0 = isoparse(data)

                return deleted_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        deleted_at = _parse_deleted_at(d.pop("deleted_at", UNSET))

        _series = d.pop("series", UNSET)
        series: Union[None, Unset, Series]
        if isinstance(_series, Unset):
            series = UNSET
        elif _series is None:
            series = None
        else:
            series = Series.from_dict(_series)

        dataset_series = cls(
            dataset_id=dataset_id,
            series_id=series_id,
            id=id,
            creator_id=creator_id,
            created_at=created_at,
            deleted_at=deleted_at,
            series=series,
        )

        dataset_series.additional_properties = d
        return dataset_series

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties


def _load_lazy_imports(ns=globals()):
    from ..models.series import Series

    ns["Series"] = Series
