from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.observation import Observation
from ...types import UNSET, Response, Unset


def _get_kwargs(
    *,
    body: Observation,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_monitor_id: Union[Unset, str] = UNSET,
    content_type: Union[Unset, str] = UNSET,
    content_encoding: Union[Unset, str] = UNSET,
    x_api_skip_invalid: Union[Unset, bool] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    headers["X-Api-Token"] = x_api_token

    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    if not isinstance(x_api_monitor_id, Unset):
        headers["X-Api-Monitor-Id"] = x_api_monitor_id

    if not isinstance(content_type, Unset):
        headers["Content-Type"] = content_type

    if not isinstance(content_encoding, Unset):
        headers["Content-Encoding"] = content_encoding

    if not isinstance(x_api_skip_invalid, Unset):
        headers["X-Api-Skip-Invalid"] = "true" if x_api_skip_invalid else "false"

    _kwargs: Dict[str, Any] = {
        "method": "post",
        "url": "/observations",
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, Observation]]:
    if response.status_code == HTTPStatus.CREATED:
        try:
            response_201 = Observation.from_dict(response.json())

            return response_201
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, Observation]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    body: Observation,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_monitor_id: Union[Unset, str] = UNSET,
    content_type: Union[Unset, str] = UNSET,
    content_encoding: Union[Unset, str] = UNSET,
    x_api_skip_invalid: Union[Unset, bool] = UNSET,
) -> Response[Union[Error, Observation]]:
    """Create a new observation.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        x_api_monitor_id (Union[Unset, str]): Associate the new observation with the monitor_id
            that observed it.  Must be used for non-native OpenZMS observation formats.
        content_type (Union[Unset, str]): Determines the type of incoming observation.  If
            unspecified, or if set to `application/openzms+observation+v1` or `application/json`; the
            request body must be a native OpenZMS Observation JSON object.  If set to
            `application/sigmf-meta`, the request body must be a SigMF metadata file as a JSON object
            (the .sigmf-data file must be uploaded separately).  If set to `application/sigmf-
            archive`, the request body must be a SigMF archive (simple tarball with one each of
            .sigmf-meta and .sigmf-data files).
        content_encoding (Union[Unset, str]): Set to `gzip` if you have compressed the request
            body with gzip.
        x_api_skip_invalid (Union[Unset, bool]): If true, skip invalid values attached to the
            observation instead of rejecting the entire observation.
        body (Observation): Observation report from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Observation]]
    """

    kwargs = _get_kwargs(
        body=body,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
        x_api_monitor_id=x_api_monitor_id,
        content_type=content_type,
        content_encoding=content_encoding,
        x_api_skip_invalid=x_api_skip_invalid,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    *,
    client: Union[AuthenticatedClient, Client],
    body: Observation,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_monitor_id: Union[Unset, str] = UNSET,
    content_type: Union[Unset, str] = UNSET,
    content_encoding: Union[Unset, str] = UNSET,
    x_api_skip_invalid: Union[Unset, bool] = UNSET,
) -> Optional[Union[Error, Observation]]:
    """Create a new observation.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        x_api_monitor_id (Union[Unset, str]): Associate the new observation with the monitor_id
            that observed it.  Must be used for non-native OpenZMS observation formats.
        content_type (Union[Unset, str]): Determines the type of incoming observation.  If
            unspecified, or if set to `application/openzms+observation+v1` or `application/json`; the
            request body must be a native OpenZMS Observation JSON object.  If set to
            `application/sigmf-meta`, the request body must be a SigMF metadata file as a JSON object
            (the .sigmf-data file must be uploaded separately).  If set to `application/sigmf-
            archive`, the request body must be a SigMF archive (simple tarball with one each of
            .sigmf-meta and .sigmf-data files).
        content_encoding (Union[Unset, str]): Set to `gzip` if you have compressed the request
            body with gzip.
        x_api_skip_invalid (Union[Unset, bool]): If true, skip invalid values attached to the
            observation instead of rejecting the entire observation.
        body (Observation): Observation report from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Observation]
    """

    return sync_detailed(
        client=client,
        body=body,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
        x_api_monitor_id=x_api_monitor_id,
        content_type=content_type,
        content_encoding=content_encoding,
        x_api_skip_invalid=x_api_skip_invalid,
    ).parsed


async def asyncio_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    body: Observation,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_monitor_id: Union[Unset, str] = UNSET,
    content_type: Union[Unset, str] = UNSET,
    content_encoding: Union[Unset, str] = UNSET,
    x_api_skip_invalid: Union[Unset, bool] = UNSET,
) -> Response[Union[Error, Observation]]:
    """Create a new observation.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        x_api_monitor_id (Union[Unset, str]): Associate the new observation with the monitor_id
            that observed it.  Must be used for non-native OpenZMS observation formats.
        content_type (Union[Unset, str]): Determines the type of incoming observation.  If
            unspecified, or if set to `application/openzms+observation+v1` or `application/json`; the
            request body must be a native OpenZMS Observation JSON object.  If set to
            `application/sigmf-meta`, the request body must be a SigMF metadata file as a JSON object
            (the .sigmf-data file must be uploaded separately).  If set to `application/sigmf-
            archive`, the request body must be a SigMF archive (simple tarball with one each of
            .sigmf-meta and .sigmf-data files).
        content_encoding (Union[Unset, str]): Set to `gzip` if you have compressed the request
            body with gzip.
        x_api_skip_invalid (Union[Unset, bool]): If true, skip invalid values attached to the
            observation instead of rejecting the entire observation.
        body (Observation): Observation report from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Observation]]
    """

    kwargs = _get_kwargs(
        body=body,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
        x_api_monitor_id=x_api_monitor_id,
        content_type=content_type,
        content_encoding=content_encoding,
        x_api_skip_invalid=x_api_skip_invalid,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    *,
    client: Union[AuthenticatedClient, Client],
    body: Observation,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
    x_api_monitor_id: Union[Unset, str] = UNSET,
    content_type: Union[Unset, str] = UNSET,
    content_encoding: Union[Unset, str] = UNSET,
    x_api_skip_invalid: Union[Unset, bool] = UNSET,
) -> Optional[Union[Error, Observation]]:
    """Create a new observation.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        x_api_monitor_id (Union[Unset, str]): Associate the new observation with the monitor_id
            that observed it.  Must be used for non-native OpenZMS observation formats.
        content_type (Union[Unset, str]): Determines the type of incoming observation.  If
            unspecified, or if set to `application/openzms+observation+v1` or `application/json`; the
            request body must be a native OpenZMS Observation JSON object.  If set to
            `application/sigmf-meta`, the request body must be a SigMF metadata file as a JSON object
            (the .sigmf-data file must be uploaded separately).  If set to `application/sigmf-
            archive`, the request body must be a SigMF archive (simple tarball with one each of
            .sigmf-meta and .sigmf-data files).
        content_encoding (Union[Unset, str]): Set to `gzip` if you have compressed the request
            body with gzip.
        x_api_skip_invalid (Union[Unset, bool]): If true, skip invalid values attached to the
            observation instead of rejecting the entire observation.
        body (Observation): Observation report from a monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Observation]
    """

    return (
        await asyncio_detailed(
            client=client,
            body=body,
            x_api_token=x_api_token,
            x_api_elaborate=x_api_elaborate,
            x_api_monitor_id=x_api_monitor_id,
            content_type=content_type,
            content_encoding=content_encoding,
            x_api_skip_invalid=x_api_skip_invalid,
        )
    ).parsed
