#!/bin/python

import time
import uuid
import os
import sys
import signal
import traceback
import argparse
import logging
import datetime
import base64
import select
import random
import datetime

from . import DefaultDestEnvAction
from zmsclient.dst.client import ZmsDstClient
from zmsclient.dst.v1.models import (Observation, ObservationKind)

LOG = logging.getLogger(__name__)

# Only when a daemon
LOGFILE = "/local/logs/zms-monitor-sim.log"

class FreqRangeDesc:
    def __init__(self, min, max, power, power_factor=None):
        self.min = min
        self.max = max
        self.power = power
        self.power_factor = power_factor

    @classmethod
    def from_string(cls, s):
        sa = s.split(",")
        if len(sa) != 4:
            raise ValueError("invalid frequency range descriptor")
        return FreqRangeDesc(*[float(x) for x in sa])

    def __repr__(self):
        return f"FreqRangeDesc(min={self.min},max={self.max},power={self.power},power_factor={self.power_factor})"

    def violate(self, shift, gain):
        return FreqRangeDesc(self.min + shift, self.max + shift, self.power + gain, self.power_factor)

#
# Create an observation.
#
def create_observation(dst_client, monitor_id, element_id,
                       band, step_size, grants=[], incumbents=[],
                       type="ota", format="csv", description="",
                       violation=False, violation_shift=None,
                       violation_gain=None, impotent=False):
    o = Observation(
        element_id=element_id, monitor_id=monitor_id,
        kind=ObservationKind.PSD, types=f"{type},sweep", format_=f"{format}-{type}",
        description=description, min_freq=int(band.min),
        max_freq=int(band.max),
        starts_at=datetime.datetime.now(datetime.timezone.utc),
        violation=violation)

    LOG.info("Creating observation: %r", o)
    if impotent:
        return

    if violation and (violation_shift or violation_gain):
        grants = [ x.violate(violation_shift, violation_gain) for x in grants ]

    data = b"frequency,power,center_freq,abovefloor,violation\n"
    center = band.min + (band.max - band.min) / 2
    for fi in range(int(band.min), int(band.max) + 1, int(step_size)):
        is_violation = 0
        power = band.power + (random.random() * 2 - 1) * band.power * band.power_factor
        for tx in grants:
            if fi < tx.min or fi > tx.max:
                continue
            txpower = tx.power + (random.random() * 2 - 1) * tx.power * tx.power_factor
            if fi < (tx.min + step_size):
                txpower -= 0.75 * (txpower - band.power)
            elif fi < (tx.min + step_size * 2):
                txpower -= 0.25 * (txpower - band.power)
            elif fi > (tx.max - step_size):
                txpower -= 0.75 * (txpower - band.power)
            elif fi > (tx.max - step_size * 2):
                txpower -= 0.25 * (txpower - band.power)
            power = max(txpower, power)
            if violation:
                is_violation = 1
        for tx in incumbents:
            if fi < tx.min or fi > tx.max:
                continue
            txpower = tx.power + (random.random() * 2 - 1) * tx.power * tx.power_factor
            if fi < (tx.min + step_size):
                txpower -= 0.75 * (txpower - band.power)
            elif fi < (tx.min + step_size * 2):
                txpower -= 0.25 * (txpower - band.power)
            elif fi > (tx.max - step_size):
                txpower -= 0.75 * (txpower - band.power)
            elif fi > (tx.max - step_size * 2):
                txpower -= 0.25 * (txpower - band.power)
            power = max(txpower, power)
        above = power - band.power
        nl = f"{float(fi)/1e6},{power},{center/1e6},{above},{is_violation}\n"
        data += nl.encode()
    o.data = base64.b64encode(data).decode()

    try:
        response = dst_client.create_observation(body=o)
        LOG.debug("created observation: %r", response)
        return response
    except:
        pass

def init_main():
    parser = argparse.ArgumentParser(
        prog="monitorsim",
        description="Send simulated power spectral density data to an OpenZMS DST.")
    parser.register("type", "FreqRangeDesc", FreqRangeDesc.from_string)
    parser.add_argument(
        "-b", "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Impotent: do not create observations in DST")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "--token", action=DefaultDestEnvAction, type=str, required=True,
        help="OpenZMS token")
    parser.add_argument(
        "--monitor-id", action=DefaultDestEnvAction, type=str, required=True,
        help="Monitor ID to associate with observation")
    parser.add_argument(
        "--element-id", action=DefaultDestEnvAction, type=str, required=True,
        help="Element ID containing monitor")
    parser.add_argument(
        "--dst-http", action=DefaultDestEnvAction, type=str, required=True,
        help="DST URL")
    parser.add_argument(
        "--type", default="ota", type=str, choices=["ota", "inline"],
        help="Specify the monitor observation type: ota or inline .")
    parser.add_argument(
        "--band", type='FreqRangeDesc', default="900e6,928e6,-120.0,0.005",
        help="A frequency range descriptor that defines the noise characteristics for the PSD: 'minfreq,maxfreq,noise-power,noise-power-rand-pct' .")
    parser.add_argument(
        "--grant", type='FreqRangeDesc', default=[], action="append",
        help="A frequency range descriptor whose range is in the band that defines power characteristics for a particular OpenZMS grant transmitter in the band: 'minfreq,maxfreq,power,power-rand-pct' .")
    parser.add_argument(
        "--incumbent", type='FreqRangeDesc', default=[], action="append",
        help="A frequency range descriptor whose range is in the band that defines power characteristics for a particular incumbent transmitter in the band: 'minfreq,maxfreq,power,power-rand-pct' .")
    parser.add_argument(
        "--step-size", type=float, default=50e3,
        help="Step size for power bins in PSD.")
    parser.add_argument(
        "--interval", type=float, default=15.0,
        help="Interval (seconds) between observations.")
    parser.add_argument(
        "--violation-shift", type=float, default=-2e6,
        help="Frequency shift (Hz) when violations are toggled on.")
    parser.add_argument(
        "--violation-gain", type=float, default=0.0,
        help="Power gain (dB) when violations are toggled on.")

    args = parser.parse_args(sys.argv[1:])

    if args.debug:
        LOG.setLevel(logging.DEBUG)
        logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
        logging.getLogger('zmsclient').setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)

    if args.band.max % args.step_size != 0:
        raise ValueError(f"band max ({args.band.max}) must be a multiple of step-size")
    for tx in [*args.grant, *args.incumbent]:
        if tx.min % args.step_size != 0:
            raise ValueError(f"tx min ({tx.min}) must be a multiple of step-size")
        if tx.max % args.step_size != 0:
            raise ValueError(f"tx max ({tx.max}) must be a multiple of step-size")

    dst_client = ZmsDstClient(args.dst_http, args.token,
                              detailed=False, raise_on_unexpected_status=True)

    return dst_client, args

def sigh_exit(signalnum, frame):
    exit(0)

violations_enabled = False
def sigh_enable_violate(*args):
    global violations_enabled
    violations_enabled = True
    LOG.info("Enabled violations")

def sigh_disable_violate(*args):
    global violations_enabled
    violations_enabled = False
    LOG.info("Disabled violations")

def main():
    dst_client, args = init_main()

    if args.daemon:
        try:
            fp = open(args.logfile, "a");
            sys.stdout = fp
            sys.stderr = fp
            sys.stdin.close()
            logging.basicConfig(stream=fp)
        except:
            print(f"Could not open log file ({args.logfile}); aborting.")
            sys.exit(1)
        pid = os.fork()
        if pid:
            sys.exit(0)
        os.setsid()
    else:
        logging.basicConfig()

    signal.signal(signal.SIGINT, sigh_exit)
    signal.signal(signal.SIGTERM, sigh_exit)
    signal.signal(signal.SIGHUP, sigh_exit)
    signal.signal(signal.SIGUSR1, sigh_enable_violate)
    signal.signal(signal.SIGUSR2, sigh_disable_violate)

    LOG.debug(f"band: {args.band}")

    while True:
        create_observation(
            dst_client, args.monitor_id, args.element_id,
            args.band, args.step_size, args.grant, args.incumbent,
            type=args.type, format="psd-csv",
            violation=violations_enabled,
            violation_shift=args.violation_shift,
            violation_gain=args.violation_gain,
            impotent=args.impotent)
        time.sleep(args.interval)

if __name__ == "__main__":
    main()
