import datetime
from typing import TYPE_CHECKING, Any, Dict, List, Type, TypeVar, Union, cast

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.series_geolocation import SeriesGeolocation
    from ..models.series_grant import SeriesGrant


T = TypeVar("T", bound="Series")


@_attrs_define
class Series:
    """A series of metric values reported from a monitor.

    Attributes:
        metric_id (str): The metric id associated with this series.
        monitor_id (str): The monitor id associated with this series.
        id (Union[Unset, str]): The id of the series.
        element_id (Union[Unset, str]): The element id associated with this series monitor.
        store_id (Union[Unset, str]): The store id in which this series is stored.
        name (Union[None, Unset, str]): The name of the series.
        description (Union[None, Unset, str]): A brief description of the series.
        is_default (Union[Unset, bool]): True if this is the default series for the metric/monitor pair.  Only one
            unfinalized, undeleted default series may exist for a given metric/monitor pair.
        starts_at (Union[None, Unset, datetime.datetime]): The time at which the first value in the series starts, if
            any values.
        ends_at (Union[None, Unset, datetime.datetime]): The time at which the last value in the series ends, if any
            values.
        creator_id (Union[Unset, str]): The user id of the creator.
        updater_id (Union[None, Unset, str]): The user id of the updater.
        created_at (Union[Unset, datetime.datetime]): Creation time.
        updated_at (Union[None, Unset, datetime.datetime]): Last modification time.
        finalized_at (Union[None, Unset, datetime.datetime]): Time at which series was finalized, if any.  Once the
            series is finalized, no new values may be added.
        deleted_at (Union[None, Unset, datetime.datetime]): Deletion time.
        geolocation (Union[None, Unset, SeriesGeolocation]): Geolocation Series for another Series.
        grants (Union[Unset, List['SeriesGrant']]): A list of grant IDs associated with this series, if any, when
            elaborate is enabled.
    """

    metric_id: str
    monitor_id: str
    id: Union[Unset, str] = UNSET
    element_id: Union[Unset, str] = UNSET
    store_id: Union[Unset, str] = UNSET
    name: Union[None, Unset, str] = UNSET
    description: Union[None, Unset, str] = UNSET
    is_default: Union[Unset, bool] = UNSET
    starts_at: Union[None, Unset, datetime.datetime] = UNSET
    ends_at: Union[None, Unset, datetime.datetime] = UNSET
    creator_id: Union[Unset, str] = UNSET
    updater_id: Union[None, Unset, str] = UNSET
    created_at: Union[Unset, datetime.datetime] = UNSET
    updated_at: Union[None, Unset, datetime.datetime] = UNSET
    finalized_at: Union[None, Unset, datetime.datetime] = UNSET
    deleted_at: Union[None, Unset, datetime.datetime] = UNSET
    geolocation: Union[None, Unset, "SeriesGeolocation"] = UNSET
    grants: Union[Unset, List["SeriesGrant"]] = UNSET
    additional_properties: Dict[str, Any] = _attrs_field(init=False, factory=dict)
    _immutable_on = {
        "id": {"put": True, "post": True},
        "metric_id": {"put": True, "post": None},
        "monitor_id": {"put": True, "post": None},
        "element_id": {"put": True, "post": None},
        "store_id": {"put": True, "post": None},
        "name": {"put": None, "post": None},
        "description": {"put": None, "post": None},
        "is_default": {"put": True, "post": None},
        "starts_at": {"put": None, "post": None},
        "ends_at": {"put": None, "post": None},
        "creator_id": {"put": True, "post": None},
        "updater_id": {"put": True, "post": True},
        "created_at": {"put": True, "post": True},
        "updated_at": {"put": True, "post": True},
        "finalized_at": {"put": None, "post": True},
        "deleted_at": {"put": True, "post": True},
        "geolocation": {"put": True, "post": True},
        "grants": {"put": True, "post": True},
    }

    def to_dict(self) -> Dict[str, Any]:
        metric_id = self.metric_id

        monitor_id = self.monitor_id

        id = self.id

        element_id = self.element_id

        store_id = self.store_id

        name: Union[None, Unset, str]
        if isinstance(self.name, Unset):
            name = UNSET
        else:
            name = self.name

        description: Union[None, Unset, str]
        if isinstance(self.description, Unset):
            description = UNSET
        else:
            description = self.description

        is_default = self.is_default

        starts_at: Union[None, Unset, str]
        if isinstance(self.starts_at, Unset):
            starts_at = UNSET
        elif isinstance(self.starts_at, datetime.datetime):
            starts_at = self.starts_at.isoformat()
        else:
            starts_at = self.starts_at

        ends_at: Union[None, Unset, str]
        if isinstance(self.ends_at, Unset):
            ends_at = UNSET
        elif isinstance(self.ends_at, datetime.datetime):
            ends_at = self.ends_at.isoformat()
        else:
            ends_at = self.ends_at

        creator_id = self.creator_id

        updater_id: Union[None, Unset, str]
        if isinstance(self.updater_id, Unset):
            updater_id = UNSET
        else:
            updater_id = self.updater_id

        created_at: Union[Unset, str] = UNSET
        if not isinstance(self.created_at, Unset):
            created_at = self.created_at.isoformat()

        updated_at: Union[None, Unset, str]
        if isinstance(self.updated_at, Unset):
            updated_at = UNSET
        elif isinstance(self.updated_at, datetime.datetime):
            updated_at = self.updated_at.isoformat()
        else:
            updated_at = self.updated_at

        finalized_at: Union[None, Unset, str]
        if isinstance(self.finalized_at, Unset):
            finalized_at = UNSET
        elif isinstance(self.finalized_at, datetime.datetime):
            finalized_at = self.finalized_at.isoformat()
        else:
            finalized_at = self.finalized_at

        deleted_at: Union[None, Unset, str]
        if isinstance(self.deleted_at, Unset):
            deleted_at = UNSET
        elif isinstance(self.deleted_at, datetime.datetime):
            deleted_at = self.deleted_at.isoformat()
        else:
            deleted_at = self.deleted_at

        geolocation: Union[None, Unset, Dict[str, Any]] = UNSET
        if self.geolocation is None:
            geolocation = None
        elif not isinstance(self.geolocation, Unset):
            geolocation = self.geolocation.to_dict()

        grants: Union[Unset, List[Dict[str, Any]]] = UNSET
        if not isinstance(self.grants, Unset):
            grants = []
            for grants_item_data in self.grants:
                grants_item = grants_item_data.to_dict()
                grants.append(grants_item)

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "metric_id": metric_id,
                "monitor_id": monitor_id,
            }
        )
        if id is not UNSET:
            field_dict["id"] = id
        if element_id is not UNSET:
            field_dict["element_id"] = element_id
        if store_id is not UNSET:
            field_dict["store_id"] = store_id
        if name is not UNSET:
            field_dict["name"] = name
        if description is not UNSET:
            field_dict["description"] = description
        if is_default is not UNSET:
            field_dict["is_default"] = is_default
        if starts_at is not UNSET:
            field_dict["starts_at"] = starts_at
        if ends_at is not UNSET:
            field_dict["ends_at"] = ends_at
        if creator_id is not UNSET:
            field_dict["creator_id"] = creator_id
        if updater_id is not UNSET:
            field_dict["updater_id"] = updater_id
        if created_at is not UNSET:
            field_dict["created_at"] = created_at
        if updated_at is not UNSET:
            field_dict["updated_at"] = updated_at
        if finalized_at is not UNSET:
            field_dict["finalized_at"] = finalized_at
        if deleted_at is not UNSET:
            field_dict["deleted_at"] = deleted_at
        if geolocation is not UNSET:
            field_dict["geolocation"] = geolocation
        if grants is not UNSET:
            field_dict["grants"] = grants

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        from ..models.series_geolocation import SeriesGeolocation
        from ..models.series_grant import SeriesGrant

        d = src_dict.copy()
        metric_id = d.pop("metric_id")

        monitor_id = d.pop("monitor_id")

        id = d.pop("id", UNSET)

        element_id = d.pop("element_id", UNSET)

        store_id = d.pop("store_id", UNSET)

        def _parse_name(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        name = _parse_name(d.pop("name", UNSET))

        def _parse_description(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        description = _parse_description(d.pop("description", UNSET))

        is_default = d.pop("is_default", UNSET)

        def _parse_starts_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                starts_at_type_0 = isoparse(data)

                return starts_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        starts_at = _parse_starts_at(d.pop("starts_at", UNSET))

        def _parse_ends_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                ends_at_type_0 = isoparse(data)

                return ends_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        ends_at = _parse_ends_at(d.pop("ends_at", UNSET))

        creator_id = d.pop("creator_id", UNSET)

        def _parse_updater_id(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        updater_id = _parse_updater_id(d.pop("updater_id", UNSET))

        _created_at = d.pop("created_at", UNSET)
        created_at: Union[Unset, datetime.datetime]
        if isinstance(_created_at, Unset):
            created_at = UNSET
        else:
            created_at = isoparse(_created_at)

        def _parse_updated_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                updated_at_type_0 = isoparse(data)

                return updated_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        updated_at = _parse_updated_at(d.pop("updated_at", UNSET))

        def _parse_finalized_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                finalized_at_type_0 = isoparse(data)

                return finalized_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        finalized_at = _parse_finalized_at(d.pop("finalized_at", UNSET))

        def _parse_deleted_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                deleted_at_type_0 = isoparse(data)

                return deleted_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        deleted_at = _parse_deleted_at(d.pop("deleted_at", UNSET))

        _geolocation = d.pop("geolocation", UNSET)
        geolocation: Union[None, Unset, SeriesGeolocation]
        if isinstance(_geolocation, Unset):
            geolocation = UNSET
        elif _geolocation is None:
            geolocation = None
        else:
            geolocation = SeriesGeolocation.from_dict(_geolocation)

        grants = []
        _grants = d.pop("grants", UNSET)
        for grants_item_data in _grants or []:
            grants_item = SeriesGrant.from_dict(grants_item_data)

            grants.append(grants_item)

        series = cls(
            metric_id=metric_id,
            monitor_id=monitor_id,
            id=id,
            element_id=element_id,
            store_id=store_id,
            name=name,
            description=description,
            is_default=is_default,
            starts_at=starts_at,
            ends_at=ends_at,
            creator_id=creator_id,
            updater_id=updater_id,
            created_at=created_at,
            updated_at=updated_at,
            finalized_at=finalized_at,
            deleted_at=deleted_at,
            geolocation=geolocation,
            grants=grants,
        )

        series.additional_properties = d
        return series

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties


def _load_lazy_imports(ns=globals()):
    from ..models.series_geolocation import SeriesGeolocation
    from ..models.series_grant import SeriesGrant

    ns["SeriesGeolocation"] = SeriesGeolocation
    ns["SeriesGrant"] = SeriesGrant
