#!/bin/python

import time
import uuid
import os
import sys
import signal
import traceback
import argparse
import logging
import datetime
import datetime
from typing import (Dict)

from zmsclient.tools import DefaultDestEnvAction
from zmsclient.dst.client import ZmsDstClient
from zmsclient.dst.v1.models import (Series)
from zmsclient.dst.v1.models import Error as DstError

from . import (MetricsHelper, BaseDataType)
from .viavi import ViaviCellAdvisorCsvDataType
from .gpx import GpxDataType
from .gnettrackpro import GNetTrackProCsvDataType
from .nemo import (NemoCellCsvDataType, NemoGpsCsvDataType)

LOG = logging.getLogger(__name__)

DATATYPES = {
    "viavi-celladvisor-csv": ViaviCellAdvisorCsvDataType,
    "gpx": GpxDataType,
    "gnettrackpro-csv": GNetTrackProCsvDataType,
    "nemo-cell-csv": NemoCellCsvDataType,
    "nemo-gps-csv": NemoGpsCsvDataType,
}

def create_argparser(datatypes: Dict[str, BaseDataType]) -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        prog="zmsclient-monitor-uploader",
        description="An OpenZMS utility to upload post hoc data from monitors (e.g., into Series or Datasets).")
    parser.add_argument(
        "-d", "--debug", default=False, action="store_true",
        help="Enable debug mode (DEBUG logging level).")
    parser.add_argument(
        "-v", "--verbose", default=False, action="store_true",
        help="Enable verbose mode (INFO logging level).")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Impotent: do not create values in DST; just parse and log what would have been done.")
    parser.add_argument(
        "--dst-http", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="DST URL")
    parser.add_argument(
        "--token", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="OpenZMS token")
    parser.add_argument(
        "--element-id", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Element ID containing monitor")
    parser.add_argument(
        "--monitor-id", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Monitor ID to associate with data")
    parser.add_argument(
        "--dataset-id", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Dataset ID to associate with data")
    parser.add_argument(
        "--series-id", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Series ID in which to upload data")
    parser.add_argument(
        "--series-name", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Name to use when creating series in which to upload data")
    parser.add_argument(
        "--series-description", action=DefaultDestEnvAction, type=str, required=False, default=None,
        help="Description to use when creating series in which to upload data")
    parser.add_argument(
        "--data-type", type=str, choices=list(datatypes.keys()),
         required=True, help="Data type")
    parser.add_argument(
        "--data-file", type=str, required=True, help="Primary data file to upload.")
    parser.add_argument(
        "--continue-on-error", default=False, action="store_true",
        help="Continue even if some values fail to parse or upload; otherwise, abort.")
    
    for dt in datatypes.values():
        dt.add_args(parser)
    
    return parser

def sigh(signalnum, frame):
    LOG.info("Exiting on signal %d", signalnum)
    exit(0)

def main():
    parser = create_argparser(DATATYPES)
    
    args = parser.parse_args(sys.argv[1:])

    logging.basicConfig()

    if args.debug:
        LOG.setLevel(logging.DEBUG)
        logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    elif args.verbose:
        LOG.setLevel(logging.INFO)
        logging.getLogger('zmsclient').setLevel(logging.INFO)
    else:
        logging.getLogger().setLevel(logging.WARN)

    LOG.debug(f"args: {args}")

    if not args.impotent:
        if not args.dst_http:
            raise ValueError("A --dst-http must be specified")
        if not args.token:
            raise ValueError("A --token must be specified")
        if not args.element_id:
            raise ValueError("A --element-id must be specified")
        if not args.monitor_id:
            raise ValueError("A --monitor-id must be specified")
        if not args.element_id:
            raise ValueError("A --element-id must be specified")
        if not args.series_id and not args.series_name:
            raise ValueError("Either --series-id or --series-name must be specified")
    else:
        if not args.element_id:
            args.element_id = str(uuid.uuid4())
            LOG.info(f"Using random element ID {args.element_id}")
        if not args.monitor_id:
            args.monitor_id = str(uuid.uuid4())
            LOG.info(f"Using random monitor ID {args.monitor_id}")

    if not args.data_type and not args.data_file:
        raise ValueError("--data-type and --data-file must be specified")

    signal.signal(signal.SIGINT, sigh)
    signal.signal(signal.SIGTERM, sigh)
    signal.signal(signal.SIGHUP, sigh)

    client = ZmsDstClient(
        args.dst_http, args.token,
        detailed=False, raise_on_unexpected_status=True)
    mh = MetricsHelper(
        client, monitor_id=args.monitor_id, element_id=args.element_id,
        impotent=args.impotent)
    dt = DATATYPES[args.data_type](args, mh)

    # Load values from the data file.
    #
    # NB: load_data should simply parse and cache the data.  We have not yet
    # created or fetched the necessary metrics (and possibly series), so there
    # is not enough information to create proper Values -- they will not have
    # metric_id nor series_id set validly.  Those fields must be set in the
    # prepare_values() method.
    dt.load_data()
    # NB: both MetricsHelper and the DataType subclass must check for impotent
    # mode!
    dt.init_metrics(create_if_missing=True)
    dt.init_series(create_if_missing=True)
    dt.prepare_values()
    dt.upload_data()

if __name__ == "__main__":
    main()
