import datetime
from typing import TYPE_CHECKING, Any, Dict, List, Type, TypeVar, Union, cast

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.any_object import AnyObject
    from ..models.monitor_task_state import MonitorTaskState


T = TypeVar("T", bound="MonitorTask")


@_attrs_define
class MonitorTask:
    """An task scheduled by OpenZMS and run by a specific monitor.

    Attributes:
        monitor_action_id (str): The id of the associated monitor action.
        id (Union[Unset, str]): The id of the monitor task resource.
        monitor_id (Union[Unset, str]): The id of the associated monitor.
        parameters (Union[None, Unset, AnyObject]):
        start (Union[None, Unset, datetime.datetime]): The time when the task should start.  If unspecified, equivalent
            to `now`.  May be a loose suggestion of a desired but not required start time, further specified by the hard
            `start_by` constraint.
        duration (Union[None, Unset, float]): The total task duration (seconds).  If interval is set, duration limits
            the number of repeat task executions.  If interval is not set, duration limits the execution time of the single
            task execution.  Duration is intended to impose a wall clock time limit on total task execution, including
            repeats if interval is set.  It is not intended to take the place of lower-level durations *within* a task
            execution, such as dwell time at a single center frequency during a sweep; those should be exposed as action
            parameter defs.
        interval (Union[None, Unset, float]): The interval (seconds); signals that this task should be repeated until
            deleted, or until start + duration is reached.  If unspecified, the action must run only once.
        creator_id (Union[Unset, str]): The user id of the creator.
        created_at (Union[Unset, datetime.datetime]): Creation time.
        deleted_at (Union[None, Unset, datetime.datetime]): Deletion time.
        state (Union[None, Unset, MonitorTaskState]): The state of a monitor task.
    """

    monitor_action_id: str
    id: Union[Unset, str] = UNSET
    monitor_id: Union[Unset, str] = UNSET
    parameters: Union[None, Unset, "AnyObject"] = UNSET
    start: Union[None, Unset, datetime.datetime] = UNSET
    duration: Union[None, Unset, float] = UNSET
    interval: Union[None, Unset, float] = UNSET
    creator_id: Union[Unset, str] = UNSET
    created_at: Union[Unset, datetime.datetime] = UNSET
    deleted_at: Union[None, Unset, datetime.datetime] = UNSET
    state: Union[None, Unset, "MonitorTaskState"] = UNSET
    additional_properties: Dict[str, Any] = _attrs_field(init=False, factory=dict)
    _immutable_on = {
        "id": {"put": True, "post": True},
        "monitor_id": {"put": True, "post": None},
        "monitor_action_id": {"put": True, "post": None},
        "parameters": {"put": None, "post": None},
        "start": {"put": True, "post": None},
        "duration": {"put": True, "post": None},
        "interval": {"put": True, "post": None},
        "creator_id": {"put": True, "post": True},
        "created_at": {"put": True, "post": True},
        "deleted_at": {"put": True, "post": True},
        "state": {"put": True, "post": True},
    }

    def to_dict(self) -> Dict[str, Any]:
        monitor_action_id = self.monitor_action_id

        id = self.id

        monitor_id = self.monitor_id

        parameters: Union[None, Unset, Dict[str, Any]] = UNSET
        if self.parameters is None:
            parameters = None
        elif not isinstance(self.parameters, Unset):
            parameters = self.parameters.to_dict()

        start: Union[None, Unset, str]
        if isinstance(self.start, Unset):
            start = UNSET
        elif isinstance(self.start, datetime.datetime):
            start = self.start.isoformat()
        else:
            start = self.start

        duration: Union[None, Unset, float]
        if isinstance(self.duration, Unset):
            duration = UNSET
        else:
            duration = self.duration

        interval: Union[None, Unset, float]
        if isinstance(self.interval, Unset):
            interval = UNSET
        else:
            interval = self.interval

        creator_id = self.creator_id

        created_at: Union[Unset, str] = UNSET
        if not isinstance(self.created_at, Unset):
            created_at = self.created_at.isoformat()

        deleted_at: Union[None, Unset, str]
        if isinstance(self.deleted_at, Unset):
            deleted_at = UNSET
        elif isinstance(self.deleted_at, datetime.datetime):
            deleted_at = self.deleted_at.isoformat()
        else:
            deleted_at = self.deleted_at

        state: Union[None, Unset, Dict[str, Any]] = UNSET
        if self.state is None:
            state = None
        elif not isinstance(self.state, Unset):
            state = self.state.to_dict()

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "monitor_action_id": monitor_action_id,
            }
        )
        if id is not UNSET:
            field_dict["id"] = id
        if monitor_id is not UNSET:
            field_dict["monitor_id"] = monitor_id
        if parameters is not UNSET:
            field_dict["parameters"] = parameters
        if start is not UNSET:
            field_dict["start"] = start
        if duration is not UNSET:
            field_dict["duration"] = duration
        if interval is not UNSET:
            field_dict["interval"] = interval
        if creator_id is not UNSET:
            field_dict["creator_id"] = creator_id
        if created_at is not UNSET:
            field_dict["created_at"] = created_at
        if deleted_at is not UNSET:
            field_dict["deleted_at"] = deleted_at
        if state is not UNSET:
            field_dict["state"] = state

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        from ..models.any_object import AnyObject
        from ..models.monitor_task_state import MonitorTaskState

        d = src_dict.copy()
        monitor_action_id = d.pop("monitor_action_id")

        id = d.pop("id", UNSET)

        monitor_id = d.pop("monitor_id", UNSET)

        _parameters = d.pop("parameters", UNSET)
        parameters: Union[None, Unset, AnyObject]
        if isinstance(_parameters, Unset):
            parameters = UNSET
        elif _parameters is None:
            parameters = None
        else:
            parameters = AnyObject.from_dict(_parameters)

        def _parse_start(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                start_type_0 = isoparse(data)

                return start_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        start = _parse_start(d.pop("start", UNSET))

        def _parse_duration(data: object) -> Union[None, Unset, float]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, float], data)

        duration = _parse_duration(d.pop("duration", UNSET))

        def _parse_interval(data: object) -> Union[None, Unset, float]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, float], data)

        interval = _parse_interval(d.pop("interval", UNSET))

        creator_id = d.pop("creator_id", UNSET)

        _created_at = d.pop("created_at", UNSET)
        created_at: Union[Unset, datetime.datetime]
        if isinstance(_created_at, Unset):
            created_at = UNSET
        else:
            created_at = isoparse(_created_at)

        def _parse_deleted_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                deleted_at_type_0 = isoparse(data)

                return deleted_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        deleted_at = _parse_deleted_at(d.pop("deleted_at", UNSET))

        _state = d.pop("state", UNSET)
        state: Union[None, Unset, MonitorTaskState]
        if isinstance(_state, Unset):
            state = UNSET
        elif _state is None:
            state = None
        else:
            state = MonitorTaskState.from_dict(_state)

        monitor_task = cls(
            monitor_action_id=monitor_action_id,
            id=id,
            monitor_id=monitor_id,
            parameters=parameters,
            start=start,
            duration=duration,
            interval=interval,
            creator_id=creator_id,
            created_at=created_at,
            deleted_at=deleted_at,
            state=state,
        )

        monitor_task.additional_properties = d
        return monitor_task

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties


def _load_lazy_imports(ns=globals()):
    from ..models.any_object import AnyObject
    from ..models.monitor_task_state import MonitorTaskState

    ns["AnyObject"] = AnyObject
    ns["MonitorTaskState"] = MonitorTaskState
