from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.monitor_task import MonitorTask
from ...types import UNSET, Response, Unset


def _get_kwargs(
    monitor_id: str,
    *,
    body: MonitorTask,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    headers["X-Api-Token"] = x_api_token

    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    _kwargs: Dict[str, Any] = {
        "method": "post",
        "url": "/monitors/{monitor_id}/tasks".format(
            monitor_id=monitor_id,
        ),
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, MonitorTask]]:
    if response.status_code == HTTPStatus.CREATED:
        try:
            response_201 = MonitorTask.from_dict(response.json())

            return response_201
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, MonitorTask]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    monitor_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: MonitorTask,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, MonitorTask]]:
    """Create new monitor task.

    Args:
        monitor_id (str): The target monitor.
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        body (MonitorTask): An task scheduled by OpenZMS and run by a specific monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, MonitorTask]]
    """

    kwargs = _get_kwargs(
        monitor_id=monitor_id,
        body=body,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    monitor_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: MonitorTask,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, MonitorTask]]:
    """Create new monitor task.

    Args:
        monitor_id (str): The target monitor.
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        body (MonitorTask): An task scheduled by OpenZMS and run by a specific monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, MonitorTask]
    """

    return sync_detailed(
        monitor_id=monitor_id,
        client=client,
        body=body,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    ).parsed


async def asyncio_detailed(
    monitor_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: MonitorTask,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, MonitorTask]]:
    """Create new monitor task.

    Args:
        monitor_id (str): The target monitor.
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        body (MonitorTask): An task scheduled by OpenZMS and run by a specific monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, MonitorTask]]
    """

    kwargs = _get_kwargs(
        monitor_id=monitor_id,
        body=body,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    monitor_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: MonitorTask,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, MonitorTask]]:
    """Create new monitor task.

    Args:
        monitor_id (str): The target monitor.
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):
        body (MonitorTask): An task scheduled by OpenZMS and run by a specific monitor.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, MonitorTask]
    """

    return (
        await asyncio_detailed(
            monitor_id=monitor_id,
            client=client,
            body=body,
            x_api_token=x_api_token,
            x_api_elaborate=x_api_elaborate,
        )
    ).parsed
