from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.location_list import LocationList
from ...types import UNSET, Response, Unset


def _get_kwargs(
    *,
    element_id: Union[Unset, str] = UNSET,
    name: Union[Unset, str] = UNSET,
    deleted: Union[Unset, bool] = False,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    headers["X-Api-Token"] = x_api_token

    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    params: Dict[str, Any] = {}

    params["element_id"] = element_id

    params["name"] = name

    params["deleted"] = deleted

    params["page"] = page

    params["items_per_page"] = items_per_page

    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    _kwargs: Dict[str, Any] = {
        "method": "get",
        "url": "/locations",
        "params": params,
    }

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, LocationList]]:
    if response.status_code == HTTPStatus.OK:
        try:
            response_200 = LocationList.from_dict(response.json())

            return response_200
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, LocationList]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    element_id: Union[Unset, str] = UNSET,
    name: Union[Unset, str] = UNSET,
    deleted: Union[Unset, bool] = False,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, LocationList]]:
    """Retrieve a list of locations.

    Args:
        element_id (Union[Unset, str]): Filters by element.
        name (Union[Unset, str]): Filters by name.
        deleted (Union[Unset, bool]): If true, show only deleted locations. Default: False.
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, LocationList]]
    """

    kwargs = _get_kwargs(
        element_id=element_id,
        name=name,
        deleted=deleted,
        page=page,
        items_per_page=items_per_page,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    *,
    client: Union[AuthenticatedClient, Client],
    element_id: Union[Unset, str] = UNSET,
    name: Union[Unset, str] = UNSET,
    deleted: Union[Unset, bool] = False,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, LocationList]]:
    """Retrieve a list of locations.

    Args:
        element_id (Union[Unset, str]): Filters by element.
        name (Union[Unset, str]): Filters by name.
        deleted (Union[Unset, bool]): If true, show only deleted locations. Default: False.
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, LocationList]
    """

    return sync_detailed(
        client=client,
        element_id=element_id,
        name=name,
        deleted=deleted,
        page=page,
        items_per_page=items_per_page,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    ).parsed


async def asyncio_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    element_id: Union[Unset, str] = UNSET,
    name: Union[Unset, str] = UNSET,
    deleted: Union[Unset, bool] = False,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, LocationList]]:
    """Retrieve a list of locations.

    Args:
        element_id (Union[Unset, str]): Filters by element.
        name (Union[Unset, str]): Filters by name.
        deleted (Union[Unset, bool]): If true, show only deleted locations. Default: False.
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, LocationList]]
    """

    kwargs = _get_kwargs(
        element_id=element_id,
        name=name,
        deleted=deleted,
        page=page,
        items_per_page=items_per_page,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    *,
    client: Union[AuthenticatedClient, Client],
    element_id: Union[Unset, str] = UNSET,
    name: Union[Unset, str] = UNSET,
    deleted: Union[Unset, bool] = False,
    page: Union[Unset, int] = UNSET,
    items_per_page: Union[Unset, int] = UNSET,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, LocationList]]:
    """Retrieve a list of locations.

    Args:
        element_id (Union[Unset, str]): Filters by element.
        name (Union[Unset, str]): Filters by name.
        deleted (Union[Unset, bool]): If true, show only deleted locations. Default: False.
        page (Union[Unset, int]):
        items_per_page (Union[Unset, int]):
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, LocationList]
    """

    return (
        await asyncio_detailed(
            client=client,
            element_id=element_id,
            name=name,
            deleted=deleted,
            page=page,
            items_per_page=items_per_page,
            x_api_token=x_api_token,
            x_api_elaborate=x_api_elaborate,
        )
    ).parsed
