from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.subscription_list import SubscriptionList
from ...types import UNSET, Response, Unset


def _get_kwargs(
    *,
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    headers["X-Api-Token"] = x_api_token

    if not isinstance(x_api_elaborate, Unset):
        headers["X-Api-Elaborate"] = x_api_elaborate

    _kwargs: Dict[str, Any] = {
        "method": "get",
        "url": "/subscriptions",
    }

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, SubscriptionList]]:
    if response.status_code == HTTPStatus.OK:
        try:
            response_200 = SubscriptionList.from_dict(response.json())

            return response_200
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, SubscriptionList]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, SubscriptionList]]:
    """Retrieve a list of subscriptions; admin only.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, SubscriptionList]]
    """

    kwargs = _get_kwargs(
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    *,
    client: Union[AuthenticatedClient, Client],
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, SubscriptionList]]:
    """Retrieve a list of subscriptions; admin only.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, SubscriptionList]
    """

    return sync_detailed(
        client=client,
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    ).parsed


async def asyncio_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Response[Union[Error, SubscriptionList]]:
    """Retrieve a list of subscriptions; admin only.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, SubscriptionList]]
    """

    kwargs = _get_kwargs(
        x_api_token=x_api_token,
        x_api_elaborate=x_api_elaborate,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    *,
    client: Union[AuthenticatedClient, Client],
    x_api_token: str,
    x_api_elaborate: Union[Unset, str] = UNSET,
) -> Optional[Union[Error, SubscriptionList]]:
    """Retrieve a list of subscriptions; admin only.

    Args:
        x_api_token (str):
        x_api_elaborate (Union[Unset, str]):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, SubscriptionList]
    """

    return (
        await asyncio_detailed(
            client=client,
            x_api_token=x_api_token,
            x_api_elaborate=x_api_elaborate,
        )
    ).parsed
