#!/bin/python

import time
import uuid
import os
import os.path
import sys
import signal
import traceback
import argparse
import logging
import asyncio
import shlex
import subprocess
import datetime
import dateutil.parser
import threading
import socket
import re

from zmsclient.zmc.client import ZmsZmcClient
from zmsclient.zmc.v1.models import (
    Subscription, EventFilter, Error,
    UpdateGrantOpStatus, GrantOpStatus,
    GrantConstraint, Constraint)    
from zmsclient.zmc.v1.models import Grant as ZMSGrant
from zmsclient.common.subscription import ZmsSubscriptionCallback
import manifest

HOSTNAME = None
LOG = None
EXTID = None

# Not sure where to pick these up
EVENT_TYPE_REPLACED    =     1
EVENT_TYPE_CREATED     =     2
EVENT_SOURCETYPE_ZMC   =     2
EVENT_CODE_GRANT       =     2006

# Only when a daemon
LOGFILE = "/local/logs/tx-wrapper.log"

# Command to run.
TXCMDTEST = "/bin/sleep 1000"
if os.path.exists("/usr/lib/uhd/examples/tx_waveforms"):
    TXWAVEFORMS = "/usr/lib/uhd/examples/tx_waveforms"
elif os.path.exists("/usr/libexec/uhd/examples/tx_waveforms"):
    TXWAVEFORMS = "/usr/libexec/uhd/examples/tx_waveforms"
else:
    TXWAVEFORMS = "tx_waveforms"
TXCMD = TXWAVEFORMS + " --ant TX/RX --channel 0 " +\
    "--rate %r --gain 50 --freq %r "

#
# This needs to be in a library someplace.
#
class DefaultDestEnvAction(argparse.Action):
    def __init__(self, option_strings, dest, required=True, default=None, **kwargs):
        """An argparse.Action that initializes the default value of the arg from an environment variable named `dest.upper()` (where dest is the storage location of the value post-parse, e.g. `args.dest`); and, if the arg was required, *unsets* it from being required, so that argparse does not fail the parse if the argument is not supplied.  This is certainly a bit unfortunate since it changes the helptext behavior, but nothing to do about that."""
        dest_upper = dest.upper()
        if dest_upper in os.environ:
            default = os.environ[dest_upper]
        if required and default:
            required = False
        super(DefaultDestEnvAction, self).__init__(
            option_strings, dest, default=default, required=required, **kwargs)

    def __call__(self, parser, namespace, values, option_string=None):
        setattr(namespace, self.dest, values)
# Ditto
class FreqRangeDesc:
    def __init__(self, min, max, eirp, width=None):
        self.min = min
        self.max = max
        self.eirp = eirp
        self.width = width

    @classmethod
    def from_string(cls, s):
        sa = s.split(",")
        if len(sa) != 3 and len(sa) != 4:
            raise ValueError("invalid frequency range descriptor")

        width = None
        if len(sa) == 4:
            width = int(sa[3])
            pass
        
        return FreqRangeDesc(*[float(x) for x in sa[0:3]], width)

    def __repr__(self):
        return f"FreqRangeDesc(min={self.min},max={self.max},eirp={self.eirp},width={self.width})"
        
#
# Initial load of the grant, chasing the replacement chain to the
# current (active) grant.
#
def loadGrant(zmc_client, grant_id):
    LOG.info("Loading initial grant %r", grant_id)
    grant = zmc_client.get_grant(grant_id, elaborate=True)
    if not grant:
        LOG.error("Could not get grant: %r " % (grant_id,))
        sys.exit(-1)
        pass

    while grant.replacement:
        grant_id = grant.replacement.new_grant_id
        LOG.info("Loading next grant %r", grant_id)
        grant = zmc_client.get_grant(grant_id, elaborate=True)
        if not grant:
            LOG.error("Could not get grant: %r " % (grant_id,))
            sys.exit(-1)
            pass
        pass

    return grant

#
# Wrap the ZMS grant.
#
class Grant():
    def __init__(self, zmcclient, grant, args, lock, condition):
        self.lock = lock
        self.condition = condition
        self._grant = grant
        self.zmcclient = zmcclient
        self.args = args
        self.delete = not grant
        pass

    @property
    def grant(self):
        return self._grant

    @grant.setter
    def grant(self, grant):
        self._grant = grant
        pass

    @property
    def constraints(self):
        return self.grant.constraints

    @property
    def status(self):
        return self.grant.status

    @property
    def id(self):
        return self.grant.id
    
    @property
    def replacement(self):
        return self.grant.replacement
    
    @property
    def expires_at(self):
        return self.grant.expires_at

    @property
    def status_ack_by(self):
        return self.grant.status_ack_by

    def revoked(self):
        return self._grant == None

    def createGrant(self):
        starts = datetime.datetime.now(datetime.timezone.utc)
        expires = starts + datetime.timedelta(hours=2)
        constraint = Constraint(
            min_freq = int(self.args.band.min),
            max_freq = int(self.args.band.max),
            max_eirp = self.args.band.eirp,
            exclusive = True)
        if self.args.band.width:
            constraint.bandwidth = int(self.args.band.width)
            pass
        grantreq = ZMSGrant(
            element_id = self.args.element_id,
            name = HOSTNAME,
            description = "Powder Experiment " + HOSTNAME,
            starts_at = starts,
            expires_at = expires,
            constraints = [ GrantConstraint(constraint=constraint) ],
            allow_skip_acks = not self.args.heartbeat,
        )
        if EXTID:
            grantreq.ext_id = EXTID
            pass

        try:
            grant = self.zmcclient.create_grant(body=grantreq)
        except Exception as ex:
            LOG.exception(ex)
            LOG.error("Failed to create_grant at the ZMC")
            sys.exit(-1)
            pass
        if type(grant) is Error:
            LOG.error("Create Grant Error: %r" % grant)
            sys,exit(-1)

        #
        # There can be a delay, so lets poll for a bit until it goes
        # active or is denied/deleted.
        #
        LOG.info("Waiting for grant to be approved")
        for x in range(30):
            time.sleep(2)
            try:
                grant = self.zmcclient.get_grant(grant.id, elaborate=True)
            except Exception as ex:
                LOG.exception(ex)
                LOG.error("Failed to get_grant at the ZMC, waiting ...")
                continue
            if grant.status in ["active", "denied", "deleted", "pending"]:
                break
            pass
        if grant.status != "active":
            sys.exit("Failed to get initial grant %r" % grant.status)
            pass
        
        self._grant = grant
        return self

    def deleteGrant(self):
        if not self.delete:
            return
        
        with self.lock:
            if not self.grant:
                return
            LOG.info("Deleting grant")
            response = self.zmcclient.delete_grant(
                grant_id=self.grant.id)
            if type(response) is Error:
                LOG.error("Could not delete grant: %r", response)
                return
            self._grant = None
            pass
        pass
        
    def sendOpStatus(self):
        with self.lock:
            if not self.grant:
                return
            
            LOG.info("sendOpStatus: %r", self.grant.status)
            opstatus = self.grant.status
            if opstatus == "pending" or opstatus == "revoked":
                opstatus = "paused"
                pass
            
            try:
                response = self.zmcclient.update_grant_op_status(
                    grant_id=self.grant.id,
                    body=UpdateGrantOpStatus(GrantOpStatus(opstatus)))
            except Exception as ex:
                LOG.exception(ex)
                LOG.info("sendOpStatus failed")
                return
            
            if type(response) is Error:
                LOG.error("Bad response to opstatus update: %r", response)
                return;
            ackby = response.status_ack_by
            LOG.info("sendOpStatus: next ackby: %s",
                     ackby.astimezone().strftime("%Y-%m-%d %H:%M:%S"))
            self.grant.status_ack_by = ackby
            pass
        pass

    def updateStatus(self, newstatus, force=False):
        with self.lock:
            oldstatus = self.grant.status
            self.grant.status = newstatus
            LOG.info("updateStatus: %r %r", oldstatus, newstatus)
            if force or oldstatus != newstatus:
                self.sendOpStatus()
                pass
            if newstatus == "revoked":
                self._grant = None
            pass
        pass

    def replace(self):
        grantreq = ZMSGrant.from_dict(self.grant.to_dict());
        grantreq.constraints[0].constraint.min_freq = int(self.args.band.min)
        grantreq.constraints[0].constraint.max_freq = int(self.args.band.max)

        try:
            response = self.zmcclient.replace_grant(
                grant_id=self.grant.id, body=grantreq)
        except Exception as ex:
            LOG.exception(ex)
            LOG.info("Replace grant failed")
            return

        if type(response) is Error:
            LOG.error("Bad response to grant replace: %r", response)
            return;
        LOG.info("Replace grant: %r", response)
        pass
    pass

#
# Optional heartbeat protocol handling with the ZMS
#
class HeartBeat():
    def __init__(self, grant):
        super(HeartBeat, self).__init__()
        self.grant = grant
        self.done = False
        pass
    
    async def start(self):
        LOG.info("Heartbeat starting")
        lastackby = 0

        self.grant.sendOpStatus()       
        while not self.done:
            # We can just slowly spin when nothing to do.
            if self.grant.revoked():
                await asyncio.sleep(5)
                lastackby = 0
                continue
            
            #
            # Count down till when the next heartbeat needs to go.
            #
            ackby    = self.grant.status_ack_by
            now      = datetime.datetime.now(datetime.timezone.utc)
            if ackby == lastackby:
                # Something went wrong last update. Keep trying
                seconds = 10
            else:
                lastackby = ackby
                duration  = ackby - now
                seconds   = duration.total_seconds()
                pass

            if seconds >= 50:
                seconds = seconds - 30
                pass

            LOG.info("Heartbeat sleeping for %d seconds", seconds)
            while seconds > 0:
                await asyncio.sleep(2)
                if self.done:
                    break
                seconds -= 2
                pass
            LOG.info("Heartbeat done sleeping")
            self.grant.sendOpStatus()
            if self.done:
                break;
            pass
        LOG.info("Heartbeat is exiting")
        pass

    def stop(self):
        LOG.info("HeartBeat stopping")
        self.done = True
        pass

    pass

#
# Subclass ZmsSubscriptionCallback, to handle grant events.
#
class ZMCSubscriptionCallback(ZmsSubscriptionCallback):
    def __init__(self, zmcclient, element_id, 
                 args=None, grant=None, lock=None, condition=None, **kwargs):
        super(ZMCSubscriptionCallback, self).__init__(zmcclient, **kwargs)
        self.ZMC = zmcclient
        self.element_id = element_id
        self.args = args
        self.impotent = args.impotent
        self.grant = grant
        self.runstate = "stopped"
        self.child = None
        self.grant_id = grant.id

    async def start(self):
        if self.grant.status == "active":
            await self.startTransmitter()
            if self.runstate == "running":
                self.grant.updateStatus(self.grant.status, force=True);
                pass
            pass
        try:
            await self.run_callbacks()
        except Exception as ex:
            LOG.exception(ex)
            await self.stopTransmitter()
            pass
        pass

    async def stop(self):
        if self.child:
            await self.stopTransmitter()
            pass
        self.grant.deleteGrant()
        pass

    async def on_event(self, ws, evt, message):
        if evt.header.source_type != EVENT_SOURCETYPE_ZMC:
            LOG.error("on_event: unexpected source type: %r (%r)",
                      evt.header.source_type, message)
            return

        # Since we are subscribed to all ZMC events for this element,
        # there will be chatter we do not care about.
        if (evt.header.code != EVENT_CODE_GRANT):
            return

        try:
            await self.handleEvent(evt.object_)
        except Exception as ex:
            LOG.exception(ex)
            pass
        pass

    async def startTransmitter(self):
        min_freq = self.grant.constraints[0].constraint.min_freq
        max_freq = self.grant.constraints[0].constraint.max_freq

        if self.args.impotent:
            command = TXCMDTEST
        elif self.args.txcommand:
            # For external script
            os.environ["TXMIN_FREQ"] = str(min_freq)
            os.environ["TXMAX_FREQ"] = str(max_freq)
            command = self.args.txcommand + " start"
        else:
            freq = min_freq + ((max_freq - min_freq) / 2)
            rate = (max_freq - min_freq) / 4
            command = TXCMD % (rate, freq)
            pass
        LOG.info(" %r", command)
        
        if self.child:
            LOG.info(" Stopping child that should not be running")
            self.child.terminate()
            await self.child.wait()
            self.child = None
            pass
        fp = open("/tmp/tx.log", "w")
        self.child = await asyncio.create_subprocess_exec(
            *(shlex.split(command)), stdout=fp, stderr=subprocess.STDOUT)
        if self.child:
            # Give the process time to start.
            try:
                await asyncio.wait_for(self.child.wait(), timeout=3)
            except asyncio.TimeoutError:
                pass
            
            if self.child.returncode != None:
                LOG.error("Could not start TX process")
                self.child = None
                return
            pass
        self.runstate = "running"
        pass

    async def stopTransmitter(self):
        LOG.info("stopTransmitter")
        if not self.child:
            LOG.error(" Child is not running")
            return

        if self.args.txcommand:
            result = subprocess.run(
                [self.args.txcommand, 'stop'], capture_output=True, text=True)
            if result.returncode:
                LOG.error(result.stderr)
                LOG.error("External script failed to stop transmitter. Exiting")
                raise asyncio.CancelledError
        else:
            self.child.terminate()
            pass

        try:
            await asyncio.wait_for(self.child.wait(), timeout=5)
        except TimeoutExpired:
            LOG.error("Failed to stop transmitter. Exiting")
            raise asyncio.CancelledError

        self.child = None
        self.runstate = "stopped"
        pass

    async def handleEvent(self, grant):
        old_grant_id = grant.id
        LOG.info("Grant event: %r - %r", old_grant_id, grant.status)

        #
        # Look for stop/start not related to grant replacement.
        #
        if not grant.replacement or not grant.replacement.new_grant_id:
            LOG.info("  Not a replacement grant")

            # Has to be an event for our current grant.
            if old_grant_id != self.grant.id:
                LOG.info("  Not our current grant, ignoring")
                return
 
            # Ignore the "replacing" state, has no meaning here.
            if grant.status == "replacing":
                LOG.info("  Ignoring replacing state")
                return

            if os.path.exists("/tmp/violate"):
                LOG.error("  Forcing a violation!")
                return

            if grant.status == "active" and self.runstate == "stopped":
                await self.startTransmitter()
                if self.runstate == "running":
                    self.grant.updateStatus(grant.status);
                    pass
            elif (grant.status in ["pending", "paused", "revoked"] and
                  self.runstate == "running"):
                await self.stopTransmitter()
                if self.runstate == "stopped" and not self.args.replace_on_pending:
                    self.grant.updateStatus(grant.status);
                    pass
                if self.args.replace_on_pending:
                    if grant.status == "pending":
                        self.grant.replace()
                    else:
                        self.grant.updateStatus(grant.status);
                        pass
                    pass
                pass
            return

        #
        # Grant replacement. Might be an extension or a change in spectrum.
        # Make sure it is for our grant.
        #
        if grant.replacement.grant_id != self.grant.id:
            LOG.info("Ignoring grant replacement for another grant: %r",
                     grant.replacement.grant_id)
            return
        
        new_grant_id = grant.replacement.new_grant_id
        grant = None
        LOG.info("Grant Replacement: %r -> %r", old_grant_id, new_grant_id)

        #
        # Since we loop for it to go active, ignore any events that
        # come in after that.
        #
        if new_grant_id == self.grant.id:
            LOG.info("  Our current grant, ignoring")
            return

        #
        # There can be a delay, so lets poll for a bit until it goes
        # active or is denied/deleted.
        #
        for x in range(30):
            time.sleep(2)
            grant = self.ZMC.get_grant(new_grant_id, elaborate=True)
            if grant.status in ["active", "denied", "deleted", "pending"]:
                break
            pass

        #
        # We get a new grant (replacement) even if the grant extension is
        # denied. We have to ignore that.
        #
        if grant.status != "active":
            LOG.error("  The replaced grant was not approved!")
            LOG.error("  Ignoring grant change")
            return

        LOG.info("  Expiration: %s", grant.expires_at)

        #
        # Lets see if the range changed. If no change, then just make
        # sure the transmitter is running. Otherwise, stop and restart
        # on new range.
        #
        old_min_freq = self.grant.constraints[0].constraint.min_freq
        old_max_freq = self.grant.constraints[0].constraint.max_freq
        new_min_freq = grant.constraints[0].constraint.min_freq
        new_max_freq = grant.constraints[0].constraint.max_freq
        LOG.info("  Old frequency min/max: %r/%r", old_min_freq, old_max_freq)
        LOG.info("  New frequency min/max: %r/%r", new_min_freq, new_max_freq)

        if os.path.exists("/tmp/violate"):
            LOG.error("  Forcing a violation!")
            return
        
        # Update our grant since it has changed internally.
        self.grant.grant = grant
        if self.runstate == "stopped":
            await self.startTransmitter()
            if self.runstate == "running":
                self.grant.updateStatus(grant.status);
                pass
            return

        if new_min_freq != old_min_freq or new_max_freq != old_max_freq:
            await self.stopTransmitter()
            await self.startTransmitter()
            self.grant.updateStatus(grant.status);
            return
        pass

# The hander has to be outside the async main.
def set_signal_handler(signum, task_to_cancel):
    def handler(_signum, _frame):
        asyncio.get_running_loop().call_soon_threadsafe(task_to_cancel.cancel)
    signal.signal(signum, handler)

def init_main():
    parser = argparse.ArgumentParser(
        prog="tx_wrapper",
        description="Handle grant change events for a transmitter")
    parser.add_argument(
        "-b", "--daemon", default=False, action="store_true",
        help="Daemonize")
    parser.add_argument(
        "-d", "--debug", default=0, action="count",
        help="Increase debug level: defaults to INFO; add once for zmsclient DEBUG; add twice to set the root logger level to DEBUG")
    parser.add_argument(
        "-n", "--impotent", default=False, action="store_true",
        help="Do not actually transmit, just play pretend.")
    parser.add_argument(
        "--logfile", default=LOGFILE, type=str,
        help="Redirect logging to a file when daemonizing.")
    parser.add_argument(
        "--txcommand", type=str,
        help="Optional external program to start and stop the radio transmitter. minimum and maximum frequencies are supplied as the environment variables (TXMAX_FREQ,TXMAX_FREQ). The script should expect a single argument, 'start' or 'stop'. Otherwise a builtin Powder specific command line is used.")
    
    parser.add_argument(
        "--cli", default=False, action="store_true",
        help="CLI mode instead of relying on a Powder experiment")
    parser.add_argument(
        "--grant-id", action=DefaultDestEnvAction, type=str, required=False,
        help="Grant ID that needs to watched")
    parser.register("type", "FreqRangeDesc", FreqRangeDesc.from_string)
    parser.add_argument(
        "--band", type='FreqRangeDesc', default="3360e6,3550e6,-70.0,10000",
        help="A frequency range descriptor that defines band characteristics: 'minfreq,maxfreq,max_eirp,width' (width is optional provide an empty string).")
    parser.add_argument(
        "--element-id", action=DefaultDestEnvAction, type=str, required=False,
        help="Local element ID")
    parser.add_argument(
        "--element-token", action=DefaultDestEnvAction, type=str, required=False,
        help="Local element token")
    parser.add_argument(
        "--zmc-http", action=DefaultDestEnvAction, type=str, required=False,
        help="Local ZMC URL")
    parser.add_argument(
        "--element-userid", action=DefaultDestEnvAction, type=str, required=False,
        help="User id that is bound to the element token. "+
        "Not required if using an admin token")
    parser.add_argument(
        "--replace-on-pending", default=False, action="store_true",
        help="Try to replace current grant when paused")
    parser.add_argument(
        "--heartbeat", default=False, action="store_true",
        help="Enable ZMC heartbeat protocol")

    #
    # Need these from either the command line or the manifest
    #
    element_id = None
    element_token = None
    element_userid = None
    zmc_http = None
    grant_id = None

    args = parser.parse_args(sys.argv[1:])

    if args.cli:
        if not (args.element_id and args.element_token and args.zmc_http):
            sys.exit("Please supply all ZMC arguments")
            pass

        if args.replace_on_pending and not args.band:
            sys.exit("Must provide --band info when requesting replace-on-pending")
            pass

        element_id = args.element_id
        element_token = args.element_token
        zmc_http = args.zmc_http
        if args.element_userid:
            element_userid = args.element_userid
            pass
        if args.grant_id:
            grant_id = args.grant_id
        elif not args.band:
            sys.exit("Must supply a grant ID or --band description")
            pass
    else:
        # Grab everything we need from the manifest.
        manifest_dict = manifest.parse_manifest()

        if not "rdzinfo" in manifest_dict:
            sys.exit("No rdzinfo in the manifest")
            pass
        rdzinfo = manifest_dict["rdzinfo"]
        if not "element-id" in rdzinfo:
            sys.exit("No element-id in the manifest rdzinfo")
            pass
        element_id = rdzinfo["element-id"]
        if not "zmc-http" in rdzinfo:
            sys.exit("No zmc-http in the manifest rdzinfo")
            pass
        zmc_http = rdzinfo["zmc-http"]
        if not "element-token" in manifest_dict["passwords"]:
            sys.exit("No element-token in the manifest passwords")
            pass
        element_token = manifest_dict["passwords"]["element-token"]
        #
        # the zms user id is optional, if we do not get it, then we can assume
        # that the token is a privileged token or we got an admin token.
        #
        element_userid = None
        if "user-id" in rdzinfo:
            element_userid = rdzinfo["user-id"]
            pass
        elif "admin-token" in manifest_dict["passwords"]:
            element_token = manifest_dict["passwords"]["admin-token"]
            pass

        #
        # Grant are in the spectrum section. For now just one grant, or
        # none at all. 
        #
        if ("spectrum" in manifest_dict and
            len(manifest_dict["spectrum"].keys()) > 0):
            grant_keys = list(manifest_dict["spectrum"].keys())
            grant_id = grant_keys[0]
            pass

        args.element_id = element_id
        args.element_token = element_token
        args.element_userid = element_userid
        args.zmc_http = zmc_http
        pass

    #
    # Heartbeats do not make sense in the Powder context where a grant
    # is supplied.
    #
    if grant_id and args.heartbeat:
        sys.exit("Not allowed to use heartbeats when supplying a Powder grant")
        pass

    #
    # Going to need this when running in Powder and allocating a grant
    #
    if not grant_id and os.path.exists("/var/emulab"):
        result = subprocess.run(
            ["geni-get", 'slice_urn'], capture_output=True, text=True)
        if result.returncode:
            sys.exit("Could not get slice urn")
            pass

        matched = re.match(r"^[^\+]+\+([^\+]+)\+slice\+(.*)$", result.stdout)
        if not matched:
            sys.exit("Could not decode geni slice urn")
            pass
        tokens = matched.group(1).split(":")
        if len(tokens) != 2 and len(tokens) != 3:
            sys.exit("Bad domain:project in geni slice urn")
            pass

        global EXTID
        EXTID = "smarttx:" + tokens[1] + ":" + matched.group(2)
        pass

    global LOG
    LOG = logging.getLogger(__name__)    
    if args.debug:
        LOG.setLevel(logging.DEBUG)
        #logging.getLogger('zmsclient').setLevel(logging.DEBUG)
    else:
        LOG.setLevel(logging.INFO)
        logging.getLogger('zmsclient').setLevel(logging.INFO)
    if args.debug > 1:
        logging.getLogger().setLevel(logging.DEBUG)
        pass

    ZMC = ZmsZmcClient(zmc_http, element_token,
                       detailed=False, raise_on_unexpected_status=True)

    #
    # When doing heartbeats, we need thread locking.
    #
    lock = threading.RLock()
    condition = threading.Condition(lock=lock)

    if grant_id:
        # Need to load the grant we are watching
        print("Loading grant");
        grant = Grant(ZMC, loadGrant(ZMC, grant_id), args, lock, condition)
        pass
    else:
        print("Requesting grant");
        grant = Grant(ZMC, None, args, lock, condition).createGrant()
        pass
    
    # Then subscribe to the events
    if element_userid:
        filter = EventFilter(
            element_ids=[element_id], user_ids=[element_userid])
    else:
        filter = EventFilter(
            element_ids=[element_id])
        pass
    ZMCsubscription = ZMCSubscriptionCallback(
        ZMC, element_id,
        subscription=Subscription(
            id=str(uuid.uuid4()), filters=[filter]),
        args=args, grant=grant, lock=lock, condition=condition,
        reconnect_on_error=True)
    
    beater = None
    if args.heartbeat:
        beater = HeartBeat(grant)
        pass

    return ZMCsubscription, beater, args.daemon, args.logfile

async def async_main(ZMCsubscription, beater):
    this_task = asyncio.current_task();
    set_signal_handler(signal.SIGINT, this_task)
    set_signal_handler(signal.SIGHUP, this_task)
    set_signal_handler(signal.SIGTERM, this_task)

    subs = [ZMCsubscription]
    if beater:
        subs.append(beater)
        pass

    try:
        runnable = [sub.start() for sub in subs]
        await asyncio.gather(*runnable)
    except asyncio.CancelledError:
        if beater:
            beater.stop()
            pass
        await ZMCsubscription.stop()
        pass
    pass

#
# The reason for this split of main, is cause we need to do the daemon
# fork() before heading into the asyncio main.
#
def main():
    global HOSTNAME
    HOSTNAME = socket.gethostname()

    ZMCsubscription, beater, daemonize, logfile = init_main()
    format = "%(levelname)s:%(asctime)s:%(message)s"

    if daemonize:
        try:
            fp = open(logfile, "a");
            sys.stdout = fp
            sys.stderr = fp
            sys.stdin.close();
            logging.basicConfig(stream=fp, format=format)
            pass
        except:
            print("Could not open log file for append")
            sys.exit(1);
            pass
        pid = os.fork()
        if pid:
            sys.exit(0)
        os.setsid();
    else:
        logging.basicConfig(format=format)

    asyncio.run(async_main(ZMCsubscription, beater))


if __name__ == "__main__":
    main()
