import datetime
from typing import TYPE_CHECKING, Any, Dict, List, Type, TypeVar, Union, cast

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.any_object import AnyObject


T = TypeVar("T", bound="ExperimentCreate")


@_attrs_define
class ExperimentCreate:
    """Create a cloudlab experiment.

    Attributes:
        name (str): A human readable name for the experiment
        project (str): The project the experiment is instantiated in.
        profile_name (str): The name of the profile.
        profile_project (str): The name of the profile project
        group (Union[None, Unset, str]): The project subgroup the experiment is instantiated in.
        start_at (Union[None, Unset, datetime.datetime]): Schedule experiment to start at a future time
        stop_at (Union[None, Unset, datetime.datetime]): Schedule experiment to stop at a future time
        duration (Union[None, Unset, int]): Initial experiment duration in hours
        paramset_name (Union[None, Unset, str]): Optional name of a parameter set to apply to the profile
        paramset_owner (Union[None, Unset, str]): The owner of the parameter set
        bindings (Union[None, Unset, AnyObject]):
        refspec (Union[Unset, str]): For a repository based profile, optionally specify a refspec[:hash] to use instead
            of the HEAD of the default branch
    """

    name: str
    project: str
    profile_name: str
    profile_project: str
    group: Union[None, Unset, str] = UNSET
    start_at: Union[None, Unset, datetime.datetime] = UNSET
    stop_at: Union[None, Unset, datetime.datetime] = UNSET
    duration: Union[None, Unset, int] = UNSET
    paramset_name: Union[None, Unset, str] = UNSET
    paramset_owner: Union[None, Unset, str] = UNSET
    bindings: Union[None, Unset, "AnyObject"] = UNSET
    refspec: Union[Unset, str] = UNSET
    additional_properties: Dict[str, Any] = _attrs_field(init=False, factory=dict)
    _immutable_on = {
        "name": {"put": True, "post": None},
        "project": {"put": True, "post": None},
        "group": {"put": True, "post": None},
        "profile_name": {"put": True, "post": None},
        "profile_project": {"put": True, "post": None},
        "start_at": {"put": True, "post": None},
        "stop_at": {"put": True, "post": None},
        "duration": {"put": True, "post": None},
        "paramset_name": {"put": True, "post": None},
        "paramset_owner": {"put": True, "post": None},
        "bindings": {"put": None, "post": None},
        "refspec": {"put": True, "post": None},
    }

    def to_dict(self) -> Dict[str, Any]:
        name = self.name

        project = self.project

        profile_name = self.profile_name

        profile_project = self.profile_project

        group: Union[None, Unset, str]
        if isinstance(self.group, Unset):
            group = UNSET
        else:
            group = self.group

        start_at: Union[None, Unset, str]
        if isinstance(self.start_at, Unset):
            start_at = UNSET
        elif isinstance(self.start_at, datetime.datetime):
            start_at = self.start_at.isoformat()
        else:
            start_at = self.start_at

        stop_at: Union[None, Unset, str]
        if isinstance(self.stop_at, Unset):
            stop_at = UNSET
        elif isinstance(self.stop_at, datetime.datetime):
            stop_at = self.stop_at.isoformat()
        else:
            stop_at = self.stop_at

        duration: Union[None, Unset, int]
        if isinstance(self.duration, Unset):
            duration = UNSET
        else:
            duration = self.duration

        paramset_name: Union[None, Unset, str]
        if isinstance(self.paramset_name, Unset):
            paramset_name = UNSET
        else:
            paramset_name = self.paramset_name

        paramset_owner: Union[None, Unset, str]
        if isinstance(self.paramset_owner, Unset):
            paramset_owner = UNSET
        else:
            paramset_owner = self.paramset_owner

        bindings: Union[None, Unset, Dict[str, Any]] = UNSET
        if self.bindings is None:
            bindings = None
        elif not isinstance(self.bindings, Unset):
            bindings = self.bindings.to_dict()

        refspec = self.refspec

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "name": name,
                "project": project,
                "profile_name": profile_name,
                "profile_project": profile_project,
            }
        )
        if group is not UNSET:
            field_dict["group"] = group
        if start_at is not UNSET:
            field_dict["start_at"] = start_at
        if stop_at is not UNSET:
            field_dict["stop_at"] = stop_at
        if duration is not UNSET:
            field_dict["duration"] = duration
        if paramset_name is not UNSET:
            field_dict["paramset_name"] = paramset_name
        if paramset_owner is not UNSET:
            field_dict["paramset_owner"] = paramset_owner
        if bindings is not UNSET:
            field_dict["bindings"] = bindings
        if refspec is not UNSET:
            field_dict["refspec"] = refspec

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        from ..models.any_object import AnyObject

        d = src_dict.copy()
        name = d.pop("name")

        project = d.pop("project")

        profile_name = d.pop("profile_name")

        profile_project = d.pop("profile_project")

        def _parse_group(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        group = _parse_group(d.pop("group", UNSET))

        def _parse_start_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                start_at_type_0 = isoparse(data)

                return start_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        start_at = _parse_start_at(d.pop("start_at", UNSET))

        def _parse_stop_at(data: object) -> Union[None, Unset, datetime.datetime]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, str):
                    raise TypeError()
                stop_at_type_0 = isoparse(data)

                return stop_at_type_0
            except:  # noqa: E722
                pass
            return cast(Union[None, Unset, datetime.datetime], data)

        stop_at = _parse_stop_at(d.pop("stop_at", UNSET))

        def _parse_duration(data: object) -> Union[None, Unset, int]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, int], data)

        duration = _parse_duration(d.pop("duration", UNSET))

        def _parse_paramset_name(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        paramset_name = _parse_paramset_name(d.pop("paramset_name", UNSET))

        def _parse_paramset_owner(data: object) -> Union[None, Unset, str]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            return cast(Union[None, Unset, str], data)

        paramset_owner = _parse_paramset_owner(d.pop("paramset_owner", UNSET))

        _bindings = d.pop("bindings", UNSET)
        bindings: Union[None, Unset, AnyObject]
        if isinstance(_bindings, Unset):
            bindings = UNSET
        elif _bindings is None:
            bindings = None
        else:
            bindings = AnyObject.from_dict(_bindings)

        refspec = d.pop("refspec", UNSET)

        experiment_create = cls(
            name=name,
            project=project,
            profile_name=profile_name,
            profile_project=profile_project,
            group=group,
            start_at=start_at,
            stop_at=stop_at,
            duration=duration,
            paramset_name=paramset_name,
            paramset_owner=paramset_owner,
            bindings=bindings,
            refspec=refspec,
        )

        experiment_create.additional_properties = d
        return experiment_create

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties


def _load_lazy_imports(ns=globals()):
    from ..models.any_object import AnyObject

    ns["AnyObject"] = AnyObject
