from http import HTTPStatus
from typing import Any, Dict, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.error import Error
from ...models.experiment import Experiment
from ...models.experiment_modify import ExperimentModify
from ...types import Response


def _get_kwargs(
    experiment_id: str,
    *,
    body: ExperimentModify,
    x_api_token: str,
) -> Dict[str, Any]:
    headers: Dict[str, Any] = {}
    headers["X-Api-Token"] = x_api_token

    _kwargs: Dict[str, Any] = {
        "method": "patch",
        "url": "/experiments/{experiment_id}".format(
            experiment_id=experiment_id,
        ),
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Error, Experiment]]:
    if response.status_code == HTTPStatus.OK:
        try:
            response_200 = Experiment.from_dict(response.json())

            return response_200
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.BAD_REQUEST:
        try:
            response_400 = Error.from_dict(response.json())

            return response_400
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.UNAUTHORIZED:
        try:
            response_401 = Error.from_dict(response.json())

            return response_401
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.FORBIDDEN:
        try:
            response_403 = Error.from_dict(response.json())

            return response_403
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if response.status_code == HTTPStatus.NOT_FOUND:
        try:
            response_404 = Error.from_dict(response.json())

            return response_404
        except Exception:
            if client.raise_on_undecodable_content:
                raise errors.UndecodableContent(
                    response.status_code, response.content, response
                )
            else:
                raise
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content, response)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Error, Experiment]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    experiment_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: ExperimentModify,
    x_api_token: str,
) -> Response[Union[Error, Experiment]]:
    """Modify a running experiment

    Args:
        experiment_id (str): The target experiment ID.
        x_api_token (str):
        body (ExperimentModify): Modify a running experiment

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Experiment]]
    """

    kwargs = _get_kwargs(
        experiment_id=experiment_id,
        body=body,
        x_api_token=x_api_token,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    experiment_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: ExperimentModify,
    x_api_token: str,
) -> Optional[Union[Error, Experiment]]:
    """Modify a running experiment

    Args:
        experiment_id (str): The target experiment ID.
        x_api_token (str):
        body (ExperimentModify): Modify a running experiment

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Experiment]
    """

    return sync_detailed(
        experiment_id=experiment_id,
        client=client,
        body=body,
        x_api_token=x_api_token,
    ).parsed


async def asyncio_detailed(
    experiment_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: ExperimentModify,
    x_api_token: str,
) -> Response[Union[Error, Experiment]]:
    """Modify a running experiment

    Args:
        experiment_id (str): The target experiment ID.
        x_api_token (str):
        body (ExperimentModify): Modify a running experiment

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Error, Experiment]]
    """

    kwargs = _get_kwargs(
        experiment_id=experiment_id,
        body=body,
        x_api_token=x_api_token,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    experiment_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: ExperimentModify,
    x_api_token: str,
) -> Optional[Union[Error, Experiment]]:
    """Modify a running experiment

    Args:
        experiment_id (str): The target experiment ID.
        x_api_token (str):
        body (ExperimentModify): Modify a running experiment

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        errors.UndecodableContent: If the server returns undecodable content and Client.raise_on_undecodable_content is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Error, Experiment]
    """

    return (
        await asyncio_detailed(
            experiment_id=experiment_id,
            client=client,
            body=body,
            x_api_token=x_api_token,
        )
    ).parsed
