import logging
import typing
from typing import Dict, Union, Any, Optional, Callable, _UnionGenericAlias, Annotated, Sequence, ForwardRef
import inspect
import importlib
import enum
import typer
import click
import json
from datetime import datetime, timezone

__package__root = __package__.split(".")[0]

LOG = logging.getLogger('cloudlabclient.cli')

#
# Rather than intercept all parameters with datetime type annotations, hook
# click.DateTime type to support additional datetime formats.
#
# https://typer.tiangolo.com/tutorial/parameter-types/datetime/
#
__click_DateTime = click.DateTime
class ZmsDateTime(__click_DateTime):
    def __init__(self, formats: Optional[Sequence[str]] = None):
        self.formats: Sequence[str] = formats or [
            "%Y-%m-%dT%H:%M:%S%z",
            "%Y-%m-%dT%H:%M:%S.%f%z",
            "%Y-%m-%d",
            "%Y-%m-%dT%H:%M:%S",
            "%Y-%m-%d %H:%M:%S",
        ]

    def convert(self, value, param, ctx) -> Any:
        if value == "now":
            return datetime.now(timezone.utc)
        ret = super(ZmsDateTime, self).convert(value, param, ctx)
        if isinstance(ret, datetime) and ret.tzinfo is None:
            ret = ret.astimezone()
        return ret

click.DateTime = ZmsDateTime

_have_griffe = False
_tried_griffe = False
def load_griffe():
    global _have_griffe
    global _tried_griffe

    if _tried_griffe:
        return _have_griffe

    _tried_griffe = True

    exlist = []
    try:
        from griffe import Docstring
        from griffe import (
            DocstringSectionText, DocstringSectionParameters,
            DocstringSectionAttributes)
        _have_griffe = True
        globals()['Docstring'] = Docstring
        globals()['DocstringSectionText'] = DocstringSectionText
        globals()['DocstringSectionParameters'] = DocstringSectionParameters
        globals()['DocstringSectionAttributes'] = DocstringSectionAttributes
        return True
    except Exception as ex:
        exlist.append(ex)

    try:
        from griffe import Docstring
        from griffe.docstrings.dataclasses import (
            DocstringSectionText, DocstringSectionParameters,
            DocstringSectionAttributes)
        _have_griffe = True
        globals()['Docstring'] = Docstring
        globals()['DocstringSectionText'] = DocstringSectionText
        globals()['DocstringSectionParameters'] = DocstringSectionParameters
        globals()['DocstringSectionAttributes'] = DocstringSectionAttributes
        return True
    except Exception as ex:
        exlist.append(ex)

    if exlist:
        LOG.info("griffe not installed")
        for ex in exlist:
            LOG.exception(ex)

    return False

def parse_docstring(doc: str):
    if not load_griffe():
        return (None, None)

    parsed = Docstring(doc, parser="google", parser_options=dict(warn_unknown_params=False)).parsed
    title = None
    parameters = dict()
    for p in parsed:
        if isinstance(p, DocstringSectionText):
            title = p.value
        elif isinstance(p, DocstringSectionParameters):
            for pp in p.value:
                parameters[pp.name] = pp.description
        elif isinstance(p, DocstringSectionAttributes):
            for pp in p.value:
                parameters[pp.name] = pp.description
    return (title, parameters)

def sanitize_api_func_annotations(annotations: Dict[str, Any]) -> (Dict[str, Any], Any):
    """
    Reduce Union type annotations into the meaningful value type only.
    The openapi-python-client generator uses Unions to capture
    (Unset, None, <Value>) parameter cases and to model Responses.
    If there is a case we do not support, such as a Union type with multiple
    types that are not Unset or None, we raise a TypeError.
    """
    sanitized = dict()
    for ak, av in annotations.items():
        if ak == "client":
            sanitized[ak] = av.__args__[1]
        elif ak == "return":
            sanitized[ak] = av.__origin__
        elif isinstance(av, _UnionGenericAlias):
            vclass = None
            for ut in av.__args__:
                if ut.__name__ == "Unset" or ut.__name__ == "None":
                    continue
                elif not vclass is None:
                    raise TypeError(f"cannot sanitize Union type with multiple concrete subtypes ({ak})")
                else:
                    vclass = ut
            sanitized[ak] = av
        else:
            sanitized[ak] = av

    return sanitized

def is_zmsclient_unset_type(t: Any):
    if hasattr(t, "__module__"):
        ma = t.__module__.split(".")
        if ma[0] == __package__root and len(ma) > 2 and ma[-1] == "types" and t.__name__ == "Unset":
            return True
    return False

def is_zmsclient_model_type(t: Any):
    if hasattr(t, "__module__"):
        ma = t.__module__.split(".")
        if ma[0] == __package__root and len(ma) > 0 and ma[-2] == "models" and hasattr(t, "from_dict"):
            return True
    return False

def find_zmsclient_unset_type(t: Any):
    if not hasattr(t, "__module__"):
        return None
    ma = t.__module__.split(".")
    if ma[0] != __package__root:
        return None
    nmn = ".".join(ma[0:3]) + ".types"
    try:
        nmod = importlib.import_module(nmn)
        return nmod.Unset
    except Exception as ex:
        LOG.debug(f"no Unset type for type '{t}': {ex}")
    return None

def is_zmsclient_single_body_model_func(f: Callable):
    fname = f.__name__
    sig = inspect.signature(f)
    #LOG.debug("is_zmsclient_single_body_model_func(%s): signature: %r", fname, sig)
    for p in list(sig.parameters.values()):
        if p.name == "body" and is_zmsclient_model_type(p.annotation):
            return True
    return False

class ZmsClickType(click.ParamType):
    name = "unset"

    def __init__(self, base_type: Any, base_click_type=None, none_type=None, unset_type=None, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if isinstance(base_type, ForwardRef):
            base_type = base_type._evaluate(
                globals(), locals(), recursive_guard=set())
        elif isinstance(base_type, str):
            LOG.debug(" non-ForwardRef str literal base_type: %r", base_type)
            btf = ForwardRef(base_type)
            base_type = btf._evaluate(
                globals(), locals(), recursive_guard=set())
        LOG.debug("ZmsClickType: %r %r", base_type, base_click_type)
        self.base_type = base_type
        self.name = getattr(base_type, "__name__", str(base_type))
        self.none_type = none_type
        self.unset_type = unset_type
        self.base_click_type = base_click_type
        self.base_convert = None
        self.base_post_convert = None
        #
        # See if there is a click type for this type.  If this is one of the
        # autogenerated models in zmsclient, probably not, unless it is an
        # Enum.
        #
        if self.base_click_type is None:
            # First, try to see if this is a supported Click type.
            try:
                self.base_click_type = typer.main.get_click_type(
                    annotation=base_type, parameter_info=typer.models.ParameterInfo())
                self.base_convert = self.base_click_type.convert
            except Exception as e:
                # Or, if this is one of our generated model types (a
                # dict that can become a model), use that conversion.
                if is_zmsclient_model_type(base_type):
                    self.base_convert = self.zmsclient_base_type_convert
                else:
                    # Else, this is an unsupported type.
                    LOG.error("failed to convert base_type: %r", base_type)
                    raise e
        else:
            self.base_convert = self.base_click_type.convert

        # Second, if this is a core Click type, see if this is one of the
        # Typer-assisted post-converters (e.g. an Enum -- Typer converts
        # the str into an Enum.value, which is what the generated bindings
        # expect.  But the Typer converters just take a single value arg,
        # so wrap them.
        if self.base_click_type is not None:
            post_conv = None
            try:
                post_conv = typer.main.determine_type_convertor(base_type)
                if post_conv:
                    self.base_post_convert = post_conv
            except:
                pass

    def zmsclient_base_type_convert(self, value, param, ctx):
        d = None
        try:
            d = json.loads(value)
        except Exception as e:
            self.fail(
                "{value!r} is not a valid JSON dict that can be converted into {base_type.__name__} ({ex}).".format(value=value, base_type=self.base_type, ex=repr(e)), param, ctx
            )
        try:
            d = self.base_type.from_dict(d)
        except Exception as e:
            self.fail(
                "{value!r} is not a valid {base_type.__name__} ({base_type.__name__}.from_dict: {ex}).".format(value=value, base_type=self.base_type, ex=repr(e)), param, ctx
            )
        return d

    def convert(self, value, param, ctx):
        LOG.debug(f"convert({self.__repr__()}): {value}, {param}, {ctx}, {self.base_convert}")
        if self.unset_type is not None and isinstance(value, self.unset_type):
            return value
        if self.none_type is not None and isinstance(value, self.none_type):
            return value
        if type(value) == str and value.startswith("@"):
            filename = value[1:]
            with open(filename) as fp:
                value = fp.read()
                pass
            if False and param.name != "body":
                value = json.loads(value)
                pass
            pass
        res = self.base_convert(value, param, ctx)
        if self.base_post_convert is not None:
            res = self.base_post_convert(res)
        return res

    def get_metavar(self, param):
        if self.base_click_type is not None:
            return self.base_click_type.get_metavar(param)
        return super(ZmsClickType, self).get_metavar(param)

    def __repr__(self):
        return f"<ZmsClientType({self.name}):base_type={self.base_type},none_type={self.none_type},unset_type={self.unset_type},base_click_type={self.base_click_type}>"

    def __str__(self):
        return self.name

def get_click_or_zms_type(annotation: Any) -> Any:
    try:
        return typer.main.get_click_type(
            annotation=annotation, parameter_info=typer.models.ParameterInfo())
    except Exception as e:
        return ZmsClickType(annotation)


def is_field_immutable(field_name: str, verb: str, immutable_on: dict) -> bool:
    if not immutable_on or not field_name:
        return False

    if verb == "create":
        return immutable_on.get(field_name, {}).get("post", False)

    if verb == "update" or verb == "modify":
        return immutable_on.get(field_name, {}).get("put", False)

    return False

def sanitize_api_func_params(fname: str, params: list, skip_complex: bool = False, force_optional: bool = False, unset_type: Any | None = None, immutable_on=None, verb=None, dsparams=None):
    LOG.debug("sanitize_api_func_params(%s):", fname)
    for p in params:
        LOG.debug("  parameter: %r", p)

    # We need to mutate the signature in several ways so that we can pass it
    # through to typer.
    #
    # 1) Strip the `client` parameter away.  We compose it dynamically in the
    # wrapper function from the ContextObj.
    #
    # 2) In general, we need to flatten away "meaningless" Union annotations.
    # The openapi-python-client generator uses Unions to capture (Unset, None,
    # <Value>) parameter cases and to model Responses.  If there is a case we
    # do not support, such as a Union type with multiple types that are not
    # Unset or None, we raise a TypeError.
    #
    #   a) The common header params (x_api_token, x_api_elaborate,
    #   x_api_force_update) also include Unions down in their type definitions,
    #   and we already lift them into the global options rather than making
    #   them per-subcommand options for ease of use, so we skip them here as
    #   well.  NB: at runtime, we only pass these headers if the original api
    #   invocation function included them!
    #
    # 3) For create commands that have a `body` parameter, we add extra
    # typer.Parameter foo to help typer decode.  We accept either body; or any
    # other args into the constructor of the target class.  If both body and
    # args are set, we throw an error.  It would be nice to only add other args
    # for fields that do not have `x-immutable-on-post` set, but we do not have
    # that info in the generated model class.
    #
    # 4) For update commands that have a `body` parameter, we handle as with
    # create, but we also add the attributes of the body class as parameters.
    # It would be ideal to only add those that do not have `x-immutable-on-put`
    # set True, but that information is not present in the generated models.
    #
    new_params = list()
    client_class = None
    want_headers = dict()
    for p in params:
        dshelp = None
        if dsparams and p.name in dsparams:
            dshelp = dsparams[p.name]
        if p.name == "client":
            if isinstance(p.annotation, _UnionGenericAlias):
                for uat in p.annotation.__args__:
                    if uat.__name__ == "Client":
                        client_class = uat
                        break
            if client_class is None:
                raise TypeError(f"cannot extract 'client' class for function '{fname}'")
        elif p.name in ("x_api_token", "x_api_elaborate", "x_api_force_update"):
            want_headers[p.name] = True
            continue

        elif isinstance(p.annotation, _UnionGenericAlias):
            base_type = None
            union_unset_type = None
            none_type = None
            do_skip_complex = False
            for uat in p.annotation.__args__:
                if is_zmsclient_unset_type(uat):
                    union_unset_type = uat
                elif isinstance(uat, type(None)) or uat == type(None):
                    none_type = uat
                elif base_type is not None:
                    msg = f"cannot sanitize Union param ({p.name}) with multiple concrete subtypes (next={uat}, first={base_type}, annotation={p.annotation})"
                    if skip_complex:
                        LOG.info(f" skipping complex param ({msg})")
                        do_skip_complex = True
                        break
                    else:
                        raise TypeError(msg)
                else:
                    #LOG.debug(f"else base_type={uat} {uat.__name__} {type(None)}")
                    base_type = uat
            if do_skip_complex:
                continue
            if base_type is None:
                raise TypeError(f"cannot sanitize Union type with no concrete subtypes ({p.__name__})")
            # We need to construct a wrapper click type using ZmsClickType.
            # This wrapper class allows values of the base type, and None or
            # Unset, if part of the original Union.
            zctype = None
            try:
                zctype = ZmsClickType(base_type, none_type=none_type, unset_type=union_unset_type)
            except Exception as ex:
                if skip_complex:
                    LOG.info(f" skipping complex param ({p.name}) with no click type: {ex}")
                    continue
                raise ex

            # Now construct a parameter annotation type; not sure how to do
            # this by direct typing.Optional instantiation.
            ndefault = None
            if p.default == inspect._empty and not force_optional:
                ndefault = typer.Option(click_type=zctype, help=dshelp)
                def __f(x: Annotated[ZmsClickType, typer.Option(click_type=zctype, help=dshelp)]):
                    pass
            else:
                if force_optional and unset_type:
                    pdefault = unset_type()
                else:
                    pdefault = p.default
                ndefault = typer.Option(default=pdefault, click_type=zctype, help=dshelp)
                def __f(x: Optional[ZmsClickType] = typer.Option(default=pdefault, click_type=zctype, help=dshelp)):
                    pass
            pannotation = inspect.signature(__f).parameters["x"].annotation
            np = inspect.Parameter(p.name, p.kind, default=ndefault, annotation=pannotation)

            if is_field_immutable(np.name, verb, immutable_on):
                continue

            new_params.append(np)
            LOG.debug(f"  replaced '{p.name}' with {zctype.__repr__()}")
        elif not isinstance(p.annotation, enum.Enum) and is_zmsclient_model_type(p.annotation):
            # We need to construct a wrapper click type using ZmsClickType.
            # Enums are special: the click and typer parser/converters *mostly*
            # do the right thing for us (the click parser recognizes values in
            # the Enum (or error if not), and typer wraps that to convert the
            # str from click to an instance of the Enum class; so our wrapper
            # just has to call both of these.  Note that in this case, we do
            # not handle None | Unset, since this is bare Enum.
            zctype = ZmsClickType(p.annotation)
            # Now construct a parameter annotation type; not sure how to do
            # this by direct typing.Optional instantiation.
            ndefault = None
            if p.default == inspect._empty and not force_optional:
                ndefault = typer.Option(click_type=zctype, help=dshelp)
                def __f(x: Annotated[ZmsClickType, typer.Option(click_type=zctype, help=dshelp)]):
                    pass
            else:
                if force_optional and unset_type:
                    pdefault = unset_type()
                else:
                    pdefault = p.default
                ndefault = typer.Option(default=pdefault, click_type=zctype, help=dshelp)
                def __f(x: Optional[ZmsClickType] = typer.Option(default=pdefault, click_type=zctype, help=dshelp)):
                    pass
            np = inspect.Parameter(p.name, p.kind, default=ndefault, annotation=p.annotation)

            if is_field_immutable(np.name, verb, immutable_on):
                continue

            new_params.append(np)
            LOG.debug(f"  wrap-replaced '{p.name}' with {zctype.__repr__()}")
        else:
            #
            # Add support for forced long-name parameters by default.  (In the
            # above cases, this is happening by default, since the Union type
            # case is only output by the stub generator for optional
            # parameters.)
            #
            # Generally we need to wrap the type in ZmsClientType if there is
            # no base click type, to avoid an error.  But a trick: either typer
            # or click cannot handle Annotated|Optional[p.annotation,
            # typer.Option] when p.annotation is bool and the parameter is
            # required, so we have to wrap it in a ZmsClickType and pass it
            # through with click_type.
            #
            if force_optional:
                zctype = ZmsClickType(p.annotation, unset_type=unset_type)
            elif p.annotation == bool:
                zctype = ZmsClickType(p.annotation)
            elif hasattr(p.annotation, '__mro__') and enum.Enum in p.annotation.__mro__:
                zctype = ZmsClickType(p.annotation)
            else:
                zctype = get_click_or_zms_type(p.annotation)
            if p.default == inspect._empty and not force_optional:
                ndefault = typer.Option(
                    default="--" + p.name.replace("_", "-"), click_type=zctype, help=dshelp)
                def __f(x: Annotated[zctype, ndefault]):
                    pass
            else:
                if force_optional and unset_type:
                    pdefault = unset_type()
                else:
                    pdefault = p.default
                ndefault = typer.Option(default=pdefault, click_type=zctype, help=dshelp)
                def __f(x: Optional[zctype] = ndefault):
                    pass

            fp = list(inspect.signature(__f).parameters.values())[0]
            np = inspect.Parameter(p.name, fp.kind, default=fp.default, annotation=fp.annotation)

            if is_field_immutable(np.name, verb, immutable_on):
                continue

            new_params.append(np)
            LOG.debug(f"  included '{p.name}, {p.annotation}' with {zctype.__repr__()}")
    return (client_class, new_params, want_headers)

def wrap_api_func(verb: str, f: Callable, explode_body: bool = False, skip_complex: bool = False, unset_type: Any = None, explode_prefetcher: Callable | None = None) -> Callable:
    if f.__module__:
        fname = f.__module__.split(".")[-1] + "." + f.__name__
    else:
        fname = f.__name__
    sig = inspect.signature(f)
    LOG.debug(f"wrap_api_func({fname}) (unset={unset_type}, pre={explode_prefetcher})")

    # Cache parsed docstring for later annotation into Typer/Click.
    try:
        (dstitle, dsparams) = parse_docstring(f.__doc__)
    except Exception as e:
        LOG.exception(e)

    # Sanitize the func params and generate any necessary click/typer type
    # wrappers.
    func_params = list(sig.parameters.values())
    (client_class, new_params, want_headers) = sanitize_api_func_params(fname, func_params, unset_type=unset_type, dsparams=dsparams)
    # Check for the single `body` param case and handle it.
    exploded_body_model_class = None
    exploded_body_param_names = None
    duplicate_exploded_body_param_names = None
    _new_params = []
    new_param_names = [p.name for p in new_params]
    for p in new_params:
        if p.name == "body" and is_zmsclient_model_type(p.annotation) and explode_body:
            # Explode the single-body model type into per-field kwargs,
            # replacing the `body` arg.
            exploded_body_model_class = p.annotation
            # Load any lazy import types (ForwardRefs) so that forward refs
            # resolve later on.
            exm = inspect.getmodule(exploded_body_model_class)
            if hasattr(exm, '_load_lazy_imports'):
                LOG.debug("    loading lazy imports in exploded body module %r", exm)
                exm._load_lazy_imports(ns=globals())
            # Get immutable_on field from model schema.
            immutable_on = getattr(exploded_body_model_class, '_immutable_on', {})
            exploded_body_params = []
            exploded_body_param_names = []
            duplicate_exploded_body_param_names = []
            LOG.debug(f"immutable_on: {immutable_on}")
            msig = inspect.signature(exploded_body_model_class.__init__)
            LOG.debug(" exploding body model %r:", exploded_body_model_class)
            try:
                (dstitle, dsparams) = parse_docstring(exploded_body_model_class.__doc__)
                LOG.debug(" exploded body model attributes: %r", dsparams)
            except Exception as e:
                LOG.exception(e)

            for p in list(msig.parameters.values()):
                #LOG.debug("  parameter: %r", p)
                if p.name == "self" or p.name == "additional_properties":
                    continue
                exploded_body_params.append(p)
            bfunc_name = "%s.%s" % (
                exploded_body_model_class.__name__,
                exploded_body_model_class.__init__.__name__)
            force_optional = explode_prefetcher != None
            (_, new_exploded_body_params, _) = sanitize_api_func_params(
                bfunc_name, exploded_body_params, skip_complex=skip_complex,
                force_optional=force_optional, unset_type=unset_type, immutable_on=immutable_on, verb=verb, dsparams=dsparams)
            for p in new_exploded_body_params:
                # If this is a duplicate param name, handle by assume value
                # equivalency.
                if p.name in new_param_names:
                    duplicate_exploded_body_param_names.append(p.name)
                else:
                    _new_params.append(p)
                    exploded_body_param_names.append(p.name)
        else:
            _new_params.append(p)
    # NB: sort _new_params so that params with no defaults are at the head of
    # the list.  This supports an exploded body param as well as required or
    # optional args (e.g., query args).
    new_params = sorted(_new_params, key=lambda x: int(x.default != inspect._empty))
    new_param_names = [p.name for p in new_params]

    # If the class name is Response, just use that instead.  It is typical for
    # the result to be either an Error or an Object, both cases of which the
    # result_callback handles naturally.
    return_annotation = sig.return_annotation
    if return_annotation.__origin__.__name__ == "Response":
        return_annotation = return_annotation.__origin__
    LOG.debug("wrap_api_func(%s): client class: %r", fname, client_class)
    LOG.debug("wrap_api_func(%s): new params:", fname)
    for p in new_params:
        LOG.debug("  parameter: %r", p)

    #@functools.wraps(f)
    def wrapped_api_func(ctx: typer.Context, *args, **kwargs) -> Any:
        # Figure out which URL (at runtime) to pass to the Client instance.
        svc = client_class.__module__.split('.')[1]
        url = ctx.obj.get_service_url(svc)

        # Construct the Client to pass to the API endpoint.
        client = client_class(
            base_url=url, timeout=ctx.obj.timeout,
            verify_ssl=ctx.obj.verify_ssl, follow_redirects=ctx.obj.follow_redirects,
            raise_on_unexpected_status=ctx.obj.raise_on_unexpected_status,
            raise_on_undecodable_content=ctx.obj.raise_on_undecodable_content)

        nkwargs = dict()
        # If this was an exploded single-body param case, replace the kwargs in
        # exploded_body_param_names with
        # body=exploded_body_model_class(**mkwargs)
        if explode_body and exploded_body_param_names:
            mkwargs = dict()
            # Assemble the prefetcher if instructed.
            if explode_prefetcher:
                # NB: we do not want to prefetch with elaborate.
                flkwargs = dict()
                for (pname, ctxattr) in (("x_api_token", "token"),):
                    if getattr(ctx.obj, ctxattr, None) and want_headers.get(pname, None):
                        flkwargs[pname] = str(getattr(ctx.obj, ctxattr, None))
                # NB: filter out `body` and kwargs that are not present in the
                # getter's function signature.
                pf_sig = inspect.signature(explode_prefetcher)
                pf_param_names = pf_sig.parameters.keys()
                fnkwargs = dict()
                for (k, v) in kwargs.items():
                    if k != "body" and k in pf_param_names:
                        fnkwargs[k] = v
                resp = explode_prefetcher(*args, client=client, **flkwargs, **fnkwargs)
                fetched = resp.parsed
                LOG.debug(f"prefetched body object: {fetched}")
                if resp.status_code >= 400 and resp.status_code <= 600:
                    return resp
                mkwargs = fetched.to_dict()
            for (k, v) in kwargs.items():
                if unset_type != None and isinstance(v, unset_type):
                    continue
                # NB: because we transformed the prefetched content into a dict
                # so that we could iterate over the keys, to override them; we
                # must also transform model objects into dicts, so that we can
                # call from_dict on the top-level object as well as the
                # exploded children.
                if explode_prefetcher and is_zmsclient_model_type(v):
                    v = v.to_dict()
                if k in duplicate_exploded_body_param_names:
                    nkwargs[k] = v
                    mkwargs[k] = v
                elif k not in exploded_body_param_names:
                    nkwargs[k] = v
                else:
                    mkwargs[k] = v
            if explode_prefetcher:
                nkwargs['body'] = exploded_body_model_class.from_dict(mkwargs)
            else:
                nkwargs['body'] = exploded_body_model_class(**mkwargs)
            LOG.debug(f"exploded_body_model_class: {nkwargs['body']}")
        else:
            nkwargs = kwargs

        # Assemble headers.
        #
        # NB: rather than always passing headers, only pass them if they were
        # options for the endpoint, as described by the function signature.
        #headers = ctx.obj.get_headers()
        lkwargs = dict()
        for (pname, ctxattr) in (("x_api_token", "token"), ("x_api_elaborate", "elaborate"), ("x_api_force_update", "force")):
            if getattr(ctx.obj, ctxattr, None) and want_headers.get(pname, None):
                lkwargs[pname] = str(getattr(ctx.obj, ctxattr, None))

        LOG.debug(f"call {fname}({args}, client={client}, {lkwargs}, {nkwargs})")

        # Actually invoke the wrapped API endpoint function.
        ret = f(*args, client=client, **lkwargs, **nkwargs)
        LOG.debug("ret = %r", ret)
        return ret

    # Grab the ctx parameter's signature, then "properly" forward the sanitized
    # annotations of the wrapped api function:
    new_params.insert(0, inspect.Parameter("ctx", inspect.Parameter.POSITIONAL_OR_KEYWORD, annotation=typer.Context))
    new_wrapped_sig = sig.replace(parameters=new_params, return_annotation=return_annotation)
    wrapped_api_func.__signature__ = new_wrapped_sig
    if dstitle:
        wrapped_api_func.__doc__ = dstitle

    if LOG.level <= logging.DEBUG:
        new_wrapped_sig = inspect.signature(wrapped_api_func)
        LOG.debug("wrap_api_func(%s): wrapped:", fname)
        for p in list(new_wrapped_sig.parameters.values()):
            LOG.debug("  parameter: %r", p)
        LOG.debug("  return: %r", new_wrapped_sig.return_annotation)

    return wrapped_api_func
