# portal-api (Cloudlab/Powder Python 3 API client library and CLI tools)

This repository contains a Python 3 API client library based on API
bindings (models and API function wrappers) automatically generated
from the `openapi.json` file in this repository. Bindings are generated using
[our fork](https://gitlab.flux.utah.edu/openzms/openapi-python-client) of
`openapi-python-client`, which has additional support for some of the
`openapi` extension attributes we provide to help generators create better
code.

This repository also provides a [`typer`](https://typer.tiangolo.com/)-based
dynamically-generated CLI tool.  On invocation, the CLI tool scans a subset
of the generated API function wrappers and uses a combination of `typer` and
`click` to wrap them into a CLI tool that exposes each API endpoint as a
subcommand.

Auto generated API documentation is available at
http://emulab.pages.flux.utah.edu/portal-api, most of the examples
below are included there, on the right hand side. Look for the `CLI`
and `Python` tabs in the `Request Samples` section of each endpoint.

The full usage manual for Cloudlab is at https://docs.cloudlab.us/ and
the Powder usage manual is at https://docs.powderwireless.net/. The 
rest of this document will make more sense if you read one of those
first.

## Installing from source

Most likely you will want to install this library and its tools with
`virtualenv` (drop the `'[cli]'` if you don't require the CLI):

```
python -m venv path/to/your/venv
. path/to/your/venv/bin/activate
pip install .'[cli]'
```

Then you can run `portal-cli --help`, or use the library.  Leave the
`virtualenv` via `deactivate`. 

You can also install locally with `pip`:

```
pip install --user .'[cli]'
```

## Using the Portal API

In order to use the Portal API, you need to download a `token` from
the Cloudlab (or Powder) web UI. After you log in, click on your name
in the upper right and select `Portal API Token`. A token will be
downloaded to your desktop. At the moment you can have only one token,
which is valid for two months. Eventually this will change to allow
longer tokens and token refresh. But for now you will need to download
a new token periodically.

Your token is not encrypted and does not require a password to use.
You should never store your token in a public repository!

Once you have your token and are ready to use it, the easiest thing
to do is export it as an environment variable, along with the URL of
the server. 

	export PORTAL_TOKEN="your_token_string"
	export PORTAL_HTTP="https://boss.emulab.net:43794"

You can also pass the token and url via command line arguments.

	portal-api --token "your_token_string" --portal-url "https://..." ...

For example, to get a list of your experiments:

	mypc> portal-api experiment list
	[
        {
            "created_at": "2025-02-19T14:19:16+00:00",
            "creator": "stoller",
            "expires_at": null,
            "group": null,
            "id": "7ce46d72-eecc-11ef-af1a-e4434b2381fc",
            "last_snapshot_status": null,
            "name": "lbs-frontend",
            "profile_id": "7f0cfde3-cf91-11ef-828b-e4434b2381fc",
            "profile_name": "small-lan",
            "profile_project": "emulab-ops",
            "project": "testbed",
            "started_at": "2025-02-19T14:19:19+00:00",
            "status": "ready",
            "expires_at": "2025-02-20T06:00:00+00:00",
            "aggregates": {
                "urn:publicid:IDN+emulab.net+authority+cm": {
                    "name": null,
                    "nodes": [
                        {
                            "client_id": "node0",
                            "rawstate": "ISUP",
                            "startup_status": null,
                            "state": "started",
                            "status": "ready",
                            "urn": "urn:publicid:IDN+emulab.net+authority+cm"
                        }
                    ],
                    "status": "ready",
                    "urn": "urn:publicid:IDN+emulab.net+authority+cm"
                }
            },
        }
    ]

And a fragment of python code that does the same:

	from cloudlabclient.portal.client import PortalClient
	from cloudlabclient.portal.v1.models import ExperimentList

	Portal = PortalClient("https://boss.emulab.net:43794",
                          "your_token_string",
                          detailed=False, raise_on_unexpected_status=True)
	
	explist = Portal.list_experiments()
	print(str(explist.experiments))


## Experiments

The following sections include examples of various operations. Only
some will include a python version, the rest can be gleaned from the
auto generated API documentation at
http://emulab.pages.flux.utah.edu/portal-api

Several of the examples reference files that are contained in the
`cloudlabclient/tests` directory of this repository. 

### Creating an experiment

For this example, we are using the `small-lan` profile, which is a
parameterized profile that requires a set of binding variables. The
bindings are provided as a json object, and while you can do that
on the command line, it is easier to put them in a file and provide
the filename on the command line, prefaced with an `@` sign. Here
are the bindings we are going to use to instantiate a 1 node
experiment at the Emulab cluster:

	{
		"nodeCount": "1",
		"phystype": "d710",
		"osImage": "default",
	}

Which are stored in a file in this repository. The command line to
start this experiment, and have it terminate in one hour:
	
	EXPID=`portal-cli experiment create --name apitest --project myproject \
		--profile-name small-lan --profile-project PortalProfiles --duration 1 \
		--bindings @cloudlabclient/tests/experiments/bindings-1node.json | jq -r .id`
	export EXPID
	
EXPID is needed for the couple of examples. A python fragment to do
the same:

	from cloudlabclient.portal.client import PortalClient
	from cloudlabclient.portal.v1.models import ExperimentCreate, AnyObject

	Portal = PortalClient("https://boss.emulab.net:43794",
						  "your_token_string",
						  detailed=False, raise_on_unexpected_status=True)

	myexp = Portal.create_experiment(body = ExperimentCreate(
		name = "apitest",
		project = "testbed",
		duration = 1,
		profile_name = "small-lan",
		profile_project = "PortalProfiles",
		bindings = AnyObject.from_dict({
			"nodeCount": "1",
			"phystype": "d710",
			"osImage": "default"
		})
	))

### Getting experiment status

To get the experiment status you need the experiment ID (from
above). The return value is the same as the return value from
`create_experiment` and has been slightly abbreviated:

	portal-cli experiment get --experiment-id $EXPID
	{
		"aggregates": {
			"urn:publicid:IDN+emulab.net+authority+cm": {
				"nodes": [
					{
						"client_id": "node0",
						"rawstate": "ISUP",
						"startup_status": null,
						"state": "started",
						"status": "ready",
						"urn": "urn:publicid:IDN+emulab.net+authority+cm"
					}
				],
				"status": "ready",
				"urn": "urn:publicid:IDN+emulab.net+authority+cm"
			}
		},
		"bindings": {
			"nodeCount": "1",
			"osImage": "default",
			"phystype": "d710",
		},
		"created_at": "2025-07-28T18:01:30+00:00",
		"expires_at": "2025-07-28T19:01:30+00:00",
		"creator": "stoller",
		"id": "e0174522-6bdc-11f0-bc80-e4434b2381fc",
		"name": "apitest",
		"profile_id": "33e0df61-0f4d-11f0-828b-e4434b2381fc",
		"profile_name": "small-lan",
		"profile_project": "PortalProfiles",
		"project": "testbed",
		"status": "ready",
	}
	
### Modify an experiment.

Experiments that are instantiated from `parameterized profiles` can be
modified when they are in the `ready` state. Modification is done by 
providing a new set of bindings, different then the original bindings
that were supplied when creating the experiment. Using the one node
example above, this set of bindings will add another node:

	{
		"nodeCount": "2",
		"phystype": "d710",
		"osImage": "default",
	}

These bindings are also stored in a file in this repository. The
command line to start the experiment modification is:

	portal-cli experiment modify --experiment-id $EXPID \
		--bindings @cloudlabclient/tests/experiments/bindings-2node.json
	
### Getting experiment manifests

Once the experiment is ready, you can ask for the manifests(s):

	portal-cli experiment manifests get --experiment-id $EXPID

This will return a dictionary keyed by the aggregate (cluster)
URNs. The value is a string which will need to be XML decoded.

### Terminating an experiment

	portal-cli experiment terminate --experiment-id $EXPID
	
After a little while:

	portal-cli experiment get --experiment-id $EXPID
	{
		"code": 404,
		"error": "No such experiment",
	}

### Extending an experiment

Experiments can be extended in hour units. In general, the first week
or two will be granted automatically. Beyond that, an administrator
will need to approve the request. If the extension request is
immediately rejected, an error value will be returned. If your request
requires administrator intervention, you will receive email when it is
approved (or rejected). The return value is the new experiment status,
you can use the new value of `expires_at` to determine how much
time was granted.

	portal-cli experiment extend --experiment-id $EXPID --extend-by 2
	
Instead of `extend-by` you can supply an absolute GMT timestamp:

	portal-cli experiment extend --experiment-id $EXPID --expires-at "2025-07-28T19:01:30+00:00"

You may optionally supply a `--reason` option to provide a pithy yet
informative reason for your extension. This is sometimes helpful when
your request requires administrator approval and the Portal is very
busy. If you are working on a near deadline, be sure to mention that.
	
	portal-cli experiment extend --experiment-id $EXPID --extend-by 2
	--reason @/tmp/reason.txt
	
### Reboot/reload/powercycle/stop/start nodes in an experiment

Reboot (or reload, powercycle, stop, start) nodes in an experiment.
This can be done for individual nodes:

	portal-cli experiment node reboot --experiment-id $EXPID --client-id node0

or for all nodes in an experiment:

	portal-cli experiment nodes reboot --experiment-id $EXPID

Note `node` vs `nodes` in the two command lines.

### Taking an image snapshot of a node in an experiment

Image snapshots are done to individual nodes, one at a time. Once the
snapshot is started, you should periodically poll for the snapshot
status to see the progress (ongoing size of the image is reported) and
to know when the snapshot is complete. Or if the snapshot has failed.
Both the experiment and the node must be in the `ready` state to start
a snapshot. The node you want to snapshot is specified with the
`--client-id` argument and refers to the ID the node was given in the
instantiated profile source code.

To start a snapshot:

    SNAPID=`portal-cli experiment snapshot start --experiment-id $EXPID \
	       --client-id node0 --image-name apitest | jq -r '.id'`
	export SNAPID

To track the snapshot status:

	portal-cli experiment snapshot get --experiment-id $EXPID --snapshot-id $SNAPID`
	{
		"id": "497f6eca-6276-4993-bfeb-53cbbbba6f08",
		"status": "imaging",
		"status_timestamp": "2020-12-18T21:06:28.000Z",
		"image_size": 0,
		"image_urn": "string",
		"error_message": null,
	}
	
Imaging is complete when the status is `ready` or `failed`. The node
might take a little while longer to reboot and come back up,

## Reservation Groups

The next few sections describe how to use the API to manage
reservation groups.

### Creating a reservation group

Reservation groups are sets of node types (or reservable nodes) that
are reserved for you (or your project). The entire group of resources
has a start and expiration time. A group can have multiple node types
(or reservable nodes), each at different Cloudlab or Powder clusters.
More information on reservation groups can be found in the usage
manuals mention in the first section above.

This example uses a data file in this repository:

	{
		"nodetypes" : [
			{
				"count": 1,
				"nodetype": "d710",
				"urn": "urn:publicid:IDN+emulab.net+authority+cm"
			}
		]
	}

This command will create a 1 hour reservation for a single node,
starting now. Provide a GMT datetime to start in the future. Instead
of duration, you can provide `--expires_at` (also a GMT datetime) to
end at a specific time. 

	RESID=`portal-cli resgroup create --project testbed \
          --reason "My awesome research project" \
	      --start-at now --duration 1 \
          --nodetypes @cloudlabclient/tests/resgroups/res-create.json | jq -r '.id`
	export $RESID

A python fragment to do the same. Note that `start_at` is optional,
if not supplied it means "now".

    from cloudlabclient.portal.client import PortalClient
    from cloudlabclient.portal.v1.models import (
        ResGroup,
        ResGroupNodeTypes,
        ResGroupNodeType
    )

    Portal = PortalClient("https://boss.emulab.net:43794",
    		              "your_token_string",
                          detailed=False, raise_on_unexpected_status=True)
	
    myres = Portal.create_resgroup(
        duration = 1,
        body = ResGroup(
    	project = "testbed",
    	reason = "My awesome research project",
    	nodetypes = ResGroupNodeTypes(
    	    nodetypes=[
    		ResGroupNodeType(
    		    count = 1,
    		    nodetype = "d710",
    		    urn = "urn:publicid:IDN+emulab.net+authority+cm"
              )
		   ]
    	)
    ))

### Modify a reservation group

Most aspects of a reservation group can be modified before the
reservation is approved. After the reservation is approved, the start
time cannot be changed, but the expiration can be changed to an
earlier time. You can always delete individual reservations from the
group. You can also change the reason any time. Using the reservation
group created above, lets modify it to add another node type. this can
be done either before or after the reservation has been approved, but
the newly added nodetype might need a separate approval from an 
administrator. This is the data file:

    {
        "nodetypes" : [
            {
                "count": 1,
                "nodetype": "d710",
                "urn": "urn:publicid:IDN+emulab.net+authority+cm"
            },
            {
                "count": 1,
                "nodetype": "d430",
                "urn": "urn:publicid:IDN+emulab.net+authority+cm"
            }
        ]
    }

Here is the CLI command to do the modify:

	portal-cli resgroup modify --resgroup-id $RESID
          --nodetypes @cloudlabclient/tests/resgroups/res-modify.json
	
A python fragment to do the same:

    from cloudlabclient.portal.client import PortalClient
    from cloudlabclient.portal.v1.models import (
        ResGroup,
        ResGroupNodeTypes,
        ResGroupNodeType
    )

    Portal = PortalClient("https://boss.emulab.net:43794",
    		              "your_token_string",
                          detailed=False, raise_on_unexpected_status=True)
	
    myres = Portal.create_resgroup(
        body = ResGroup(
    	nodetypes = ResGroupNodeTypes(
    	    nodetypes=[
    		ResGroupNodeType(
    		    count = 1,
    		    nodetype = "d710",
    		    urn = "urn:publicid:IDN+emulab.net+authority+cm"
              ),
    		ResGroupNodeType(
    		    count = 1,
    		    nodetype = "d430",
    		    urn = "urn:publicid:IDN+emulab.net+authority+cm"
              )
		   ]
    	)
    ))

### Retrieving a reservation group

To retrieve a reservation group:

    portal-cli --elaborate resgroup get --resgroup-id $RESID
    {
        "created_at": "2025-07-31T20:07:28+00:00",
        "creator": "stoller",
        "expires_at": "2025-07-31T21:07:20+00:00",
        "group": "testbed",
        "id": "d90f3097-6898-4652-8ef1-065d928c70ba",
        "nodetypes": {
            "nodetypes": [
                {
                    "approved_at": "2025-07-31T20:07:28+00:00",
                    "canceled_at": null,
                    "count": 1,
                    "deleted_at": null,
                    "error": null,
                    "errorCode": null,
                    "nodetype": "d710",
                    "reservation_id": "fb0120d1-6e49-11f0-90d9-e4434b2381fc",
                    "resgroup_id": "d90f3097-6898-4652-8ef1-065d928c70ba",
                    "urn": "urn:publicid:IDN+emulab.net+authority+cm"
                }
            ]
        },
        "project": "testbed",
        "reason": "My awesome research project",
        "start_at": "2025-07-31T20:07:20+00:00"
    }
	
### Searching for an available slot.

You can search for an available time to start a reservation group by
providing a project, duration (in hours) and a set of resources. Using
the same set of resources above:

	portal-cli resgroup search --project testbed --duration 1 \
	    --nodetypes @cloudlabclient/tests/resgroups/res-create.json 
    {
        "start_at": "2025-08-01T13:00:00+00:00",
        "expires_at": "2025-08-01T14:00:00+00:00"
    }
	
If the start time returned is acceptable, use that (and either the
same duration or the expires_at) to create a new reservation:

	portal-cli resgroup create --project testbed \
          --reason "My awesome research project" \
	      --start-at "2025-08-01T13:00:00+00:00" \
		  --expires_at "2025-08-01T14:00:00+00:00" \
          --nodetypes @cloudlabclient/tests/resgroups/res-create.json

### Deleting a reservation group

To delete a reservation group:

    portal-cli resgroup delete --resgroup-id $RESID

## Profiles

The next few sections describe how to use the API to manage profiles.
Since profiles are described with Python and `geni-lib` it might be
helpful to look at https://docs.cloudlab.us/creating-profiles.html and
https://docs.cloudlab.us/geni-lib.html first.

### Creating a profile

Here is the simplest profile you can imagine:

    """This is a test of the Portal API. 
    """

    import geni.portal as portal
    import geni.rspec.pg as rspec

    request = portal.context.makeRequestRSpec()
    node1 = request.RawPC('node')

    portal.context.printRequestRSpec()

and here is the CLI command to create this profile:

    PROFILEID=`portal-cli profile create --name apitest \
	    --project testbed \
        --script @cloudlabclient/tests/profiles/profile-create.py | jq -r '.id`
	export PROFILEID

A python fragment to do the same:

    from cloudlabclient.portal.client import PortalClient
    from cloudlabclient.portal.v1.models import ProfileCreate
    
    SCRIPT = '''\
    """This is a test of the Portal API. 
    """

    import geni.portal as portal
    import geni.rspec.pg as rspec

    request = portal.context.makeRequestRSpec()
    node1 = request.RawPC('node')

    portal.context.printRequestRSpec()'''

    Portal = PortalClient("http://boss.emulab.net:43795",
                          "your_token_string",
                          detailed=False, raise_on_unexpected_status=True)

    myprofile = Portal.create_profile(body = ProfileCreate(
        name = "apitest",
        project = "testbed",
        script = SCRIPT,
    ))

Instead of a script, you can provide the URL of a repository (with
--repository-url) that conforms to Cloudlab's repository backed
profile format. More information on repository backed profiles
is in the
[https://docs.cloudlab.us/creating-profiles.html#(part._repo-based-profiles)](Cloudlab manual)

### Trigger update to a repository backed profile

For repository back profiles, a manual update from the repository can
be triggered with:



